/*  Execute a state  */

#include <stdio.h>
#include "psf_prototype.h"
#include "psf_malloc.h"
#include "psf_standards.h"

#include "tiltype.h"
#include "states.h"
#include "closure.h"
#include "dll.h"
#include "exitcodes.h"
#include "expand.h"
#include "expressions.h"
#include "interface.h"
#include "objects.h"
#include "options.h"
#include "random.h"
#include "simulate.h"
#include "std.h"
#include "transitions.h"

static DLL  undo_list,redo_list,trace_free_list;

static void display_state();
static void display_transition();
/*
static void print_do_list();
*/

void simulation()   /* or execution for that matter */
{
  struct state *current_state;
  DLL initial_states;

  if (Option_Execution) {               /* set seed for random generator ... */
    if (Option_Initial_Seed) {
      srandstr(Option_Initial_Seed);    /* ... to the given string */
    } else {
      srandtime();                      /* ... to the current time */
    }
  } else {
    undo_list = dll_create();           /* create an undo stack */
    redo_list = dll_create();           /* create a redo stack */
    trace_free_list = dll_create();     /* create a trace free list */
  }

  initial_states = dll_copy(the_states);

 do {
  fprintf(UI_CONSOLE,"----------------------------------------------\n");

  current_state = (struct state *)
    menu_select(
      UI_CONSOLE,
      initial_states,
      display_state,
      "select the initial state\n",
      "\n [q] Quit\n",
      "qQ"
    );

  if (!current_state) {
    switch(UI_LATEST_CODE) {
      case UI_KEY:
	exit(ERR_OK);
	break;
      case UI_CANCEL /* = '#' */ : /* forgotten possibility */
		break;
      default:
	ProgrammerError;
	break;
    }
  }

  if (Option_Execution) {
    fprintf(UI_CONSOLE,"\n");
  } else {
    dll_flush(undo_list);
    dll_flush(redo_list);
  }

  while (current_state) {
    /* be sure this state itself gets processed */
    current_state->ancestor = current_state->index;
    if (Option_Execution) {
      current_state = execute_state(current_state);
    } else {
      current_state = simulate_state(current_state);
    }
    if (!current_state) {
      break;
    }
  }

 } while(TRUE);
}

struct state *explore_state(current_state)
struct state *current_state;
{
  DLL_ITEM ptr;
  struct state *the_state;

  switch (current_state->exp->type) {

    case DLK_EXP:
      current_state->deadlock = TRUE;
      break;
    case TCK_EXP:
      current_state->final = TRUE;
      break;
  }

  if (!current_state->explored) {
    DLL_FORALL(the_states,ptr) {
      the_state = (struct state *)dll_inspect(ptr);
      if (the_state->ancestor == current_state->index) {
	/* expand predecessors only! */
	expand_state(the_state);
      }
    }

#define MONITOR_DYNAMICS 0

#if MONITOR_DYNAMICS
    printf("********************************\n");
    print_state(current_state);
    printf("before epsilon closure\n");
    print_trans_for_state(current_state);
#endif

    epsilon_closure_of_state(current_state);

#if MONITOR_DYNAMICS
    printf("********************************\n");
    printf("after epsilon closure\n");
    print_trans_for_state(current_state);
#endif

    delete_epsilon_transitions(current_state);

#if MONITOR_DYNAMICS
    printf("********************************\n");
    printf("after epsilon deletion\n");
    print_trans_for_state(current_state);
#endif

    add_closure_transitions(current_state);

#if MONITOR_DYNAMICS
    printf("********************************\n");
    printf("after adding closure transitions\n");
    print_trans_for_state(current_state);
    printf("********************************\n");
#endif

    current_state->explored = TRUE;

    if (!dll_empty(current_state->transitions)) {
      current_state->final = current_state->deadlock = FALSE;
    }

  }
  return(current_state);
}

struct state *simulate_state(current_state)
struct state *current_state;
{
  struct transition *the_transition;
  static struct trace_info *last_trace;
  int ask_again;

#if 0
  print_do_list(undo_list,"undo");
  print_do_list(redo_list,"redo");
#endif

  fprintf(UI_CONSOLE,"----------------------------------------------\n");
  fprint_expression(UI_CONSOLE,current_state->exp);
  fprintf(UI_CONSOLE,"\n");

  current_state = explore_state(current_state);

  if (current_state->deadlock) {
    fprintf(UI_CONSOLE,"\n*** this is a deadlock state\n");
  } else if (current_state->final) {
    fprintf(UI_CONSOLE,"\n*** this is a final state\n");
  }

  do {

    ask_again = FALSE;

    if (!dll_empty(undo_list)) {
      if (!dll_empty(redo_list)) {
	the_transition = (struct transition *)
	  menu_select(
	    UI_CONSOLE,
	    current_state->transitions,
	    display_transition,
	    "",
	    "\n [c] Cancel, [t] Trace, [u] Undo, [r] Redo, [q] Quit\n",
	    "cCuUtTrRqQ"
	  );
      } else {
	the_transition = (struct transition *)
	  menu_select(
	    UI_CONSOLE,
	    current_state->transitions,
	    display_transition,
	    "",
	    "\n [c] Cancel, [t] Trace, [u] Undo, [q] Quit\n",
	    "cCtTuUqQ"
	  );
      }
    } else {
      if (!dll_empty(redo_list)) {
	the_transition = (struct transition *)
	  menu_select(
	    UI_CONSOLE,
	    current_state->transitions,
	    display_transition,
	    "",
	    "\n [c] Cancel, [t] Trace, [r] Redo [q] Quit\n",
	    "cCtTrRqQ"
	  );
      } else {
	the_transition = (struct transition *)
	  menu_select(
	    UI_CONSOLE,
	    current_state->transitions,
	    display_transition,
	    "",
	    "\n [c] Cancel, [t] Trace [q] Quit\n",
	    "cCtTqQ"
	  );
      }
    }

    if (!the_transition) {
      switch (UI_LATEST_CODE) {
	case UI_KEY:
	  switch (UI_LATEST_KEY) {
	    case 'q':
	    case 'Q':
		exit(ERR_OK);
		break;

	    case 'r':
	    case 'R':
	      if (dll_empty(redo_list)) {
		fprintf(stderr,"Can't Redo\n");
		ask_again = TRUE;
	      } else {
		last_trace = (struct trace_info *)dll_delete_last(redo_list);
		dll_append(undo_list,(DLL_INFO)last_trace);
		fprintf(UI_CONSOLE,"--(");
		fprint_expression(UI_CONSOLE,last_trace->transition->label);
		fprintf(UI_CONSOLE,")-->\n");
		return(last_trace->transition->target);
	      }
	      break;

	    case 'u':
	    case 'U':
	      if (dll_empty(undo_list)) {
		fprintf(stderr,"Can't Undo\n");
		ask_again = TRUE;
	      } else {
		last_trace = (struct trace_info *)dll_delete_last(undo_list);
		fprintf(UI_CONSOLE,"<--(");
		fprint_expression(UI_CONSOLE,last_trace->transition->label);
		fprintf(UI_CONSOLE,")--\n");
		dll_append(redo_list,(DLL_INFO)last_trace);
		return(last_trace->state);
	      }
	      break;

	    case 't':
	    case 'T':
	      print_trace();
	      return(current_state);
	      break;

	    case 'c':
	    case 'C':
	      return(NULL);
	      break;

	    default:
	      fprintf(stderr,"Unanticipated input: '%c'\n",UI_KEY);
	      ProgrammerError;
	      break;
	  }
	  break;

	case UI_CANCEL:
	  return(NULL);
	  break;

	default:
	  ProgrammerError;
	  break;
      }
    }
  } while (ask_again);
  dll_flush(redo_list);
  last_trace = new_trace_info(current_state,the_transition);
  dll_append(undo_list,(DLL_INFO)last_trace);
  return(the_transition->target);
}

struct state *execute_state(current_state)
struct state *current_state;
{
  struct transition *the_transition;

  current_state = explore_state(current_state);

  if (current_state) {
    if (current_state->deadlock) {
      fprintf(UI_CONSOLE,"\n*** Unsuccesful Termination (Deadlock) ***\n");
      return(NULL);
    } else if (current_state->final) {
      fprintf(UI_CONSOLE,"\n*** Succesful Termination ***\n");
      return(NULL);
    }
  }

  the_transition = random_transition(current_state);

  if (the_transition) {
    fprintf(stdout,"  ");               /* show the transition */
    fprint_expression(stdout,the_transition->label);
    fprintf(stdout,"\n");

    return(the_transition->target);
  } else {
    /* no more transitions */
    return(NULL);
  }
}

struct transition *random_transition(state) /* pick a random transition */
struct state *state;
{
  DLL_ITEM ptr;
  int choice;

  choice = randbnd(1,dll_count(state->transitions));
  DLL_FORALL (state->transitions,ptr) {
    if (!(--choice)) {
      return((struct transition *)dll_inspect(ptr));
    }
  }
  return(NULL);
}

int final_state(the_state)
struct state *the_state;
{
  return(dll_empty(the_state->transitions));
}

int final_state_by_definition(the_state)
struct state *the_state;
{
  switch (the_state->exp->type) {

    case DLK_EXP:
    case TCK_EXP:
      return(TRUE);
      break;

    default:
      return(FALSE);
      break;
  }
  return(FALSE);
}

/*
  add non-epsilon transitions of states in the epsilon closure
*/

void add_closure_transitions(the_state)
struct state *the_state;
{
  DLL_ITEM closure_ptr,transition_ptr;
  struct state *the_target;
  struct transition *the_transition;

  DLL_FORALL(the_state->epsilon_closure,closure_ptr) {
    /* get next possible target from closure list */
    the_target = (struct state *)dll_inspect(closure_ptr);
    if (the_target != the_state) {  /* do not insert transitions already there */
      switch (the_target->exp->type) {
    case DLK_EXP:
      the_state->deadlock = TRUE;
      break;
    case TCK_EXP:
      the_state->final = TRUE;
      break;
      }
      DLL_FORALL(the_target->transitions,transition_ptr) {
	/* get next transition from the target */
	the_transition = (struct transition *)dll_inspect(transition_ptr);
	if (the_transition->label->type != EPS_EXP) {
	  /* only add non-epsilon transitions */
	  add_transition(the_state,the_transition->label,
					the_transition->target);
    }
      }
    }
  }
}

/*
  delete epsilon transitions; we do not need them any longer
*/

void delete_epsilon_transitions(the_state)
struct state *the_state;
{
  DLL_ITEM transition_ptr;
  struct transition *the_transition;
  int found_an_epsilon;

  do {
    found_an_epsilon = FALSE;
    DLL_FORALL(the_state->transitions,transition_ptr) {
      /* get next transition from the state */
      the_transition = (struct transition *)dll_inspect(transition_ptr);
      if (the_transition->label->type == EPS_EXP) {
    dll_delete(the_state->transitions,transition_ptr);
    found_an_epsilon = TRUE;
    break;
      }
    }
  } while (found_an_epsilon);
}

struct trace_info *new_trace_info(state,transition)
struct state *state;
struct transition *transition;
{
  struct trace_info *tmp;

  if (dll_empty(trace_free_list)) {
    tmp = PSF_MALLOC(struct trace_info);
  } else {
    tmp = (struct trace_info *)dll_delete_first(trace_free_list);
  }
  tmp->state = state;
  tmp->transition = transition;
  return(tmp);
}

void dispose_trace_info(trace)
struct trace_info *trace;
{
  dll_append(trace_free_list,(DLL_INFO)trace);
}

void print_trace()
{
  DLL_ITEM ptr;
  struct trace_info *tmp;

  fprintf(UI_CONSOLE,"----------------------------------------------\n");
  fprintf(UI_CONSOLE,"Trace:\n\n");

  DLL_FORALL (undo_list,ptr) {
    tmp = (struct trace_info *)dll_inspect(ptr);
    fprintf(UI_CONSOLE,"  ");
    fprint_expression(UI_CONSOLE,tmp->transition->label);
    fprintf(UI_CONSOLE,"\n");
  }
  fprintf(UI_CONSOLE,"\n");
}

/*
static void print_do_list(list,type)
DLL list;
char *type;
{
  DLL_ITEM ptr;

  printf("%s list: ",type);

  DLL_FORALL(list,ptr) {
    printf("<%lX@%lX> ",dll_inspect(ptr),ptr);
  }
  printf("\n");
}
*/

static void display_state(viewport,state)
UI_VIEWPORT viewport;
MENU_ITEM state;
{
  struct state *the_state;

  the_state = (struct state *)state;
  fprint_expression(viewport,(struct expression *)the_state->exp);
}

static void display_transition(viewport,transition)
UI_VIEWPORT viewport;
MENU_ITEM transition;
{
  struct transition *the_transition;

  the_transition = (struct transition *)transition;

  fprintf(viewport,"--(");
  fprint_expression(viewport,the_transition->label);
  fprintf(viewport,")--> ");
  fprint_expression(viewport,the_transition->target->exp);
#if 0
  fprintf(viewport,"\n");
#endif
}
