/*
 * generate LTS from a TIL-file
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "psf_prototype.h"
#include "psf_malloc.h"
#include "psf_standards.h"
#include "psf_fopen.h"

#include "tiltype.h"
#include "blocks.h"
#include "check.h"
#include "states.h"
#include "closure.h"
#include "definitions.h"
#include "dll.h"
#include "exitcodes.h"
#include "expand.h"
#include "expressions.h"
#include "fieldex.h"
#include "lts_til.h"
#include "main.h"
#include "objects.h"
#include "options.h"
#include "readtil.h"
#include "simulate.h"
#include "statetrees.h"
#include "std.h"
#include "version.h"

#define CONTINOUS_OUTPUT 1	/* set to 1 to disallow buffering */
#define TILSUFFIX ".til"
#define TRANSSUFFIX ".lts"

/* globals */

struct module the_mod;
char *filename;
char *progname;

void version()
{
    fprintf(stderr, "trans version %s, last compiled %s\n\n", VERSION, __DATE__);
}

void usage()
{
    fprintf(stderr, "usage: %s [%s] [input-file [output-file]]\n",
	    progname, OPTIONS_STR);
}

int main(argc, argv)
    int argc;
    char *argv[];
{
    FILE *til_in = NULL, *til_out = NULL;
    static suffix suffixes[] = {{"TILSUFFIX", ".til"}, {NULL, NULL}};
    int name_len;
    char *in_file = NULL, *out_file = NULL;
    char *trans_suffix;
    psf_file pf;
    char *sp;

#if CONTINOUS_OUTPUT
    setbuf(stdout, (char *) NULL);
#endif

    progname = psf_basename(argv[0]);	/* update parameter list */
    process_options(argc, argv);
    argc -= optind - 1;
    argv += optind - 1;

    /* check parameters */
    switch (argc) {

    case 1:			/* we have to read from standard input */
	til_in = stdin;
	til_out = stdout;
	filename="stdin";
	break;

    case 2:			/* we have an input file */
	in_file = argv[1];
	break;

    case 3:			/* we have an input and a output file */
	in_file = argv[1];
	out_file = argv[2];
	break;

    default:
	usage();
	exit(ERR_PARAMETER);
	break;
    }

    if (!(trans_suffix = getenv("TRANSSUFFIX"))) {
	trans_suffix = TRANSSUFFIX;
    }
    if (in_file && !out_file) {
	name_len = strlen(argv[1]);
	out_file = PSF_NMALLOC(char, name_len + strlen(trans_suffix) + 1);
	strcpy(out_file, argv[1]);
	sp = strrchr(out_file, '.');
	if (sp)
	    *sp = '\0';
	strcat(out_file, trans_suffix);
    }
    /* open input file */
    if (!til_in) {
	pf = psf_fopen(argv[1], suffixes);
	til_in = pf.fp;
	filename = pf.name;
    }
    if (Option_Verbose)
	version();
    if (read_module(&the_mod, til_in)) {	/* read in til file */
	fprintf(stderr, "%s: error(s) in TIL format.\n", progname);
	exit(ERR_TIL);
    }
    if (the_mod.entries_table[PRO] == 0) {
	(void) fprintf(stderr, "%s: no processes in file \"%s\"\n",
		       progname, filename);
	exit(EXIT_SUCCESS);
    }
    if (in_file)
	fclose(til_in);
    if (check_args(&the_mod) != 0) {
	fprintf(stderr,
		"%s: Error: restricted to argument free atoms and processes\n",
		progname);
	exit(ERR_ARGUMENT);
    }
    init_definitions(&the_mod);

    read_definitions(&the_mod);
    Check_Error;

    read_communications(&the_mod);
    Check_Error;

    if (Option_Debug) {
	print_communications();
	print_sets();
    }
    transition_system();

    if (Option_Verbose) {
	if (out_file) {
	    printf("... generating \"%s\"\n", out_file);
	}
    }
    /* open output file */
    if (!til_out) {
	if ((til_out = fopen(out_file, "w")) == NULL) {
	    fprintf(stderr,
	       "%s: can't open file \"%s\" for output: ", progname, out_file);
	    perror("");
	    exit(ERR_OPEN);
	}
    }
    /*
     * setbuf(til_out,NULL);
     */

    write_til_file(&the_mod, til_out);

    if (out_file) {
	if (fclose(til_out) != 0) {	/* close til output file */
	    fprintf(stderr, "%s: couldn't close", progname);
	    perror(out_file);
	}
    }
    return (Error_Status);
}

void transition_system()
{
    int i;

    if (Option_Debug) {
	printf("\n*** Definitions as given :\n\n");
	print_definitions();
	printf("\n");
    }
    if (Option_Guardedness) {
	if (!guardedness()) {
	    printf("    execution halted!\n");
	    exit(Error_Status);
	};
    }
    if (Option_Debug) {
	printf("\n*** Information about finite traces :\n\n");
	for_all_processes(i) {
	    print_identifier(PRC_EXP, i);
	    if (finite_trace(process[i])) {
		printf(" : finite trace(s)\n");
	    } else {
		printf(" : no finite trace(s)\n");
	    }
	}
    }
    init_operator_names();
    init_states();
    init_state_tree();
    states_for_processes();

    if (Option_Simulator || Option_Execution) {
	simulation();
	/* No more things to do, down here */
	exit(ERR_OK);
    }
    if (Option_Verbose) {
	printf("... calculating states and transitions\n");
    }
    expand_states();

    if (Option_Debug) {
	print_trans_for_states();
    }
    if (Option_Verbose) {
	printf("... calculating epsilon closures\n");
    }
    init_closure();
    epsilon_closures();

    if (Option_Debug) {
	print_epsilon_closures();
    }
    if (Option_Verbose) {
	printf("... combining epsilon-equivalent states\n");
    }
    make_blocks();
    make_trans_for_blocks();

    preferred_names();

    if (Option_Debug) {
	printf("\nBlocks:\n");
	print_blocks(FALSE);
	printf("\nTrans for Blocks:\n");
	print_trans_for_blocks(FALSE);

	print_trans_for_pref_names();
    }
    if (Option_Debug) {
	printf("... %d state(s) / %d transition(s) internally\n", NR_BLOCKS, NR_TRANSITIONS);
    }
    if (Option_Verbose) {
	printf("... reachability analysis\n");
    }
    reachable_blocks();

    if (Option_Blocks) {
	printf("\nReachable Blocks:\n\n");
	print_blocks(TRUE);
	printf("\nTransitions for Reachable Blocks:\n\n");
	print_trans_for_blocks(TRUE);	/* print reachables only */
	printf("\n");
    }
    if (Option_Verbose)
	printf("... %d state(s) / %d transition(s)\n",
		REACHABLE_BLOCKS, REACHABLE_TRANSITIONS);
    if (Option_Debug || Option_List) {
	printf("\nTransitions:\n\n");
	print_reachable_blocks_and_trans();
	printf("\n");
    }
    til_indexes();
}
