/*
 * maintain expressions
 */

#include <stdio.h>
#include "psf_prototype.h"
#include "psf_malloc.h"
#include "psf_standards.h"
#include "psf_exits.h"
#include "tiltype.h"
#include "expressions.h"
#include "definitions.h"
#include "exitcodes.h"
#include "fieldex.h"
#include "main.h"
#include "objects.h"
#include "std.h"
#include "transitions.h"

struct expression the_epsilon = {EPS_EXP, 0, NULL, NULL};
struct expression *epsilon = &the_epsilon;
struct expression the_tick = {TCK_EXP, 0, NULL, NULL};
struct expression *tick = &the_tick;
struct expression the_deadlock = {DLK_EXP, 0, NULL, NULL};
struct expression *deadlock = &the_deadlock;
struct expression the_skip = {SKP_EXP, 0, NULL, NULL};
struct expression *skip = &the_skip;

struct expression *new_expression()
{
    struct expression *tmp;

    tmp = PSF_MALLOC(struct expression);

    tmp->type = UNDEFINED;
    tmp->index = UNDEFINED;
    tmp->left = NULL;
    tmp->right = NULL;
    tmp->nf = FALSE;

    return (tmp);
}

struct expression *create_expression(type, left, right)
    int type;
    struct expression *left, *right;
{
    struct expression *tmp;

    tmp = new_expression();
    tmp->type = type;
    tmp->left = left;
    tmp->right = right;

    return (tmp);
}

struct expression *clone_expression(prototype)	/* make a copy of info */
    struct expression *prototype;
{
    struct expression *tmp;

    tmp = new_expression();

    tmp->type = prototype->type;
    tmp->index = prototype->index;
    tmp->left = prototype->left;
    tmp->right = prototype->right;

    return (tmp);
}

struct expression *update_expression(expr, left, right, type)
    struct expression *expr, *left, *right;
    int type;
{
    struct expression *tmp;

#define SURE 1

#if SURE
    if (expr->left == left && expr->right == right && expr->type == type) {
	return (expr);
    } else {
#endif

	tmp = clone_expression(expr);
	tmp->left = left;
	tmp->right = right;
	tmp->type = type;
	return (tmp);

#if SURE
    }
#endif
}

void fprint_identifier_exp(file, expr)
    FILE *file;
    struct expression *expr;
{
    fprint_identifier(file, expr->type, expr->index);
}

#if 0
char *label_name(index)
    int index;
{
    switch (index) {
    case SKIP:
	return ("<skip>");
	break;
    case EPSILON:
	return ("<epsilon>");
	break;
    default:
	return (atom[index]->name);
	break;
    }
    ProgrammerError;
    return (NULL);
}

int label_index(expr)
    struct expression *expr;
{
    switch (expr->type) {
    case ATM_EXP:
	return (expr->index);
	break;
    case SKP_EXP:
	return (SKIP);
	break;
    case EPS_EXP:
	return (EPSILON);
	break;
    default:
	fprintf(stderr, "%s: expression is not a valid label", progname);
	return (UNDEFINED);
	break;
    }
    ProgrammerError;
    return (UNDEFINED);
}

#endif

void fprint_identifier(file, type, index)
    FILE *file;
    int type, index;
{
    int i;

    switch (type) {
    case ATM_EXP:
	fprintf(file, "%s", atom[index]->name);
	break;
    case PRC_EXP:
	fprintf(file, "%s", process[index]->name);
	if (process[index]->derived) {
	    for (i = 0; i < process[index]->derivative_cnt; i++) {
		fprintf(file, "'");
	    }
	    fprintf(file, "-%d", index);
	}
	break;
    default:
	fprintf(stderr, "%s: expression is not an identifier\n", progname);
	break;
    }
}

void swap_expressions(px, py)
    struct expression **px, **py;
{
    struct expression *tmp;

    tmp = *px;
    *px = *py;
    *py = tmp;
}

int cmp_expressions(exp1, exp2)
    struct expression *exp1, *exp2;
{
    int tmp;

    if (exp1->type == exp2->type) {
	switch (exp1->type) {
	case ATM_EXP:
	case PRC_EXP:
	case TCK_EXP:
	    return (exp1->index - exp2->index);
	    break;
	case SKP_EXP:
	case DLK_EXP:
	case EPS_EXP:
	    return (0);
	    break;
	case SEQ_EXP:
	case ALT_EXP:
	case PAR_EXP:
	case LMG_EXP:
	case COM_EXP:
	    tmp = cmp_expressions(exp1->left, exp2->left);
	    if (tmp == 0) {
		tmp = cmp_expressions(exp1->right, exp2->right);
	    }
	    return (tmp);
	    break;

	case ENC_EXP:
	case HID_EXP:
	    if (exp1->index == exp2->index) {
		return (cmp_expressions(exp1->right, exp2->right));
	    } else {
		return (exp1->index - exp2->index);
	    }
	    break;

	default:
	    fprintf(stderr,
		  "%s: cmp_expressions: unknown expression type\n", progname);
	    ProgrammerError;
	    break;
	}
    } else {
	return (exp1->type - exp2->type);
    }
    ProgrammerError;
    return (UNDEFINED);
}

struct expression *copy_expression(expr)
    struct expression *expr;
{
    struct expression *tmp;

    tmp = new_expression();

    tmp->nf = expr->nf;

    switch (expr->type) {
    case ATM_EXP:
    case PRC_EXP:
	tmp->type = expr->type;
	tmp->index = expr->index;
	break;
    case SKP_EXP:
    case DLK_EXP:
    case TCK_EXP:
	tmp->type = expr->type;
	break;
    case SEQ_EXP:
    case ALT_EXP:
    case PAR_EXP:
    case LMG_EXP:
    case COM_EXP:
	tmp->type = expr->type;
	tmp->left = copy_expression(expr->left);
	tmp->right = copy_expression(expr->right);
	break;

    case ENC_EXP:
    case HID_EXP:
	tmp->type = expr->type;
	tmp->index = expr->index;
	tmp->left = expr->left;
	tmp->right = copy_expression(expr->right);
	break;

    default:
	printf("copy_expression: unanticipated type (%d)\n", expr->type);
	printf("operator name: %s\n", operator_name[expr->type]);
	abort();
	break;
    }
    return (tmp);
}

/* routine for dbx use only */
void print_it(expr)
    struct expression *expr;
{
    fprint_expression(stdout, expr);
    printf("\n");
}

#define SEE_NF 0

void fprint_expression(file, expr)
    FILE *file;
    struct expression *expr;
{

#if SEE_NF
    fprintf(file, "[%d]", expr->nf);
#endif

    switch (expr->type) {
    case ATM_EXP:
    case PRC_EXP:
	fprint_identifier_exp(file, expr);
	break;
    case TCK_EXP:
	fprintf(file, "Final-State");
	break;
    case SKP_EXP:
	fprintf(file, "<skip>");
	break;
    case DLK_EXP:
	fprintf(file, "<delta>");
	break;
    case EPS_EXP:
	fprintf(file, "<epsilon>");
	break;
    case SEQ_EXP:
	fprintf(file, "(");
	fprint_expression(file, expr->left);
	fprintf(file, " . ");
	fprint_expression(file, expr->right);
	fprintf(file, ")");
	break;
    case ALT_EXP:
	fprintf(file, "(");
	fprint_expression(file, expr->left);
	fprintf(file, " + ");
	fprint_expression(file, expr->right);
	fprintf(file, ")");
	break;
    case PAR_EXP:
	fprintf(file, "(");
	fprint_expression(file, expr->left);
	fprintf(file, " || ");
	fprint_expression(file, expr->right);
	fprintf(file, ")");
	break;
    case LMG_EXP:
	fprintf(file, "(");
	fprint_expression(file, expr->left);
	fprintf(file, " ||_ ");
	fprint_expression(file, expr->right);
	fprintf(file, ")");
	break;
    case COM_EXP:
	fprintf(file, "(");
	fprint_expression(file, expr->left);
	fprintf(file, " | ");
	fprint_expression(file, expr->right);
	fprintf(file, ")");
	break;

#define SET_NAMES 1		/* set names or set contents */

    case ENC_EXP:
	fprintf(file, "encaps(");

#if SET_NAMES
	fprintf(file, "%s", ((struct set *) expr->left)->name);
#else
	fprint_set(file, ((struct set *) expr->left)->contents);
#endif

	fprintf(file, ",");
	fprint_expression(file, expr->right);
	fprintf(file, ")");
	break;

    case HID_EXP:
	fprintf(file, "hide(");

#if SET_NAMES
	fprintf(file, "%s", ((struct set *) expr->left)->name);
#else
	fprint_set(file, ((struct set *) expr->left)->contents);
#endif

	fprintf(file, ",");
	fprint_expression(file, expr->right);
	fprintf(file, ")");
	break;

    default:
	fprintf(stderr,
		"%s: print_expression: unknown expression type '%c'\n",
		progname, expr->type);
	ProgrammerError;
	break;
    }
}

#define MAX_PT_DEPTH		/* maximum depth supported */
static int pt_depth = 0;	/* depth of parse tree */

void fprint_parse_depth(file)
    FILE *file;
{
    int i;

    fprintf(file, "%3d. ", pt_depth);

    for (i = 1; i < pt_depth; i++) {
	putc(' ', file);
	putc(' ', file);
    }
}

void fprint_operator(file, operator)
    FILE *file;
    int operator;
{
    fprint_parse_depth(file);

    switch (operator) {

    case ALT_EXP:
	fprintf(file, "+\n");
	break;

    case SEQ_EXP:
	fprintf(file, ".\n");
	break;

    case PAR_EXP:
	fprintf(file, "||\n");
	break;

    case LMG_EXP:
	fprintf(file, "||_\n");
	break;

    case COM_EXP:
	fprintf(file, "|\n");
	break;

    case ENC_EXP:
	fprintf(file, "encapsulation: ");
	break;

    case HID_EXP:
	fprintf(file, "abstraction: ");
	break;

    default:
	ProgrammerError;
	break;
    }
}

void fprint_parse_tree(file, expr)
    FILE *file;
    struct expression *expr;
{

#if SEE_NF
    fprintf(file, "[%d]", expr->nf);
#endif

    pt_depth++;

    switch (expr->type) {
    case ATM_EXP:
    case PRC_EXP:
	fprint_parse_depth(file);
	fprint_identifier_exp(file, expr);
	fprintf(file, "\n");
	break;
    case TCK_EXP:
	fprint_parse_depth(file);
	fprintf(file, "Final-State");
	fprintf(file, "\n");
	break;
    case SKP_EXP:
	fprint_parse_depth(file);
	fprintf(file, "<skip>");
	fprintf(file, "\n");
	break;
    case DLK_EXP:
	fprint_parse_depth(file);
	fprintf(file, "<delta>");
	fprintf(file, "\n");
	break;
    case EPS_EXP:
	fprint_parse_depth(file);
	fprintf(file, "<epsilon>");
	fprintf(file, "\n");
	break;
    case SEQ_EXP:
    case ALT_EXP:
    case PAR_EXP:
    case LMG_EXP:
    case COM_EXP:
	fprint_parse_tree(file, expr->right);
	fprint_operator(file, expr->type);
	fprint_parse_tree(file, expr->left);
	break;

#define SET_NAMES 1		/* set names or set contents */

    case ENC_EXP:
    case HID_EXP:
	fprint_operator(file, expr->type);

#if SET_NAMES
	fprintf(file, "%s\n", ((struct set *) expr->left)->name);
#else
	fprint_set(file, ((struct set *) expr->left)->contents);
#endif

	fprintf(file, "\n");

	fprint_parse_tree(file, expr->right);
	break;

    default:
	fprintf(stderr,
		"%s: print_parse_tree: unknown expression type\n", progname);
	ProgrammerError;
	break;
    }
    pt_depth--;
}

struct expression *retrieve_expression(pexp)
    struct process_expr *pexp;
{
    struct expression *tmp;

    tmp = new_expression();

    switch (pexp->fun) {
    case AET:
	tmp->type = (pexp->proc_expr.pe2.ind.table == ATM) ?
	    ATM_EXP : PRC_EXP;
	tmp->index = pexp->proc_expr.pe2.ind.key;
	break;
    case SKP:
	tmp->type = SKP_EXP;
	break;
    case DLK:
	tmp->type = DLK_EXP;
	break;
    case ALT:
	tmp->type = ALT_EXP;
	if (pexp->proc_expr.pe3.a != 2) {
	    fprintf(stderr,
		 "%s: only binary operators allowed in process expressions\n",
		    progname);
	    exit(ERR_TIL);
	}
	tmp->left = retrieve_expression(&(pexp->proc_expr.pe3.pe[0]));
	tmp->right = retrieve_expression(&(pexp->proc_expr.pe3.pe[1]));
	break;
    case SEQ:
	tmp->type = SEQ_EXP;
	if (pexp->proc_expr.pe3.a != 2) {
	    fprintf(stderr,
		 "%s: only binary operators allowed in process expressions\n",
		    progname);
	    exit(ERR_TIL);
	}
	tmp->left = retrieve_expression(&(pexp->proc_expr.pe3.pe[0]));
	tmp->right = retrieve_expression(&(pexp->proc_expr.pe3.pe[1]));
	break;
    case PAR:
	tmp->type = PAR_EXP;
	if (pexp->proc_expr.pe3.a != 2) {
	    fprintf(stderr,
		 "%s: only binary operators allowed in process expressions\n",
		    progname);
	    exit(ERR_TIL);
	}
	tmp->left = retrieve_expression(&(pexp->proc_expr.pe3.pe[0]));
	tmp->right = retrieve_expression(&(pexp->proc_expr.pe3.pe[1]));
	break;
    case ENC:
	tmp->type = ENC_EXP;
	tmp->index = pexp->proc_expr.pe1.ind.key;
	/*
	 * printf("reading set %s: ",set[pexp->proc_expr.pe1.ind.key]->name);
	 * print_set(set[pexp->proc_expr.pe1.ind.key]->contents); printf("\n");
	 */
	tmp->left = (struct expression *) set[pexp->proc_expr.pe1.ind.key];
	tmp->right = retrieve_expression(&(pexp->proc_expr.pe1.pe[0]));
	break;
    case HID:
	tmp->type = HID_EXP;
	tmp->index = pexp->proc_expr.pe1.ind.key;
	/*
	 * printf("reading set %s: ",set[pexp->proc_expr.pe1.ind.key]->name);
	 * print_set(set[pexp->proc_expr.pe1.ind.key]->contents); printf("\n");
	 */
	tmp->left = (struct expression *) set[pexp->proc_expr.pe1.ind.key];
	tmp->right = retrieve_expression(&(pexp->proc_expr.pe1.pe[0]));
	break;
    case SUM:
    case MRG:
	fprintf(stderr,
	   "%s: only data free specifications supported, sum or merge found.\n",
		progname);
	exit(ERR_TIL);
    case IF:
	fprintf(stderr, "%s: conditional operator not supported\n", progname);
	exit(ERR_TIL);
    case INTR:
	fprintf(stderr, "%s: interrupt operator not supported\n", progname);
	exit(ERR_TIL);
    case DISR:
	fprintf(stderr, "%s: disrupt operator not supported\n", progname);
	exit(ERR_TIL);
    case PRIO:
	fprintf(stderr, "%s: priority operator not supported\n", progname);
	exit(ERR_TIL);
    case STAR:
	fprintf(stderr, "%s: Kleene star not supported\n", progname);
	exit(ERR_TIL);
    case SHARP:
	fprintf(stderr, "%s: sharp operator not supported\n", progname);
	exit(ERR_TIL);
    default:
	fprintf(stderr, "%s: unknown TIL operator '%c'\n", progname, pexp->fun);
	exit(ERR_TIL);
    }

    return (tmp);
}

struct expression *read_expression(index)
    int index;
{
    return (retrieve_expression(&(the_mod.def[index].p_expr)));
}
