/*
 * dll.c : doubly linked list structures
 */

#include <stdio.h>
#include <stdlib.h>
#include "psf_prototype.h"
#include "psf_malloc.h"
#include "psf_standards.h"
#include "sign.h"
#include "std.h"
#include "dll_check.h"
#include "dll.h"
#include "_dll.h"

/* global variables, local to this module */

static DLL_ITEM _item_freelist = NULL;	/* free list of item blocks */
static DLL	_list_freelist = NULL;	/* free list of dll blocks */

/*
 * Local, List Element Specific Routines
 *
 * _dll_create_item  :		      -> DLL_ITEM _dll_delete_item  : DLL #
 * DLL_ITEM -> DLL_INFO _dll_dispose_item : DLL_ITEM	   ->
 */

DLL_ITEM _dll_create_item(info)	    /* create a doubly linked list item */
    DLL_INFO	    info;
{
    DLL_ITEM	    ptr;

    if (!_item_freelist) {
	ptr = (DLL_ITEM) PSF_MALLOC(DLL_ITBL);
    } else {
	ptr = _item_freelist;	/* get from free list */
	_item_freelist = _item_freelist->fw;
    }
    ptr->fw = ptr->bw = NULL;
    ptr->info = info;

#if DLL_CHECK
    ptr->id = ITEM_ID;
    if (_item_guard) {
	_item_guard = FALSE;
	_dll_add(_issued_items, (DLL_INFO) ptr);
	_item_guard = TRUE;
    }
#endif

    return (ptr);
}

DLL_INFO _dll_delete_item(list, ptr)	/* delete item pointed to from list */
    DLL		    list;
    DLL_ITEM	    ptr;
{
    DLL_INFO	    info;

#if DLL_CHECK
    _dll_delete_info(_issued_items, (DLL_INFO) ptr);
#endif

    if (!list->first->fw) { /* one element left */
	info = list->first->info;
	_dll_dispose_item(list->first);
	list->first = list->last = NULL;
	return (info);
    }
    if (ptr == list->first) {	/* delete first element */
	info = ptr->info;
	ptr->fw->bw = ptr->bw;
	list->first = list->first->fw;
	_dll_dispose_item(ptr);
	return (info);
    }
    if (ptr == list->last) {/* delete last element */
	info = ptr->info;
	ptr->bw->fw = ptr->fw;
	list->last = list->last->bw;
	_dll_dispose_item(ptr);
	return (info);
    }
    info = ptr->info;	/* element in the middle */
    ptr->fw->bw = ptr->bw;
    ptr->bw->fw = ptr->fw;
    _dll_dispose_item(ptr);
    return (info);
}

void _dll_dispose_item(ptr)	/* dispose a doubly linked list item */
    DLL_ITEM	    ptr;
{
#if DLL_CHECK
    ptr->id = FREED_ITEM_ID;
#endif

    ptr->fw = _item_freelist;	/* add to free list */
    _item_freelist = ptr;
}

/*
 * DLL Creation Routines
 *
 * dll_create	 :	 -> DLL dll_s_create  : int() -> DLL dll_u_create  :
 * -> DLL dll_su_create : int() -> DLL
 *
 * _dll_create_descendent : DLL -> DLL
 */

DLL dll_create()
{		/* create a doubly linked list */
    DLL		    ptr;

    if (!_list_freelist) {
	ptr = (DLL) PSF_MALLOC(DLL_BLOCK);
    } else {
	ptr = _list_freelist;	/* get from free list */
	_list_freelist = (DLL) _list_freelist->first;
    }
    ptr->first = ptr->last = NULL;
    ptr->cmp = NULL;
    ptr->sorted = ptr->unique = FALSE;
    ptr->size = 0;

#if DLL_CHECK
    ptr->id = LIST_ID;
    if (_list_guard) {
	_list_guard = FALSE;
	_dll_add(_issued_lists, (DLL_INFO) ptr);
	_list_guard = TRUE;
    }
#endif

    return (ptr);
}

DLL dll_s_create(cmp)
    int		    (*cmp) ();
{
    DLL		    ptr;

    ptr = dll_create();
    ptr->cmp = cmp;
    ptr->sorted = TRUE;
    return (ptr);
}

DLL dll_u_create(cmp)
    int		    (*cmp) ();
{
    DLL		    ptr;

    ptr = dll_create();
    ptr->cmp = cmp;
    ptr->unique = TRUE;
    return (ptr);
}

DLL dll_su_create(cmp)
    int		    (*cmp) ();
{
    DLL		    ptr;

    ptr = dll_create();
    ptr->cmp = cmp;
    ptr->sorted = TRUE;
    ptr->unique = TRUE;
    return (ptr);
}

DLL dll_create_descendent(prototype)
    DLL		    prototype;
{
    check_list(prototype);
    return (_dll_create_descendent(prototype));
}

DLL _dll_create_descendent(prototype)
    DLL		    prototype;
{
    DLL		    ptr;

    ptr = dll_create();
    ptr->sorted = prototype->sorted;
    ptr->unique = prototype->unique;
    ptr->cmp = prototype->cmp;
    return (ptr);
}

/*
 * DLL Disposal Routines
 *
 * dll_flush   : DLL -> dll_dispose : DLL ->
 */

void dll_flush(ptr)	    /* delete all elements from a doubly linked
		 * list */
    DLL		    ptr;
{
    check_list(ptr);
    _dll_flush(ptr);
}

void _dll_flush(ptr)		/* delete all elements from a doubly linked
		 * list */
    DLL		    ptr;
{
    for (; !_dll_empty(ptr);) {
	_dll_delete_first(ptr);
    }
}

void dll_dispose(ptr)	    /* dispose a doubly linked list */
    DLL		    ptr;
{
    check_list(ptr);
    _dll_dispose(ptr);
}

void _dll_dispose(ptr)	    /* dispose a doubly linked list */
    DLL		    ptr;
{
    _dll_flush(ptr);

#if DLL_CHECK
    ptr->id = FREED_LIST_ID;
    _dll_delete_info(_issued_lists, (DLL_INFO) ptr);
#endif

    ptr->first = (DLL_ITEM) _list_freelist; /* put on free list */
    _list_freelist = ptr;
}

/*
 * Simple DLL Routines:
 *
 * dll_empty : DLL -> int dll_first : DLL -> int dll_last  : DLL -> int
 * dll_count : DLL -> int dll_copy  : DLL -> DLL
 */

int dll_empty(list)	    /* check whether list empty */
    DLL		    list;
{
    check_list(list);
    return (_dll_empty(list));
}

int _dll_empty(list)	    /* check whether list empty */
    DLL		    list;
{
    return (!(list->first) && !(list->last));
}

int dll_first(list, curr)	/* is current first element? */
    DLL		    list;
    DLL_ITEM	    curr;
{
    check_list(list);
    check_item(curr);
    return (_dll_first(list, curr));
}

int _dll_first(list, curr)	/* is current first element? */
    DLL		    list;
    DLL_ITEM	    curr;
{
    return (curr == list->first);
}

int dll_last(list, curr)	/* is current last element? */
    DLL		    list;
    DLL_ITEM	    curr;
{
    check_list(list);
    check_item(curr);
    return (_dll_last(list, curr));
}

int _dll_last(list, curr)	/* is current last element? */
    DLL		    list;
    DLL_ITEM	    curr;
{
    return (curr == list->last);
}

int dll_count(list)	    /* count the items in the list */
    DLL		    list;
{
    check_list(list);
    return (_dll_count(list));
}

int _dll_count(list)	    /* count the items in the list */
    DLL		    list;
{
    DLL_ITEM	    ptr;
    int		    cnt = 0;

    _DLL_FORALL(list, ptr) {
	cnt++;
    }
    return (cnt);
}

DLL dll_copy(list)	    /* create a copy of a dll */
    DLL		    list;
{
    check_list(list);
    return (_dll_copy(list));
}

DLL _dll_copy(list)	    /* create a copy of a dll */
    DLL		    list;
{
    DLL		    new_list;
    DLL_ITEM	    curr;

    new_list = dll_create();

    new_list->sorted = list->sorted;
    new_list->unique = list->unique;
    new_list->size = list->size;
    new_list->cmp = list->cmp;

    _DLL_FORALL(list, curr) {
	_dll_add_last(new_list, _dll_inspect(curr));
    }

    return (new_list);
}

/*
 * DLL Manoeuvring Routines:
 *
 * dll_go_first : DLL	   -> DLL_ITEM dll_go_last  : DLL      -> DLL_ITEM
 * dll_go_fw	: DLL_ITEM -> DLL_ITEM dll_go_bw    : DLL_ITEM -> DLL_ITEM
 */

DLL_ITEM dll_go_first(list)	/* go to first element of list */
    DLL		    list;
{
    check_list(list);
    return (_dll_go_first(list));
}

DLL_ITEM _dll_go_first(list)	    /* go to first element of list */
    DLL		    list;
{
    return (list->first);
}

DLL_ITEM dll_go_last(list)	/* go to last element of list */
    DLL		    list;
{
    check_list(list);
    return (_dll_go_last(list));
}

DLL_ITEM _dll_go_last(list)	/* go to last element of list */
    DLL		    list;
{
    return (list->last);
}

DLL_ITEM dll_go_fw(curr)	    /* go to next element of list */
    DLL_ITEM	    curr;
{
    check_item(curr);
    return (_dll_go_fw(curr));
}

DLL_ITEM _dll_go_fw(curr)	/* go to next element of list */
    DLL_ITEM	    curr;
{
    if (curr) {
	return (curr->fw);
    }
    return (NULL);
}

DLL_ITEM dll_go_bw(curr)	    /* go to previous element of list */
    DLL_ITEM	    curr;
{
    check_item(curr);
    return (_dll_go_bw(curr));
}

DLL_ITEM _dll_go_bw(curr)	/* go to previous element of list */
    DLL_ITEM	    curr;
{
    if (curr) {
	return (curr->bw);
    }
    return (NULL);
}

/*
 * DLL Insertion Routines:
 *
 * dll_add	     : DLL # DLL_INFO		 -> DLL_ITEM
 *
 * dll_insert	     : DLL # DLL_INFO		 -> DLL_ITEM dll_append	       :
 * DLL # DLL_INFO	     -> DLL_ITEM dll_add_before	   : DLL # DLL_ITEM #
 * DLL_INFO -> DLL_ITEM dll_add_after	  : DLL # DLL_ITEM # DLL_INFO ->
 * DLL_ITEM
 *
 *
 * DLL Local Insertion Routines:
 *
 * _dll_add_first   : DLL # DLL_INFO		-> DLL_ITEM _dll_add_last    :
 * DLL # DLL_INFO	     -> DLL_ITEM _dll_insert_at	  : DLL # DLL_ITEM #
 * DLL_INFO -> DLL_ITEM _dll_append_at	 : DLL # DLL_ITEM # DLL_INFO ->
 * DLL_ITEM
 */

DLL_ITEM dll_add(list, info)	    /* add info to the list */
    DLL		    list;
    DLL_INFO	    info;
{
    check_list(list);
    return (_dll_add(list, info));
}

DLL_ITEM _dll_add(list, info)	    /* add info to the list */
    DLL		    list;
    DLL_INFO	    info;
{
    if (list->sorted) {
	if (list->unique) {
	    return (_dll_su_add(list, info));
	} else {
	    return (_dll_s_add(list, info));
	}
    } else {
	if (list->unique) {
	    return (_dll_u_add(list, info));
	} else {
	    return (_dll_add_last(list, info));
	}
    }
}

DLL_ITEM dll_insert(list, info)	    /* insert info at the front of list */
    DLL		    list;
    DLL_INFO	    info;
{
    check_list(list);
    check_non_sorted(list);
    return (_dll_insert(list, info));
}

DLL_ITEM _dll_insert(list, info)	/* insert info at the front of list */
    DLL		    list;
    DLL_INFO	    info;
{
    DLL_ITEM	    item;

    if (list->unique) {
	if (item = _dll_search(list, info)) {
	    return (item);
	}
    }
    return (_dll_add_first(list, info));
}

DLL_ITEM dll_append(list, info)	    /* append info at the back of list */
    DLL		    list;
    DLL_INFO	    info;
{
    check_list(list);
    check_non_sorted(list);
    return (_dll_append(list, info));
}

DLL_ITEM _dll_append(list, info)	/* append info at the back of list */
    DLL		    list;
    DLL_INFO	    info;
{
    DLL_ITEM	    item;

    if (list->unique) {
	if (item = _dll_search(list, info)) {
	    return (item);
	}
    }
    return (_dll_add_last(list, info));
}

DLL_ITEM dll_add_before(list, curr, info)/* insert info at current element */
    DLL		    list;
    DLL_ITEM	    curr;
    DLL_INFO	    info;
{
    check_list(list);
    check_item(curr);
    check_non_sorted(list);
    check_on_list(list, curr);
    return (_dll_add_before(list, curr, info));
}

DLL_ITEM _dll_add_before(list, curr, info)  /* insert info at current element */
    DLL		    list;
    DLL_ITEM	    curr;
    DLL_INFO	    info;
{
    DLL_ITEM	    item;

    if (list->unique) {
	if (item = _dll_search(list, info)) {
	    _dll_delete(list, item);
	}
    }
    return (_dll_insert_at(list, curr, info));
}

DLL_ITEM dll_add_after(list, curr, info)    /* append info at current element */
    DLL		    list;
    DLL_ITEM	    curr;
    DLL_INFO	    info;
{
    check_list(list);
    check_item(curr);
    check_non_sorted(list);
    check_on_list(list, curr);
    return (_dll_add_after(list, curr, info));
}

DLL_ITEM _dll_add_after(list, curr, info)/* append info at current element */
    DLL		    list;
    DLL_ITEM	    curr;
    DLL_INFO	    info;
{
    DLL_ITEM	    item;

    if (list->unique) {
	if (item = _dll_search(list, info)) {
	    _dll_delete(list, item);
	}
    }
    return (_dll_append_at(list, curr, info));
}

DLL_ITEM _dll_add_first(list, info) /* insert info at the front of list */
    DLL		    list;
    DLL_INFO	    info;
{
    DLL_ITEM	    item;

    item = _dll_create_item(info);

    if (!list->first) { /* list is empty */
	list->first = item;
	list->last = list->first;
    } else {
	list->first->bw = item;
	item->fw = list->first;
	list->first = item;
    }
    return (item);
}

DLL_ITEM _dll_add_last(list, info)  /* append info at the back of list */
    DLL		    list;
    DLL_INFO	    info;
{
    DLL_ITEM	    item;

    item = _dll_create_item(info);

    if (!list->last) {	/* list is empty */
	list->last = item;
	list->first = list->last;
    } else {
	list->last->fw = item;
	item->bw = list->last;
	list->last = item;
    }
    return (item);
}

DLL_ITEM _dll_insert_at(list, curr, info)/* insert info at current element */
    DLL		    list;
    DLL_ITEM	    curr;
    DLL_INFO	    info;
{
    DLL_ITEM	    item;

    item = _dll_create_item(info);

    if (curr == list->first) {	/* current is first? */
	list->first->bw = item;
	item->fw = list->first;
	list->first = item;
    } else {
	item->fw = curr;
	item->bw = curr->bw;
	item->fw->bw = item;
	item->bw->fw = item;
    }
    return (item);
}

DLL_ITEM _dll_append_at(list, curr, info)/* append info at current element */
    DLL		    list;
    DLL_ITEM	    curr;
    DLL_INFO	    info;
{
    DLL_ITEM	    item;

    item = _dll_create_item(info);

    if (curr == list->last) {	/* current is last? */
	list->last->fw = item;
	item->bw = list->last;
	list->last = item;
    } else {
	item->bw = curr;
	item->fw = curr->fw;
	item->bw->bw = item;
	item->fw->fw = item;
    }
    return (item);
}

/*
 * DLL Deletion Routines:
 *
 * dll_delete	     : DLL # DLL_ITEM		 -> DLL_INFO dll_delete_first  :
 * DLL			     -> DLL_INFO dll_delete_last   : DLL
 * -> DLL_INFO dll_delete_info	 : DLL # DLL_INFO	     -> DLL_INFO
 */

DLL_INFO dll_delete(list, curr)	    /* retrieve and delete the current element */
    DLL		    list;
    DLL_ITEM	    curr;
{
    check_list(list);
    check_item(curr);
    return (_dll_delete(list, curr));
}

DLL_INFO _dll_delete(list, curr)	/* retrieve and delete the current element */
    DLL		    list;
    DLL_ITEM	    curr;
{
    return (_dll_delete_item(list, curr));
}

DLL_INFO dll_delete_first(list)	    /* retrieve and delete the front element */
    DLL		    list;
{
    check_list(list);
    return (_dll_delete_first(list));
}

DLL_INFO _dll_delete_first(list)	/* retrieve and delete the front element */
    DLL		    list;
{
    return (_dll_delete_item(list, list->first));
}

DLL_INFO dll_delete_last(list)	    /* retrieve and delete the rear element */
    DLL		    list;
{
    check_list(list);
    return (_dll_delete_last(list));
}

DLL_INFO _dll_delete_last(list)	    /* retrieve and delete the rear element */
    DLL		    list;
{
    return (_dll_delete_item(list, list->last));
}

DLL_INFO dll_delete_info(list, info)	/* delete info from list */
    DLL		    list;
    DLL_INFO	    info;
{
    check_list(list);
    return (_dll_delete_info(list, info));
}

DLL_INFO _dll_delete_info(list, info)	/* delete info from list */
    DLL		    list;
    DLL_INFO	    info;
{
    DLL_ITEM	    curr;

    if (list->sorted) {
	_DLL_FORALL(list, curr) {
	    switch (sign((*(list->cmp)) (info, _dll_inspect(curr)))) {
	    case POSITIVE:
		break;
	    case NEGATIVE:
		return (NULL);
		break;
	    case ZERO:
		return (_dll_delete(list, curr));
		break;
	    };
	}
    } else {
	_DLL_FORALL(list, curr) {
	    switch (sign((*(list->cmp)) (info, _dll_inspect(curr)))) {
	    case POSITIVE:
	    case NEGATIVE:
		break;
	    case ZERO:
		return (_dll_delete(list, curr));
		break;
	    };
	}
    }
    return (NULL);
}

/*
 * DLL Retrieval Routines:
 *
 * dll_inspect	     : DLL_ITEM			 -> DLL_INFO dll_inspect_first :
 * DLL			     -> DLL_INFO dll_inspect_last  : DLL
 * -> DLL_INFO
 */

DLL_INFO dll_inspect(curr)	/* get the current element of the list */
    DLL_ITEM	    curr;
{
    check_item(curr);
    return (_dll_inspect(curr));
}

DLL_INFO _dll_inspect(curr)	/* get the current element of the list */
    DLL_ITEM	    curr;
{
    if (!curr) {
	return (NULL);
    }
    return (curr->info);
}

DLL_INFO dll_inspect_first(list)	/* inspect the front element */
    DLL		    list;
{
    check_list(list);
    return (_dll_inspect_first(list));
}

DLL_INFO _dll_inspect_first(list)   /* inspect the front element */
    DLL		    list;
{
    return (_dll_inspect(list->first));
}

DLL_INFO dll_inspect_last(list)	    /* inspect the rear element */
    DLL		    list;
{
    check_list(list);
    return (_dll_inspect_last(list));
}

DLL_INFO _dll_inspect_last(list)	/* inspect the rear element */
    DLL		    list;
{
    return (_dll_inspect(list->last));
}

/*
 * DLL Search Routines:
 *
 * dll_search : DLL # DLL_INFO -> DLL_ITEM
 */

DLL_ITEM dll_search(list, info)
    DLL		    list;
    DLL_INFO	    info;
{
    check_list(list);
    return (_dll_search(list, info));
}

DLL_ITEM _dll_search(list, info)
    DLL		    list;
    DLL_INFO	    info;
{
    DLL_ITEM	    ptr;

    if (list->sorted) {
	_DLL_FORALL(list, ptr) {
	    switch (sign((*(list->cmp)) (info, _dll_inspect(ptr)))) {
	    case POSITIVE:
		break;
	    case NEGATIVE:
		return (NULL);
		break;
	    case ZERO:
		return (ptr);
		break;
	    };
	}
    } else {
	_DLL_FORALL(list, ptr) {
	    if (info == _dll_inspect(ptr)) {
		return (ptr);
	    };
	}
    }
    return (NULL);	/* element not found in list */
}

/*
 * DLL Set Routines:
 *
 * dll_compare	      : DLL # DLL -> int dll_union	    : DLL # DLL -> DLL
 * dll_intersection   : DLL # DLL -> DLL dll_difference	    : DLL # DLL ->
 * DLL
 *
 */

int dll_compare(x, y)	    /* compare two lists; read as "x - y" */
    DLL		    x, y;   /* 1: x bigger / 0: equal / -1: y bigger */
{
    check_list(x);
    check_list(y);
    check_order_fns(x, y);
    return (_dll_compare(x, y));
}

int _dll_compare(x, y)	    /* compare two lists; read as "x - y" */
    DLL		    x, y;   /* 1: x bigger / 0: equal / -1: y bigger */
{
    DLL_ITEM	    x_curr, y_curr;
    int		    the_result;

    for (x_curr = _dll_go_first(x), y_curr = _dll_go_first(y);
	 (x_curr) && (y_curr);
	 x_curr = _dll_go_fw(x_curr), y_curr = _dll_go_fw(y_curr)) {

	if ((the_result = (*(x->cmp)) (_dll_inspect(x_curr), _dll_inspect(y_curr)))) {
	    return (the_result);
	}
    }
    if (!x_curr) {
	if (!y_curr) {
	    return (ZERO);
	} else {
	    return (NEGATIVE);
	}
    } else {
	return (POSITIVE);
    }
}

DLL dll_union(x, y)
    DLL		    x, y;
{
    check_list(x);
    check_list(y);
    check_list_types(x, y);
    return (_dll_union(x, y));
}

DLL _dll_union(x, y)
    DLL		    x, y;
{
    DLL		    z;
    DLL_ITEM	    x_ptr, y_ptr;
    int		    add_x, add_y;

    if (x->sorted) {
	z = _dll_create_descendent(x);
	x_ptr = _dll_go_first(x);
	y_ptr = _dll_go_first(y);

	do {

	    add_x = FALSE;
	    add_y = FALSE;

	    if (x_ptr) {
		if (y_ptr) {
		    switch (sign((*(x->cmp)) (dll_inspect(x_ptr), dll_inspect(y_ptr)))) {
		    case NEGATIVE:
			add_x = TRUE;
			break;
		    case POSITIVE:
			add_y = TRUE;
			break;
		    case ZERO:
			add_x = TRUE;
			if (!x->unique) {
			    add_y = TRUE;
			} else {
			    y_ptr = _dll_go_fw(y_ptr);
			}
			break;
		    }
		} else {
		    add_x = TRUE;
		}
	    } else {
		if (y_ptr) {
		    add_y = TRUE;
		}
	    }

	    if (add_x) {
		_dll_add_last(z, _dll_inspect(x_ptr));
		x_ptr = _dll_go_fw(x_ptr);
	    }
	    if (add_y) {
		_dll_add_last(z, _dll_inspect(y_ptr));
		y_ptr = _dll_go_fw(y_ptr);
	    }
	} while (x_ptr || y_ptr);

    } else {
	z = _dll_copy(x);
	_DLL_FORALL(y, y_ptr) {
	    _dll_add(x, _dll_inspect(y_ptr));
	}
    }
    return (z);
}

DLL dll_intersection(x, y)
    DLL		    x, y;
{
    check_list(x);
    check_list(y);
    check_list_types(x, y);
    return (_dll_intersection(x, y));
}

DLL _dll_intersection(x, y)
    DLL		    x, y;
{
    DLL		    z, w;
    DLL_ITEM	    w_ptr, x_ptr, y_ptr, z_ptr;

    if (x->sorted) {
	z = _dll_create_descendent(x);
	x_ptr = _dll_go_first(x);
	y_ptr = _dll_go_first(y);

	do {

	    if (x_ptr) {
		if (y_ptr) {
		    switch (sign((*(x->cmp)) (dll_inspect(x_ptr), dll_inspect(y_ptr)))) {
		    case NEGATIVE:
			x_ptr = _dll_go_fw(x_ptr);
			break;
		    case POSITIVE:
			y_ptr = _dll_go_fw(y_ptr);
			break;
		    case ZERO:
			_dll_add_last(z, _dll_inspect(x_ptr));
			y_ptr = _dll_go_fw(y_ptr);
			x_ptr = _dll_go_fw(x_ptr);
			break;
		    }
		} else {
		    x_ptr = _dll_go_fw(x_ptr);
		}
	    } else {
		if (y_ptr) {
		    y_ptr = _dll_go_fw(y_ptr);
		}
	    }

	} while (x_ptr || y_ptr);

    } else {
	z = _dll_copy(x);
	w = _dll_copy(y);
	_DLL_FORALL(z, z_ptr) {
	    if (!(w_ptr = _dll_search(w, _dll_inspect(z_ptr)))) {
		/*
		 * watch out: tricky use of z->fw although
		 * already disposed
		 */
		_dll_delete_item(z, z_ptr);
	    } else {
		_dll_delete_item(w, w_ptr);
	    }
	}
	_dll_dispose(w);
    }
    return (z);
}

DLL dll_difference(x, y)
    DLL		    x, y;
{
    check_list(x);
    check_list(y);
    check_list_types(x, y);
    return (_dll_difference(x, y));
}

DLL _dll_difference(x, y)
    DLL		    x, y;
{
    DLL		    z, w;
    DLL_ITEM	    w_ptr, x_ptr, y_ptr, z_ptr;

    if (x->sorted) {
	z = _dll_create_descendent(x);
	x_ptr = _dll_go_first(x);
	y_ptr = _dll_go_first(y);

	do {

	    if (x_ptr) {
		if (y_ptr) {
		    switch (sign((*(x->cmp)) (dll_inspect(x_ptr), dll_inspect(y_ptr)))) {
		    case NEGATIVE:
			_dll_add_last(z, _dll_inspect(x_ptr));
			x_ptr = _dll_go_fw(x_ptr);
			break;
		    case POSITIVE:
			y_ptr = _dll_go_fw(y_ptr);
			break;
		    case ZERO:
			y_ptr = _dll_go_fw(y_ptr);
			x_ptr = _dll_go_fw(x_ptr);
			break;
		    }
		} else {
		    _dll_add_last(z, _dll_inspect(x_ptr));
		    x_ptr = _dll_go_fw(x_ptr);
		}
	    }
	} while (x_ptr && y_ptr);

    } else {
	z = _dll_copy(x);
	w = _dll_copy(y);
	_DLL_FORALL(z, z_ptr) {
	    if (w_ptr = _dll_search(w, _dll_inspect(z_ptr))) {
		/*
		 * watch out: tricky use of z->fw although
		 * already disposed
		 */
		_dll_delete_item(z, z_ptr);
		_dll_delete_item(w, w_ptr);
	    }
	}
	_dll_dispose(w);
    }
    return (z);
}

/*
 * Special DLL Routines
 */

void dll_apply(list, f)	    /* apply function 'f' to all elements from
		 * list */
    DLL		    list;
    void	    (*f) ();
{
    check_list(list);
    _dll_apply(list, f);
}

void _dll_apply(list, f)	/* apply function 'f' to all elements from
		 * list */
    DLL		    list;
    void	    (*f) ();
{
    DLL_ITEM	    curr;

    _DLL_FORALL(list, curr) {
	(*f) (_dll_inspect(curr));
    }
}

int dll_index(list, info)	/* give index to element, -1 if not on list */
    DLL		    list;   /* index starts with 0 */
    DLL_INFO	    info;
{
    check_list(list);
    return (_dll_index(list, info));
}

int _dll_index(list, info)	/* give index to element, -1 if not on list */
    DLL		    list;   /* index starts with 0 */
    DLL_INFO	    info;
{
    DLL_ITEM	    curr;
    int		    index = 0;

    if (list->sorted) {
	_DLL_FORALL(list, curr) {
	    switch (sign((*(list->cmp)) (info, _dll_inspect(curr)))) {
	    case POSITIVE:
		break;
	    case NEGATIVE:
		return (-1);
		break;
	    case ZERO:
		return (index);
		break;
	    };
	    index++;
	}
    } else {
	if (list->unique) {
	    _DLL_FORALL(list, curr) {
		switch (sign((*(list->cmp)) (info, _dll_inspect(curr)))) {
		case POSITIVE:
		case NEGATIVE:
		    break;
		case ZERO:
		    return (index);
		    break;
		};
		index++;
	    }
	} else {
	    _DLL_FORALL(list, curr) {
		if (info == _dll_inspect(curr)) {
		    return (index);
		}
		index++;
	    }
	}
    }
    return (-1);
}

/*
 * Local Specialized Routines:
 *
 * _dll_s_add : DLL # DLL_INFO -> DLL_ITEM _dll_s_delete : DLL # DLL_INFO ->
 * _dll_u_add : DLL # DLL_INFO -> _dll_u_delete : DLL # DLL_INFO ->
 * _dll_su_add : DLL # DLL_INFO -> _dll_su_delete : DLL # DLL_INFO ->
 */

DLL_ITEM _dll_s_add(list, info)	    /* insert into sorted list */
    DLL		    list;
    DLL_INFO	    info;
{
    DLL_ITEM	    curr;

    _DLL_FORALL(list, curr) {
	switch (sign((*(list->cmp)) (info, _dll_inspect(curr)))) {
	case POSITIVE:
	    break;
	case NEGATIVE:
	case ZERO:
	    return (_dll_insert_at(list, curr, info));
	    break;
	};
    }
    return (_dll_add_last(list, info));
}

DLL_ITEM _dll_u_add(list, info)	    /* insert into sorted list */
    DLL		    list;
    DLL_INFO	    info;
{
    DLL_ITEM	    curr;

    _DLL_FORALL(list, curr) {
	switch (sign((*(list->cmp)) (info, _dll_inspect(curr)))) {
	case POSITIVE:
	case NEGATIVE:
	    break;
	case ZERO:
	    return (curr);
	    break;
	};
    }
    return (_dll_add_last(list, info));
}

DLL_ITEM _dll_su_add(list, info)	/* insert into sorted list */
    DLL		    list;
    DLL_INFO	    info;
{
    DLL_ITEM	    curr;

    _DLL_FORALL(list, curr) {
	switch (sign((*(list->cmp)) (info, _dll_inspect(curr)))) {
	case POSITIVE:
	    break;
	case NEGATIVE:
	    return (_dll_insert_at(list, curr, info));
	    break;
	case ZERO:
	    return (curr);
	    break;
	};
    }
    return (_dll_add_last(list, info));
}

DLL_INFO _dll_s_delete(list, info)  /* delete from sorted list */
    DLL		    list;
    DLL_INFO	    info;
{
    DLL_ITEM	    curr;

    _DLL_FORALL(list, curr) {
	switch (sign((*(list->cmp)) (info, _dll_inspect(curr)))) {
	case POSITIVE:
	    break;
	case NEGATIVE:
	    return (NULL);
	    break;
	case ZERO:
	    return (_dll_delete(list, curr));
	    break;
	};
    }
    return (NULL);
}

void _dll_exit(error)
    int		    error;
{
    ProgrammerError;
    exit(error);
}
