/*
 * rewriting routines
 */

#include <stdio.h>
#include "psf_prototype.h"
#include "psf_malloc.h"
#include "psf_standards.h"
#include "dll.h"
#include "tables.h"
#include "objects.h"
#include "exitcode.h"
#include "options.h"
#include "tracing.h"
#include "bindings.h"

#include "rewrite.h"

#define NO_MATCH  -1

#define DEBUG_VARBIND 0
#define DEBUG_MATCH 0
#define DEBUG_FMATCH 0

#define TRACING 1
#define CAUTION 0
#define DEBUG_INFO 0


#if CAUTION
DLL term_list = NULL;		/* list containing all terms in rewrite */
DLL_ITEM tl_ptr = NULL;		/* pointer in term list */

#endif

static int rewrite_fail;		/* error in rewriting */
static int rew_level = 1;		/* nesting of rewrites */

/* function declarations */

static struct obj_info *do_rewrite_obj PROTO_ARGS((struct obj_info *, struct bind_info *));

#define rewrite_obj(t,b) (is_nf(t))?t:do_rewrite_obj(t,b)
#define has_conditions(eq_nr) equ[eq_nr].arity!=0
#define find_match(trm,b) next_match(trm,b,0)


void level_print()
{
    int i;

    printf("%2d)", rew_level);
    for (i = 0; i < rew_level; i++) {
	printf("  ");
    }
}


void delete_ref(trm)
    struct obj_info *trm;
{
    int i;
    int the_arity;

    if (is_function(trm)) {
	the_arity = fun[trm->index].arity;

	for (i = 0; i < the_arity; i++) {
	    dec_ref(trm->arg[i]);
	}

	if (the_arity > 0) {
	    /*
	     * free_arg(trm->arg,the_arity);
	     */
	    *(trm->arg) = fl_arg[--the_arity];
	    fl_arg[the_arity] = (struct obj_info *) trm->arg;
	}
    }
    free_obj(trm);
}


#if CAUTION
#define MONITOR_COMPARE 0

static int cmp_terms(t1, t2)		/* -1,0,1  t1<t2,t1=t2,t1>t2 */
    struct obj_info *t1, *t2;
{
    int i, result;

#if MONITOR_COMPARE
    printf("comparing: ");
    write_term(t1);
    printf(" & ");
    write_term(t2);
    printf("\n");
#endif

    if (t1->index < t2->index) {
	return (-1);
    }
    if (t1->index > t2->index) {
	return (1);
    }
    for (i = 0; i < fun[t1->index].arity; i++) {	/* match sub-trees */
	result = cmp_terms(t1->arg[i], t2->arg[i]);
	if (result != 0) {
	    return (result);
	}
    }
    return (0);
}

#endif


#define MONITOR_MATCH 0

/*
 * match two terms using bindings. t1 is a term to be rewritten, t2 is the lhs
 * of a possibly matching equation.
 */

static int match_terms(t1, t2, b)
    struct obj_info *t1, *t2;
    struct bind_info *b;
{
    int i;

#if MONITOR_MATCH
    printf("try to match: ");
    write_term(t1);
    printf(" & ");
    write_term(t2);
    printf("\n");
#endif

    if (is_function(t1)) {
	if (is_function(t2)) {

	    /* FUN & FUN */
	    if (t1->index != t2->index) {

#if MONITOR_MATCH
		printf("false\n");
#endif

		return (FALSE);
	    } else {
		for (i = 0; i < fun[t1->index].arity; i++) {	/* match sub-trees */
		    if (!match_terms(t1->arg[i], t2->arg[i], b)) {

#if MONITOR_MATCH
			printf("false\n");
#endif

			return (FALSE);
		    }
		}

#if MONITOR_MATCH
		printf("true\n");
#endif

		return (TRUE);
	    }

	} else {

	    /* FUN & VAR */
	    /* check type of both expressions */
	    /*
	     * This code probably not necessarry if (fun[t1->index].o_type !=
	     * var[t2->index].type) { return(FALSE); }
	     */
	    if (bound(t2->index, b)) {
		return (match_terms(t1, b->value[t2->index], b));
	    } else {
		bind_variable(t2->index, t1, b);

#if MONITOR_MATCH
		printf("true\n");
#endif

		return (TRUE);
	    }
	}
    }

#if DEBUG_INFO
    else {
	if (is_function(t2)) {
	    /* VAR & FUN */
	    return (FALSE);	/* no bindings the other way around */
	} else {
	    /* VAR & VAR */
	    if (var[t1->index].type != var[t2->index].type) {
		return (FALSE);
	    }
	    if (bound(t2->index, b)) {
		printf("variable already bound\n");
		return (match_terms(t1, b->value[t2->index], b));
	    } else {
		bind_variable(t2->index, t1, b);
		return (TRUE);
	    }
	}
    }
#endif

    return (FALSE);

}


static struct status_bits *process_term(trm)
    struct obj_info *trm;
{
    int i;
    struct status_bits *the_bits;

    trm->status.processed = TRUE;
    trm->status.nf = FALSE;

    if (is_variable(trm)) {
	trm->status.closed = FALSE;
	trm->status.nf = TRUE;
    } else if (fun[trm->index].arity == 0) {
	trm->status.closed = TRUE;
    } else {
	for (i = 0; i < fun[trm->index].arity; i++) {
	    the_bits = process_term(trm->arg[i]);
	    trm->status.closed = trm->status.closed && the_bits->closed;
	}
    }
    return (&(trm->status));
}


struct obj_info *copy_subst(obj, b)
    struct obj_info *obj;
    struct bind_info *b;
{
    struct obj_info *trm;
    int i;
    int narity;


#if DEBUG_INFO
    if (!is_processed(obj)) {
	printf("Unprocessed term: ");
	write_term(obj);
	printf("\n");
	(void) process_term(obj);
    }
#endif

    if (is_closed(obj)) {
	inc_ref(obj);
	return (obj);
    }
    if (is_function(obj)) {	/* simple copy the structure */

	alloc_obj(trm);		/* in case of function: copy top */
	trm->status = obj->status;
	trm->index = obj->index;
	trm->ref = 1;		/* reference count */
	narity = fun[obj->index].arity;
	if (narity > 0) {

	    if (fl_arg[narity - 1] == NULL) {
		trm->arg = PSF_NMALLOC(struct obj_info *, narity);
	    } else {
		trm->arg = (struct obj_info **) fl_arg[narity - 1];
		fl_arg[narity - 1] = *(trm->arg);
	    }

	    trm->status.closed = TRUE;	/* closed if children are too! */
	    for (i = 0; i < narity; i++) {	/* and copy subtrees */
		trm->arg[i] = copy_subst(obj->arg[i], b);
		trm->status.closed = trm->status.closed && trm->arg[i]->status.closed;
	    }
	}
	return (trm);		/* ready */
    } else {			/* link in value of variable */
	if (!bound(obj->index, b)) {
	    alloc_obj(trm);
	    trm->status.type = VAR_TYPE;
	    trm->status.closed = trm->status.nf = FALSE;
	    trm->index = obj->index;
	    trm->status.processed = TRUE;
	    trm->ref = 1;	/* reference count */
	    return (trm);
	}
	inc_ref(b->value[obj->index]);
	return (b->value[obj->index]);
    }
}


static int conditions_valid(the_eq, b)
    int the_eq;
    struct bind_info *b;
{
    int i, the_match;
    struct obj_info *r_obj, *l_obj;
    struct bind_info *next_bnd;
    struct bind_info *cond_bnd;


    cond_bnd = alloc_bindings();
    copy_bindings(cond_bnd, b);
    next_bnd = alloc_bindings();

#if TRACING
    if (trace_level() == HIGH) {
	level_print();
	printf("testing conditions of eq: ");
	write_term(equ[the_eq].lhs);
	printf(" = ");
	write_term(equ[the_eq].rhs);
	printf("\n");
	print_bindings(b);
    }
#endif

    for (i = 0; i < equ[the_eq].arity; i++) {	/* for all conditions */

#if TRACING
	if (trace_level() == HIGH) {
	    level_print();
	    printf("condition #%d: ", i + 1);
	    write_term(equ[the_eq].condition[i].lhs);
	    printf(" = ");
	    write_term(equ[the_eq].condition[i].rhs);
	    printf("\n");
	    print_bindings(cond_bnd);
	}
#endif

	/* copy RHS */

	rew_level++;

	r_obj = copy_subst(equ[the_eq].condition[i].rhs, cond_bnd);
	r_obj = rewrite_obj(r_obj, next_bnd);

#if TRACING
	if (trace_level() == HIGH) {
	    printf("after RHS\n");
	    print_bindings(cond_bnd);
	}
#endif

	l_obj = copy_subst(equ[the_eq].condition[i].lhs, cond_bnd);
	l_obj = rewrite_obj(l_obj, next_bnd);


#if TRACING
	if (trace_level() == HIGH) {
	    printf("after LHS\n");
	    print_bindings(cond_bnd);
	}
#endif

	rew_level--;

#if TRACING
	if (trace_level() == HIGH) {
	    level_print();
	    printf("condition: ");
	    write_term(l_obj);
	    printf(" = ");
	    write_term(r_obj);
	}
#endif

	if (is_closed(l_obj)) {
	    the_match = match_terms(l_obj, r_obj, cond_bnd);
	} else {
	    if (is_closed(r_obj)) {
		the_match = match_terms(r_obj, l_obj, cond_bnd);
	    } else {
		fflush(stdout);
		fprintf(stderr, "\n%s: error: free variables in condition\n", progname);
		exit(ERR_FREEVAR);
	    }
	}

	dec_ref(r_obj);		/* do away with redundant bindings */
	dec_ref(l_obj);

#if TRACING
	if (trace_level() == HIGH) {
	    if (the_match) {
		printf(" succeeds\n");
	    } else {
		printf(" fails\n");
		level_print();
		printf("equation \"%s\" fails\n", equ[the_eq].name);
	    }
	}
#endif

	if (!the_match) {
	    free_bindings(cond_bnd);
	    free_bindings(next_bnd);
	    return (FALSE);
	}
    }

#if TRACING
    if (trace_level() == HIGH) {
	level_print();
	printf("equation \"%s\" succeeds\n", equ[the_eq].name);
    }
#endif

    free_bindings(cond_bnd);
    free_bindings(next_bnd);
    return (TRUE);
}

/* global variable */
#if CAUTION
static int next_match_index;
#endif

static int next_match(trm, b, index)
    struct obj_info *trm;
    struct bind_info *b;
    int index;
{
    int prev_eq, the_eq, i;
    int eq_for_fn;

    eq_for_fn = fun[trm->index].eq_cnt;

    /*
     * printf("next match index: %d out of %d\n",index,eq_for_fn);
     */

    for (i = index; i < eq_for_fn; i++) {	/* for all functions */

	the_eq = fun[trm->index].eq[i];
	/*
	 * try to match term with lhs of function at hand
	 */
	if (match_terms(trm, equ[the_eq].lhs, b)) {
	    if (has_conditions(the_eq)) {
		if (!conditions_valid(the_eq, b)) {
		    clear_bindings(b);
		    continue;
		}
	    }
	    if (Option_Optimize) {
		/* update order of rewrite rules */

		equ[the_eq].matches++;
		if (i > 0) {
		    prev_eq = fun[trm->index].eq[i - 1];
		    if (equ[the_eq].matches > equ[prev_eq].matches) {
			fun[trm->index].eq[i] = prev_eq;
			fun[trm->index].eq[i - 1] = the_eq;
		    }
		}
	    }

#if CAUTION
	    if (Option_Cautious) {
		next_match_index = i + 1;
	    }
#endif

	    return (the_eq);
	}
	clear_bindings(b);
    }
#if CAUTION
    if (Option_Cautious) {
	next_match_index = NO_MATCH;
    }
#endif

    return (NO_MATCH);
}


static struct obj_info *do_rewrite_obj(obj, b)
    struct obj_info *obj;
    struct bind_info *b;
{
    int i;
    int matching_eq;
    struct obj_info *new_obj;

#if CAUTION
    int trm_cnt = 0, cycle_cnt = 0;
    int cmp_terms();
    int next_eq;
    DLL normal_forms;
    DLL_ITEM nf_ptr;
    int nf_cnt;
    int keep_index;
    int rewriting_done = FALSE;

#endif


#if 1
    /* this code not necessary for rewriting closed terms */
    if (is_variable(obj)) {	/* handle possible variables */
	if (bound(obj->index, b)) {
	    return (rewrite_obj(b->value[obj->index], b));
	} else {
	    return (obj);	/* just return the variable */
	}
    }
#endif

    do {

#if TRACING
	if (trace_level()) {
	    level_print();
	    printf("> ");
	    write_term(obj);
	    printf("\n");
	}
#endif

#if CAUTION
	if (Option_Cautious) {

	    if (tl_ptr = dll_contains(term_list, obj, cmp_terms)) {
		printf("\n%s: Error: non-terminating term rewriting system\n", progname);
		printf("%s: The term \"", progname);
		write_term(obj);
		printf("\" induces a cycle\n");

		for (; tl_ptr != NULL; tl_ptr = dll_go_fw(tl_ptr)) {
		    printf("  %d. ", cycle_cnt++);
		    write_term((struct obj_info *) dll_inspect(tl_ptr));
		    printf("\n");
		}
		printf("  %d. ", cycle_cnt++);
		write_term(obj);
		printf("\n\n");

		rewrite_fail = TRUE;

	    } else {
		inc_ref(obj);
		dll_append(term_list, (DLL_INFO) obj);

#if 0
		printf("adding: ");
		write_term(obj);
		printf("\n");
#endif

		trm_cnt++;
	    }
	}
	if (!rewrite_fail) {	/* this is tricky, see down */
#endif

	    rew_level++;

	    for (i = 0; i < fun[obj->index].arity; i++) {	/* rewrite subtrees
								 * first */
		obj->arg[i] = rewrite_obj(obj->arg[i], b);
	    }

	    rew_level--;

#if TRACING
	    if (trace_level() == HIGH) {
		level_print();
		printf(">> ");
		write_term(obj);
		printf("\n");
	    }
#endif


	    if ((matching_eq = find_match(obj, b)) == NO_MATCH) {	/* trm in nf! */

		obj->status.nf = TRUE;
	    } else {

#if CAUTION
		if (Option_Cautious) {
		    keep_index = next_match_index;
		    (void) next_match(obj, b, next_match_index);
		    if (next_match_index != NO_MATCH) {

#if TRACING
			if (trace_level() == HIGH) {
			    level_print();
			    printf("multiple matches\n");
			}
#endif

			normal_forms = dll_create();
			next_match_index = keep_index;
			nf_cnt = 0;

			while (keep_index != NO_MATCH) {

#if TRACING
			    if (trace_level() == HIGH) {
				level_print();
				printf("match #%d with eq: ", ++nf_cnt);
				write_term(equ[matching_eq].lhs);
				printf(" = ");
				write_term(equ[matching_eq].rhs);
				printf("\n");
				print_bindings(b);
			    }
#endif

			    new_obj = copy_subst(equ[matching_eq].rhs, b);
			    rew_obj = rewrite_obj(new_obj, b);
			    dll_insert_sorted_unique(normal_forms, rew_obj, cmp_terms);

			    /*
			     * 
			     * dll_insert_sorted_unique(normal_forms,rewrite_obj(
			     * new_obj,b),cmp_terms);
			     */

			    clear_bindings(b);
			    matching_eq = next_match(obj, b, keep_index);
			    keep_index = next_match_index;
			}
			if ((nf_cnt = dll_count(normal_forms)) != 1) {
			    rewrite_fail = TRUE;
			    printf("\n%s Error: non-confluent term rewriting system\n", progname);
			    printf("%s: The term \"", progname);
			    write_term(obj);
			    printf("\" has %d normal forms\n", nf_cnt);
			    nf_cnt = 0;
			    DLL_FORALL(normal_forms, nf_ptr) {
				printf("  %d. ", ++nf_cnt);
				write_term(dll_inspect(nf_ptr));
				printf("\n");
			    }
			    printf("\n");
			}
			dec_ref(obj);	/* do away with old term */
			clear_bindings(b);
			obj = dll_inspect(dll_go_first(normal_forms));
			dll_dispose(normal_forms);
			rewriting_done = TRUE;
		    } else {
			matching_eq = find_match(obj, b);	/* restore bindings */
			rewriting_done = FALSE;
		    }
		}
#endif


#if CAUTION
		if (!rewriting_done) {
#endif

#if TRACING
		    if (trace_level() == HIGH) {
			level_print();
			printf("match with eq: ");
			write_term(equ[matching_eq].lhs);
			printf(" = ");
			write_term(equ[matching_eq].rhs);
			printf("\n");
			print_bindings(b);
		    }
#endif


		    /*
		     * make a copy of rhs of matching equation substituting
		     * bound variables and rewrite this object
		     */

		    new_obj = copy_subst(equ[matching_eq].rhs, b);
		    dec_ref(obj);	/* do away with old term */
		    clear_bindings(b);
		    obj = new_obj;

#if CAUTION
		}
#endif

	    }

#if CAUTION
	}			/* this is tricky, see up */
#endif

    } while (!is_nf(obj) && !rewrite_fail);

#if CAUTION
    if (Option_Cautious) {
	for (i = 0; i < trm_cnt; i++) {

	    tmp = (struct obj_info *) dll_del_rear(term_list);

#if 0
	    printf("deleting: ");
	    write_term(tmp);
	    printf("\n");
#endif

	    dec_ref(tmp);
	}
    }
#endif

#if TRACING

#if CAUTION
    if (!rewriting_done) {
#endif

	if (trace_level()) {
	    level_print();
	    printf("= ");
	    write_term(obj);
	    printf("\n");
	}

#if CAUTION
    }
#endif

#endif

    return (obj);
}


struct obj_info *rewrite(obj)
    struct obj_info *obj;
{
    struct bind_info *bindings;
    struct obj_info *tmp;

    bindings = alloc_bindings();

    (void) process_term(obj);
    rewrite_fail = FALSE;	/* register loops in rewriting */

    tmp = rewrite_obj(obj, bindings);

    if (rewrite_fail) {
	tmp->status.failure = TRUE;
    } else {
	tmp->status.failure = FALSE;
    }

    free_bindings(bindings);

    return (tmp);
}


void process_trs()
{
    int i, j;


    fl_arg = PSF_NMALLOC(struct obj_info *, max_arity);

    for (i = 0; i < max_arity; i++) {
	fl_arg[i] = NULL;
    }

    for (i = 0; i < NR_EQU; i++) {
	process_term(equ[i].lhs);
	process_term(equ[i].rhs);
	for (j = 0; j < equ[i].arity; j++) {
	    process_term(equ[i].condition[j].lhs);
	    process_term(equ[i].condition[j].rhs);
	}
    }

#if CAUTION
    term_list = dll_create();
#endif
}
