/*
 * generate TRS tables from a TIL-file
 */

#include <stdio.h>
#include <assert.h>
#include "psf_prototype.h"
#include "psf_fopen.h"
#include "psf_malloc.h"
#include "psf_standards.h"

#include "tiltype.h"
#include "readtil.h"
#include "fieldex.h"
#include "dll.h"
#include "trs_check.h"
#include "tracing.h"
#include "version.h"
#include "initial.h"
#include "tables.h"
#include "rewrite.h"

#include "env_variables.h"

/* globals */

int Option_Constructors_Only = FALSE;
int Option_Contents = FALSE;
int Option_Dependency = FALSE;
int Option_Functions = FALSE;
int Option_Genesis = FALSE;
int Option_Rewrite = FALSE;
int Option_Size = 100;
int Option_Tracing = FALSE;
int Option_Verbose = FALSE;

int Option_Cautious = FALSE;

/* length of automatic trace in case of infinite branch */
int Option_Tracelength = 0;

int Option_Reverse = 0;
int Option_Optimize = 1;

char *filename;
char *progname;

/*
 * Error codes
 * 
 * 1 : wrong number of parameters 2 : can't open input file 3 : errors in til
 * input file 4 : errors in static check 5 : initial algebra too large
 * 
 */

#define CONTINOUS_OUTPUT 1

extern int optind;
extern char *optarg;

static void version()
{
    fprintf(stderr, "initial version %s, last compiled %s\n", 
			VERSION, __DATE__);
}

static void usage()
{
    fprintf(stderr, "usage: %s [-acdfgrv] [-s size] [til-file]\n", progname);
}

int main(argc, argv)
    int argc;
    char *argv[];
{
    psf_file pf;
    static suffix suffixes[] = {{TILSUFFIX, TILSUFFIX_DEFAULT}, {NULL, NULL}};
    struct module the_mod;
    int errors;
    int opt;

#if CONTINOUS_OUTPUT
    setbuf(stdout, NULL);
#endif


    /* process options */
    progname = psf_basename(argv[0]);
    while ((opt = getopt(argc, argv, "cdfgors:t:vh")) != EOF) {
	switch (opt) {
	case 'c':
	    Option_Contents = TRUE;
	    break;
	case 'd':
	    Option_Dependency = TRUE;
	    break;
	case 'f':
	    Option_Functions = TRUE;
	    break;
	case 'g':
	    Option_Genesis = TRUE;
	    break;
	case 'o':
	    Option_Constructors_Only = TRUE;
	    break;
	case 'r':
	    Option_Rewrite = TRUE;
	    break;
	case 's':
	    Option_Size = atoi(optarg);
	    break;
	case 't':
	    Option_Tracing = atoi(optarg);
	    switch (Option_Tracing) {
	    case 0:
	    case 1:
	    case 2:
		set_trace_level(Option_Tracing);
		break;
	    default:
		fprintf(stderr,
		    "%s: illegal trace level.  Legal values are 0, 1 and 2\n",
			progname);
		exit(1);
		break;
	    }
	    break;
	case 'v':
	    Option_Verbose = TRUE;
	    version();
	    break;
	case 'h':
	    usage();
	    exit(0);
	case '?':
	    usage();
	    exit(1);
	default:
	    assert(0);
	}
    }

    switch (argc - optind) {
    case 1:
	pf = psf_fopen(argv[optind], suffixes);
	filename = pf.name;
	if (read_module(&the_mod, pf.fp) == 1) {	/* read in til file */
	    fprintf(stderr, "error(s) in TIL format.\n");
	    exit(3);
	}
	fclose(pf.fp);		/* close til file */
	break;
    case 0:
	filename = "stdin";
	if (read_module(&the_mod, stdin) == 1) {	/* read in til file */
	    fprintf(stderr, "error(s) in TIL format.\n");
	    exit(3);
	}
	break;
    default:
	usage();
	exit(1);
    }
    errors = trs_check(&the_mod);
    if (errors) {
	fprintf(stderr, "input file contained %d error(s).\n", errors);
	exit(4);
    }
    init_trs(&the_mod);
    process_trs();
    initial();
}

void error_line()
{
    fprintf(stderr, "*** ");
}
