#include <stdio.h>
#include "psf_prototype.h"
#include "psf_standards.h"
#include "psf_malloc.h"

#include "dll.h"
#include "tables.h"
#include "objects.h"
#include "rewrite.h"
#include "tracing.h"
#include "options.h"
#include "initial.h"

static DLL_ITEM *counter;
static DLL *cnt_list;
static struct normal_form **nf_term;

static int *active;		/* is this sort still active */
static int *limit;		/* is size of I.A. bigger than size limit */
static int *shaky_base;		/* based on sort that went through limit */
static int *failure;		/* sort contains failure element */
static int *fail_base;		/* based on sort that went through limit */
static int *size;		/* the size of the initial algebra */
static int **dep;		/* dependency matrix for the sorts */
static int curr_arity;

static DLL *nf_list, *new_list;
static int rew_level = 0;

static void init_memory()
{
    int i, j;

    nf_list = PSF_NMALLOC(DLL, NR_SOR);
    new_list = PSF_NMALLOC(DLL, NR_SOR);
    active = PSF_NMALLOC(int, NR_SOR);
    size = PSF_NMALLOC(int, NR_SOR);
    shaky_base = PSF_NMALLOC(int, NR_SOR);
    limit = PSF_NMALLOC(int, NR_SOR);
    fail_base = PSF_NMALLOC(int, NR_SOR);
    failure = PSF_NMALLOC(int, NR_SOR);
    dep = PSF_NMALLOC(int *, NR_SOR);
    for (i = 0; i < NR_SOR; i++) {
	nf_list[i] = dll_create();
	new_list[i] = dll_create();
	active[i] = TRUE;
	size[i] = 0;
	shaky_base[i] = FALSE;
	limit[i] = FALSE;
	fail_base[i] = FALSE;
	failure[i] = FALSE;
	dep[i] = PSF_NMALLOC(int, NR_SOR);
	for (j = 0; j < NR_SOR; j++) {
	    dep[i][j] = FALSE;
	}
    }

    counter = PSF_NMALLOC(DLL_ITEM, max_arity);
    cnt_list = PSF_NMALLOC(DLL, max_arity);
    nf_term = PSF_NMALLOC(struct normal_form *, max_arity);
}

static void calc_dependency()
{
    int i, j;

    for (i = 0; i < NR_FUN; i++) {
	for (j = 0; j < fun[i].arity; j++) {
	    dep[fun[i].o_type][fun[i].i_type[j]] = TRUE;
	}
    }
}

static void print_dependency()
{
    int i, j, depending;

    printf("\n*** Dependency Report ***\n\n");

    for (i = 0; i < NR_SOR; i++) {
	printf("%s :", sor[i].name);
	depending = FALSE;
	for (j = 0; j < NR_SOR; j++) {
	    if (dep[i][j]) {
		printf(" %s", sor[j].name);
		depending = TRUE;
	    }
	}

	if (depending) {
	    printf("\n");
	} else {
	    printf("<none>\n");
	}
    }
}

static void print_nf()
{
    int i;
    DLL_ITEM ptr;
    struct normal_form *nf;

    printf("\n*** Initial Algebra Report ***\n\n");

    for (i = 0; i < NR_SOR; i++) {
	printf("'%s': %d element(s)\n",
	       sor[i].name, dll_count(nf_list[i]));

	if (failure[i]) {
	    printf("*** WARNING : sort contains erroneous element.\n");
	} else {
	    if (fail_base[i]) {
		printf("*** WARNING : (in)directly based on erroneous sort.\n");
	    }
	}

	if (limit[i]) {
	    printf("*** WARNING : possibly infinite sort.  ");
	    printf("Initial Algebra size limit exceeded.\n");
	} else {
	    if (shaky_base[i]) {
		printf("*** WARNING : (in)directly based on possibly infinite sort.\n");
	    }
	}

	if (Option_Contents) {
	    DLL_FORALL(nf_list[i], ptr) {
		nf = (struct normal_form *) dll_inspect(ptr);
		printf("- %d ", nf->level);
		write_term(nf->trm);
		printf("\n");
	    }
	}
	printf("\n");
    }
}

static int equal_terms(t1, t2)
    struct obj_info *t1, *t2;
{
    int i;

    if (t1->index != t2->index) {
	return (FALSE);
    } else {
	for (i = 0; i < fun[t1->index].arity; i++) {
	    if (!equal_terms(t1->arg[i], t2->arg[i])) {
		return (FALSE);
	    }
	}
	return (TRUE);
    }
}

static int add_nf(trm, index)
    struct obj_info *trm;
    int index;
{
    struct normal_form *nf, *nf_on_list;
    DLL_ITEM ptr;

    /* check first whether term is already on one of the lists */
    DLL_FORALL(nf_list[index], ptr) {
	nf_on_list = (struct normal_form *) dll_inspect(ptr);

	if (equal_terms(trm, nf_on_list->trm)) {
	    dec_ref(trm);
	    return (FALSE);
	}
    }

    DLL_FORALL(new_list[index], ptr) {
	nf_on_list = (struct normal_form *) dll_inspect(ptr);

	if (equal_terms(trm, nf_on_list->trm)) {
	    dec_ref(trm);
	    return (FALSE);
	}
    }

    nf = PSF_MALLOC(struct normal_form);
    nf->level = rew_level;
    nf->trm = trm;

    dll_append(new_list[index], (DLL_ITEM) nf);

    if (Option_Genesis) {
	printf("  '%s' : ", sor[index].name);
	write_term(trm);
	printf("\n");
	if (Option_Functions || Option_Rewrite) {
	    printf("\n");
	}
    }
    if (++size[index] > Option_Size) {
	limit[index] = TRUE;
	active[index] = FALSE;
    }
    return (TRUE);
}

/*
static void print_curr()
{
    int i;

    for (i = 0; i < curr_arity; i++) {
	printf("%d] ", i);
	write_term(nf_term[i]->trm);
	printf("\n");
    }
    printf("\n");
}
*/

static void add_constants()
{				/* add constants to the lists */
    int i;
    struct obj_info *tmp, *nf;

    for (i = 0; i < NR_FUN; i++) {
	if (fun[i].arity == 0) {

	    alloc_obj(tmp);
	    tmp->status.type = FUN_TYPE;
	    tmp->index = i;
	    tmp->status.closed = TRUE;
	    tmp->ref = 1;


	    if (Option_Rewrite) {
		write_term(tmp);
		printf(" -> ");
	    }
	    nf = rewrite(tmp);

	    if (is_failure(nf)) {
		failure[fun[i].o_type] = TRUE;
	    }
	    if (Option_Rewrite) {
		write_term(nf);
		printf("\n");
	    }
	    add_nf(nf, fun[i].o_type);
	}
    }
}

static int init_counters(index)
    int index;
{
    int i;

    curr_arity = fun[index].arity;

    for (i = 0; i < curr_arity; i++) {
	cnt_list[i] = nf_list[fun[index].i_type[i]];
	if ((counter[i] = dll_go_first(cnt_list[i])) == NULL) {
	    return (FALSE);
	} else {
	    nf_term[i] = dll_inspect(counter[i]);
	}
    }
    return (TRUE);
}

/*
static void print_perm()
{
    int i = FALSE;

    for (i = 0; i < curr_arity; i++) {
	write_term(nf_term[i]->trm);
	printf(" / ");
    }
    printf("\nn");
}
*/

static int valid_perm()
{
    int i;

    for (i = 0; i < curr_arity; i++) {
	if (nf_term[i]->level == rew_level - 1) {
	    return (TRUE);
	}
    }
    return (FALSE);
}

static int go_next_perm(index)
    int index;
{

    if (index < 0) {		/* all possibilities exhausted */
	return (FALSE);
    }
    if (dll_last(cnt_list[index], counter[index])) {
	counter[index] = dll_go_first(cnt_list[index]);
	nf_term[index] = dll_inspect(counter[index]);
	return (go_next_perm(index - 1));
    } else {
	counter[index] = dll_go_fw(counter[index]);
	nf_term[index] = dll_inspect(counter[index]);
	return (TRUE);
    }
}

static int next_perm()
{
    do {

	/*
	 * printf(".");
	 */

	if (go_next_perm(curr_arity - 1) == FALSE) {
	    return (FALSE);
	};

	/*
	 * print_perm();
	 */

    } while (!valid_perm());
    return (TRUE);
}

static void calculate_base()
{
    int i, j, curr_sort, curr_itype, change;

    for (i = 0; i < NR_FUN; i++) {
	curr_sort = fun[i].o_type;
	for (j = 0; j < fun[i].arity; j++) {
	    curr_itype = fun[i].i_type[j];
	    if (limit[curr_itype] == TRUE) {
		shaky_base[curr_sort] = TRUE;
	    }
	    if (failure[curr_itype] == TRUE) {
		fail_base[curr_sort] = TRUE;
	    }
	}
    }

    do {
	change = FALSE;
	for (i = 0; i < NR_FUN; i++) {
	    curr_sort = fun[i].o_type;
	    for (j = 0; j < fun[i].arity; j++) {
		curr_itype = fun[i].i_type[j];
		if ((shaky_base[curr_itype] == TRUE) &&
			(shaky_base[curr_sort] == FALSE)) {
		    shaky_base[curr_sort] = TRUE;
		    change = TRUE;
		}
		if ((fail_base[curr_itype] == TRUE) &&
			(fail_base[curr_sort] == FALSE)) {
		    fail_base[curr_sort] = TRUE;
		    change = TRUE;
		}
	    }
	}
    } while (change);
}

static void report_combinations(index)
    int index;
{
    int i, j;
    double combinations = 1;

    if (Option_Rewrite) {
	printf("\n");
    }
    i = index;

    printf("%3d. ", i);
    if (fun[i].operator == BINARY_OP) {
	printf("_");
    }
    printf("%s", fun[i].name);
    if (fun[i].operator) {
	printf("_");
    }
    printf(" : ");

    for (j = 0; j < fun[i].arity; j++) {
	combinations *= dll_count(nf_list[fun[i].i_type[j]]);
	if (j == 0) {
	    printf("%s ", sor[fun[i].i_type[j]].name);
	} else {
	    printf("# %s ", sor[fun[i].i_type[j]].name);
	}
    }
    printf("-> %s\n", sor[fun[i].o_type].name);
    printf("     %g combination(s)\n\n", combinations);
}

static void all_functions()
{
    int i, j, curr_sort;
    struct obj_info *trm, *nf;


    for (i = 0; i < NR_FUN; i++) {


	if (Option_Constructors_Only) {
	    if (!is_constructor(i)) {
		continue;
	    }
	}
	if (fun[i].arity != 0) {

	    /*
	     * report_combinations(i);
	     */

	    if (!active[(curr_sort = fun[i].o_type)]) {
		/*
		 * printf("--- Output Type Not Active ---\n");
		 */
		continue;
	    }
	    if (!init_counters(i)) {
		/*
		 * printf("--- Counter Initialization Failed ---\n");
		 */
		continue;
	    };

	    if (!valid_perm()) {
		if (!next_perm()) {
		    /*
		     * printf("--- No Valid Permutations ---\n");
		     */
		    continue;
		}
	    }
	    if (Option_Functions) {
		report_combinations(i);
	    }
	    /*
	     * printf("--- Calculating ---\n");
	     */

	    do {

		if (!active[curr_sort]) {
		    continue;
		}
		alloc_obj(trm);	/* new function element */
		trm->status.type = FUN_TYPE;
		trm->index = i;
		trm->status.closed = TRUE;
		trm->ref = 1;	/* reference count */
		if (curr_arity > 0) {
		    trm->arg = alloc_arg(curr_arity);

		    for (j = 0; j < curr_arity; j++) {	/* and link in subtrees */
			trm->arg[j] = nf_term[j]->trm;
			inc_ref(nf_term[j]->trm);
		    }
		}
		if (Option_Rewrite) {
		    write_term(trm);
		    printf(" -> ");
		}
		nf = rewrite(trm);

		if (is_failure(nf)) {
		    failure[curr_sort] = TRUE;
		}
		if (Option_Rewrite) {
		    write_term(nf);
		    printf("\n");
		}
		add_nf(nf, curr_sort);

	    } while (next_perm() && !limit[curr_sort]);
	}
    }
}

static void update_flags()
{
    int i, j;

    for (i = 0; i < NR_SOR; i++) {
	active[i] = FALSE;
	if (!limit[i]) {
	    for (j = 0; j < NR_SOR; j++) {
		if (dep[i][j]) {
		    active[i] = active[i] || !(dll_empty(new_list[j]));
		}
	    }
	}
    }
}

static int activity()
{
    int i;

    for (i = 0; i < NR_SOR; i++) {
	if (active[i]) {
	    return (TRUE);
	}
    }

    return (FALSE);
}


static void move_new_list()
{
    int i;
    DLL_ITEM ptr;

    for (i = 0; i < NR_SOR; i++) {
	DLL_FORALL(new_list[i], ptr) {
	    dll_append(nf_list[i], dll_inspect(ptr));
	}
	dll_flush(new_list[i]);
    }
}

static void next_segment()
{
    if (Option_Verbose) {
	if (Option_Genesis || Option_Rewrite || Option_Functions) {
	    printf("\n\n");
	}
	printf("*** Segment %d ***\n", rew_level);
	if (Option_Genesis || Option_Rewrite || Option_Functions) {
	    printf("\n");
	}
    }
}

void initial()
{
    init_memory();
    calc_dependency();

    if (Option_Dependency) {
	print_dependency();
    }
    next_segment();
    add_constants();

    update_flags();
    move_new_list();

    do {
	rew_level++;
	next_segment();

	all_functions();
	update_flags();
	move_new_list();
    } while (activity());

    calculate_base();

    print_nf();
}
