/*
	static semantic checks for term rewriting system:
*/


/*
	SYNOPSIS:
		#include "trs_check.h"
		int trs_check(module *mod);

	DESCRIPTION:
		trs_check examines the structure of the TRS embedded in a
		TIL file. It checks basically whether all variables actually
		get bound, after a succesful match of the LHS of the term
		with an equation from a TRS. Free variables are not allowed.

	RETURNS
		returns the number of errors encountered

	DIAGNOSTICS
		trs_check will write possible error messages to
		standard error.
*/


#include <stdio.h>
#include <string.h>
#include "psf_prototype.h"
#include "psf_malloc.h"
#include "tiltype.h"
#include "fieldex.h"
#include "trs_check.h"

/* declare local functions */

static void mark_vars(term,var_mask)	/* register vars of a term */
struct ae_term *term;
int *var_mask;
{
  int i;

  if (term->ind.table == VAR) {		/* if this is a var,  */
    var_mask[term->ind.key]++;		/* MARK it! */
  }

  for (i=0; i<term->a; i++) {		  /* for all possible arguments */
    mark_vars(&(term->ae_list[i]),var_mask);/* look for more vars in subtrees */
  }
}

int trs_check(mod)
struct module *mod;
{
  int i,j,k,nr_var,error_cnt=0;
  int l_free_var, r_free_var;
  int *rhs_mask,*lcond,*rcond,*explicit;

  nr_var = mod->entries_table[VAR];	/* number of vars in TIL file */
  rhs_mask = PSF_NMALLOC(int,nr_var+1); /* to hold marks for vars in RHS */
  lcond = PSF_NMALLOC(int,nr_var+1);	/* variables in LHS of condition */
  rcond = PSF_NMALLOC(int,nr_var+1);	/* variables in RHS of condition */
  explicit = PSF_NMALLOC(int,nr_var+1); /* explicit variables */


  for (i=1; i<=mod->entries_table[EQU]; i++) {	/* for all equations */
/*
    if (mod->equ[i].defined) {
*/

    /* test whether lhs of equation is a function */

      if (mod->equ[i].aet1.ind.table == VAR) {	/* LHS can't be a var */
	fprintf(stderr,"%s: left hand side of equation %s is a variable.\n",
	  progname, field_extract("n",mod->equ[i].ff,EQU,i));
	error_cnt++;
      }

    /* test whether rhs uses only variables introduced at lhs */
      for (j=1; j<=nr_var; j++) {		/* clear variable mask */
	explicit[j] = rhs_mask[j] = 0;	/* clear explicit flags */
      }

      mark_vars(&(mod->equ[i].aet1),explicit);/* mark vars in LHS in explicit */
      mark_vars(&(mod->equ[i].aet2),rhs_mask);/* mark vars in RHS in rhs_mask */

      for (j=0; j<mod->equ[i].a; j++) {	/* for all conditions */

	for (k=1; k<=nr_var; k++) {	/* clear variable mask */
	  lcond[k] = rcond[k] = 0;	/* for left/right condition */
	}

	mark_vars(&(mod->equ[i].guard[j].aet1),lcond); /* mark vars in left */
	mark_vars(&(mod->equ[i].guard[j].aet2),rcond); /* ... and right cond */

	l_free_var = r_free_var = 0;	/* no free variables */

	for (k=1; k<=nr_var; k++) {	/* for all variables */
	  if (!explicit[k]) {		/* if not already explicit */
	    if (rcond[k]) {		/* free in left cond ? */
	      r_free_var++;		/* register this */
	    }
	    if (lcond[k]) {		/* free in right cond */
	     l_free_var++;		/* register this */
	    }
	  }
	}

	if (l_free_var && r_free_var) {	/* both sides have free vars */
	  for (k=1; k<=nr_var; k++) {	/* for all vars */
	    if (rcond[k] || lcond[k]) {	/* if free either left/right */
	      fprintf(stderr,"%s: equation %s: ", /* report this */
		progname, field_extract("n",mod->equ[i].ff,EQU,i));
	      fprintf(stderr,"variable \"%s\" in conditions not explicit\n",
		field_extract("n",mod->var[k].ff,VAR,k));
	      error_cnt++;		/* increment number of errors so far */
	    }
	  }
	} else {			/* if not both sides have free vars */
	  for (k=1; k<=nr_var; k++) {	/* for all vars */
	    if (rcond[k] || lcond[k]) {	/* if either var in left/right */
	      explicit[k]++;		/* will be bound after match */
	    }
	  }
	}
      }

      for (j=1; j<=nr_var; j++) {	/* for all vars */
	if (rhs_mask[j] && !explicit[j]) {/* if var in RHS but not explicit */
	  fprintf(stderr,"%s: equation %s:",
	    progname, field_extract("n",mod->equ[i].ff,EQU,i));
	  fprintf(stderr," variable \"%s\" in right hand side not explicit.\n",
	    field_extract("n",mod->var[j].ff,VAR,j));
	  error_cnt++;			/* increment number of errors so far */
	}
      }
/*
    }
*/
  }
  return(error_cnt);			/* return #errors */
}
