/*
 * Make a list of all heads (atoms and communications), that may be
 * simulated at a certain moment.
 * This list is made by a walk down and up the process-tree.
 */

#include <stdio.h>
#include "psf_prototype.h"
#include "psf_malloc.h"
#include "tiltype.h"
#include "tilutil.h"
#include "process.h"
#include "eqm.h"
#include "msprint.h"
#include "headtail.h"
#include "sumport.h"
#include "prio.h"
#include "alloccontrol.h"
#include "choose.h"

/* Structure from which the list is built. */
struct phead {
    int tag;
    struct process *p;
    struct communication *c;
    int hide;
    struct phead *next;
};
/* define's for tag in struct phead */
#define HEAD_ATOM 0
#define HEAD_COMM 1

/*
 * Allocation/Free functions for communications.
 */
static struct communication *commFreeList = NULL;
static int commNew = 0;
static int commOld = 0;
static int commFree = 0;

static struct communication *CommAlloc()
{
    struct communication *c;

    if (commFreeList == NULL) {
	commNew ++;
	return(PSF_MALLOC(struct communication));
    } else {
	commOld ++;
	c = commFreeList;
	commFreeList = commFreeList->next;
	return(c);
    }
}

static void CommFree(c)
struct communication *c;
{
    c->next = commFreeList;
    commFreeList = c;
    commFree ++;
}

/*
 * Allocation/Free functions for elements of list.
 */
static struct phead *headFreeList = NULL;
static int headNew = 0;
static int headOld = 0;
static int headFree = 0;

static struct phead *HeadAlloc()
{
    struct phead *h;
    if (headFreeList == NULL) {
	headNew ++;
	return(PSF_MALLOC(struct phead));
    } else {
	headOld ++;
	h = headFreeList;
	headFreeList = headFreeList->next;
	return(h);
    }
}

static void HeadFree(h)
struct phead *h;
{
    h->next = headFreeList;
    headFreeList = h;
    headFree ++;
}

/*
 * HeadList manipulation functions.
 */
static struct phead *HeadListAdd(p, hl)
struct process *p;
struct phead *hl;
{
    struct phead *h;

    h = HeadAlloc();
    h->tag = HEAD_ATOM;
    h->p = p;
    h->hide = 0;
    h->next = hl;
    h->p->priority = 0;
    h->p->com_flag = 0;
    return(h);
}

static struct phead *HeadListCommAdd(c, hl)
struct communication *c;
struct phead *hl;
{
    struct phead *h;

    h = HeadAlloc();
    h->tag = HEAD_COMM;
    h->c = c;
    h->hide = 0;
    h->next = hl;
    h->c->priority = 0;
    return(h);
}

static struct phead *HeadListDelete(hl, h)
struct phead *hl;
struct phead *h;
{
    struct phead *x;

    if (h == hl) {
	hl = hl->next;
    } else {
	for (x = hl; x->next != h; x = x->next);
	x->next = h->next;
    }
    HeadFree(h);
    return(hl);
}

static struct phead *HeadListCombine(hl, l)
struct phead *hl;
struct phead *l;
{
    struct phead *h;

    if (hl == NULL)
	return(l);
    if (l == NULL)
	return(hl);
    for(h = l; h->next != NULL; h = h->next);
    h->next = hl;
    return(l);
}

/*
 * Deciding priorities of current heads.
 */
static struct phead *HeadListComputePriorities(p, hl)
struct process *p;
struct phead *hl;
{
    struct indexlist *priosets;
    struct phead *h;
    struct communication *c;
    int max_prio, prio;
    struct phead *prev;

    /* calculate highest priority */
    priosets = &p->h_t->tail->expr->proc_expr.pe5.sets;
    max_prio = 0;
    for(h = hl; h != NULL; h = h->next) {
	if (h->tag == HEAD_ATOM) {
	    if (h->hide)
		continue;
	    if (h->p->h_t->head->fun == SKP) {
		h->p->priority = 0;
		continue;
	    }
	    if (h->p->h_t->head->fun == SUM) {
		if (h->p->flag & PORT_PROCESS) {
		    h->p->priority = prio = priority_level(h->p->h_t->ae_t,
			priosets, SUM);
		} else {
		    h->p->priority = prio = MAX_PRIO;
		}
	    } else {
		if (h->p->priority < 0)
		    continue;
		h->p->priority = prio = priority_level(h->p->h_t->ae_t,
		    priosets, AET);
	    }
	    if (prio > max_prio)
		max_prio = prio;
	} else { /* HEAD_COMM */
	    if (h->hide)
		continue;
	    if (h->c->priority < 0)
		continue;
	    h->c->priority = prio = priority_level(h->c->aet, priosets, AET);
	    if (prio > max_prio)
		max_prio = prio;
	}
    }
    /* set priorities */
    h = hl;
    prev = hl;
    while (h != NULL) {
	if (h->hide) {
	    h->p->priority = 0;
	    prev = h;
	    h = h->next;
	    continue;
	}
	if (h->tag == HEAD_ATOM) {
	    if ((h->p->priority > 0 && h->p->priority < max_prio) ||
		(max_prio == MAX_PRIO && h->p->priority < MAX_PRIO)) {
		h->p->priority = -1;
		if (prev == h) {
		    h = h->next;
		    hl = h;
		    HeadFree(prev);
		    prev = h;
		} else {
		    prev->next = h->next;
		    HeadFree(h);
		    h = prev->next;
		}
	    } else {
		prev = h;
		h = h->next;
	    }
	} else { /* HEAD_COMM */
	    if ((h->c->priority > 0 && h->c->priority < max_prio) ||
		(max_prio == MAX_PRIO && h->c->priority < MAX_PRIO)) {
		h->c->priority = -1;
		if (prev == h) {
		    h = h->next;
		    hl = h;
		    HeadFree(prev);
		    prev = h;
		} else {
		    prev->next = h->next;
		    HeadFree(h);
		    h = prev->next;
		}
	    } else {
		prev = h;
		h = h->next;
	    }
	}
    }
    return(hl);
}

/*
 * Filter out encapsulated heads, and mark heads that are hidden.
 */
struct phead *HeadListComputeEncapsHide(sl, slPar, hl)
struct set_list *sl;
struct set_list *slPar;
struct phead *hl;
{
    struct phead *h, *next;
    struct set_list *s;

    for (h = hl; h != NULL; h = next) {
	next = h->next;
	if (h->hide)
	    continue;
	if (h->tag == HEAD_ATOM) {
	    if (h->p->priority == MAX_PRIO)
		continue;
	    if (h->p->encaps_flag != NULL) {
		for(s = sl; s != slPar; s = s->next) {
		    if (h->p->encaps_flag == s) {
			hl = HeadListDelete(hl, h);
			break;
		    }
		}
	    } else if (h->p->hide_flag != NULL) {
		for(s = sl; s != slPar; s = s->next) {
		    if (h->p->hide_flag == s) {
			h->hide = 1;
			break;
		    }
		}
	    }
	} else { /* HEAD_COMM */
	    if (h->c->priority == MAX_PRIO)
		continue;
	    if (h->c->encaps_flag != NULL) {
		for(s = sl; s != slPar; s = s->next) {
		    if (h->c->encaps_flag == s) {
			hl = HeadListDelete(hl, h);
			break;
		    }
		}
	    } else if (h->c->hide_flag != NULL) {
		for(s = sl; s != slPar; s = s->next) {
		    if (h->c->hide_flag == s) {
			h->hide = 1;
			break;
		    }
		}
	    }
	}
    }
    return(hl);
}

static struct communication *commList = NULL;

/*
 * Fill in a communication structure.
 */
static struct communication *CommCreate(p1, p2, aet, par, sub)
struct process *p1;
struct process *p2;
struct ae_term *aet;
struct process *par;
subst_t *sub;
{
    struct communication *c;

    c = CommAlloc();
    c->aet = aet;
    c->p_left = p1;
    c->p_right = p2;
    c->merge = par;
    c->sub = sub;
    c->next = commList;
    commList = c;
    c->encaps_flag = atom_in_set_list(aet, par->set, ENC);
    c->hide_flag = atom_in_set_list(aet, par->set, HID);
    if (c->hide_flag)
	msprintf("skip ");
    msprint_ae_term(aet);
    c->string = get_ms();
    alloc_string();
    return(c);
}

/*
 * Find a partner for communication.
 */
static struct phead *HeadListFindCommPartner(par, p, nrCom, ic, com, l, eqm,
    sub, gl)
struct process *par;
struct process *p;
int nrCom;
int ic;
struct com_tuple *com;
struct phead *l;
eqm_t *eqm;
subst_t *sub;
struct phead *gl;
{
    struct phead *h, *cl;
    struct communication *c;
    subst_t *rsub, *nsub;
    struct ae_term *aeth, *aet;

    cl = gl;
    for (h = l; h != NULL; h = h -> next) {
	if (h->tag == HEAD_COMM)
	    continue;
	if (h->hide)
	    continue;
	if (h->p->h_t->head->fun == SKP)
	    continue;
	if (h->p->h_t->head->fun == SUM) {
	    if ((p->flag & PORT_PROCESS) || ! (h->p->flag & PORT_PROCESS))
		continue;
	    if ((rsub = check_sumport_com(nrCom, 1 - ic, h->p->h_t->ae_t,
		&com->aet[1 - ic], sub, h->p->h_t->sub, h->p)) == NULL)
		continue;
	} else {
	    if (h->p->h_t->ae_t->ind.key != com->aet[1 - ic].ind.key)
		continue;
	    aeth = term_instantiate(&com->aet[1 - ic], sub, eqm);
	    rsub = term_match(aeth, h->p->h_t->ae_t);
	    ae_term_free(aeth);
	    if (rsub == NULL)
		continue;
	    if (!check_set_var(rsub)) {
		subst_free(rsub);
		continue;
	    }
	}
	aeth = term_instantiate(&com->aet[2], sub, eqm);
	aet = term_instantiate(aeth, rsub, eqm);
	ae_term_free(aeth);

	if (p->flag & PORT_PROCESS) {
	    if (! replace_var(rsub, &com->aet[ic], p->h_t->ae_t, p->sumvar,
		p->nr_sumvar)) {
		subst_free(rsub);
		ae_term_free(aet);
		continue;
	    }
	    c = CommCreate(p, h->p, aet, par, rsub);
	} else if (h->p->flag & PORT_PROCESS) {
/*
	    nsub = copy_add_sub_to_sub(sub, NULL);
	    if (! replace_var(nsub, &com->aet[1 - ic], h->p->h_t->ae_t,
		h->p->sumvar, h->p->nr_sumvar)) {
		subst_free(rsub);
		ae_term_free(aet);
		continue;
	    }
	    c = CommCreate(p, h->p, aet, par, nsub);
	    subst_free(rsub);
*/
	    c = CommCreate(p, h->p, aet, par, rsub);
	} else {
	    if (ic)
		c = CommCreate(h->p, p, aet, par, NULL);
	    else
		c = CommCreate(p, h->p, aet, par, NULL);
	    subst_free(rsub);
	}
	cl = HeadListCommAdd(c, cl);
/* uncomment this on replacing the old make head system
*/
	p->com_flag = h->p->com_flag = 1;
    }
    return(cl);
}

/*
 * Find communications between heads from hl and heads from l.
 */
static struct phead *HeadListComputeComm(par, hlNew, hlOth, mod, eqm, hl)
struct process *par;
struct phead *hlNew;
struct phead *hlOth;
struct module *mod;
eqm_t *eqm;
struct phead *hl;
{
    struct phead *h;
    int i;
    subst_t *sub;

    for (h = hlNew; h != NULL; h = h->next) {
	if (h->tag == HEAD_COMM)
	    continue;
	if (h->hide)
	    continue;
	if (h->p->h_t->head->fun == SKP)
	    continue;
	if (h->p->h_t->head->fun == SUM && ! (h->p->flag & PORT_PROCESS))
	    continue;
	for (i = 1; i <= mod->entries_table[COM]; i ++) {
	    if (h->p->flag & PORT_PROCESS) {
		if ((sub = check_sumport_com(i, 0, h->p->h_t->ae_t,
		    &mod->com[i].aet[0], NULL, h->p->h_t->sub, h->p)) == NULL)
		    goto right_side;
	    } else {
		if (h->p->h_t->ae_t->ind.key != mod->com[i].aet[0].ind.key)
		    goto right_side;
		if ((sub = term_match(&mod->com[i].aet[0], h->p->h_t->ae_t))
		    == NULL)
		    goto right_side;
		if (!check_set_var(sub)) {
		    subst_free(sub);
		    goto right_side;
		}
	    }
	    hl = HeadListFindCommPartner(par, h->p, i, 0, &mod->com[i], hlOth,
		eqm, sub, hl);
	    subst_free(sub);
    /*
     * Every now and then a goto must be used, to prevent it from
     * becoming obsolete.
     * Besides, whatever is the difference from a call to a function
     * with return statements in the middle.
     */
right_side:
	    if (h->p->flag & PORT_PROCESS) {
		if ((sub = check_sumport_com(i, 1, h->p->h_t->ae_t,
		    &mod->com[i].aet[1], NULL, h->p->h_t->sub, h->p)) == NULL)
		    continue;
	    } else {
		if (h->p->h_t->ae_t->ind.key != mod->com[i].aet[1].ind.key)
		    continue;
		if ((sub = term_match(&mod->com[i].aet[1], h->p->h_t->ae_t))
		    == NULL)
		    continue;
		if (!check_set_var(sub)) {
		    subst_free(sub);
		    continue;
		}
	    }
	    hl = HeadListFindCommPartner(par, h->p, i, 1, &mod->com[i], hlOth, 
		eqm, sub, hl);
	    subst_free(sub);
	}
    }
    return(hl);
}

/*
 * Make head-list for process p.
 */
static struct phead *HeadListCompute(p, mod, eqm)
struct process *p;
struct module *mod;
eqm_t *eqm;
{
    struct phead *hl, *hlChild;
    int i;

    if (p == NULL)
	return(NULL);
    if (p->flag & DEADLOCK || p->flag & DISABLED_PROCESS)
	return(NULL);
    hl = NULL;
    if (p->nr_children) {
	hl = HeadListCompute(p->child[0], mod, eqm);
	for (i = 1; i < p->nr_children; i ++) {
	    hlChild = HeadListCompute(p->child[i], mod, eqm);
	    switch(p->h_t->tail->expr->fun) {
	    case PAR:
		hl = HeadListComputeComm(p, hl, hlChild, mod, eqm, hl);
		hl = HeadListCombine(hl, hlChild);
		break;
	    case MRG:
		hl = HeadListComputeComm(p, hlChild, hl, mod, eqm, hl);
		hl = HeadListCombine(hl, hlChild);
		break;
	    default:
		hl = HeadListCombine(hl, hlChild);
		break;
	    }
	}
	switch(p->h_t->tail->expr->fun) {
	case PRIO:
	    hl = HeadListComputePriorities(p, hl);
	    break;
	default:
	    break;
	}
    } else { /* AET | SKP | SUM */
	hl = HeadListAdd(p, NULL);
    }
    if (p->parent != NULL) {
	if (p->set != p->parent->set) {
	    hl = HeadListComputeEncapsHide(p->set, p->parent->set, hl);
	}
    }
    return(hl);
}

static struct phead *headList = NULL;

void PrintHeadList()
{
    struct phead *hl;

    for(hl = headList; hl != NULL; hl = hl->next) {
	fprintf(stderr, "%s ", (hl->tag == HEAD_ATOM) ? "atom" : "comm");
	if (hl->hide)
	    fprintf(stderr, "hide ");
	if (hl->tag == HEAD_ATOM) {
	    fprintf(stderr, "%s", hl->p->h_t->string);
	} else {
	    fprintf(stderr, "%s", hl->c->string);
	}
	fprintf(stderr, "\n");
    }
}

static char buf[2048];
static int listsize;

static void MakeChooseList(hl)
struct phead *hl;
{
    ChooseListReset();

    for(; hl != NULL; hl = hl->next) {
	if (hl->tag == HEAD_ATOM) {
	    if (hl->p->h_t->head->fun == SKP)
		ChooseListAdd(0, "skip", hl->p->h_t->string);
	    else
		ChooseListAdd(hl->p->h_t->ae_t->ind.key, "atom", hl->p->h_t->string);
	} else
	    ChooseListAdd(hl->c->aet->ind.key, "comm", hl->c->string);
    }
}

void DestroyHeadList()
{
    struct phead *h;

    while (headList != NULL) {
	if (headList->tag == HEAD_COMM) {
	    ae_term_free(headList->c->aet);
	    free_string(headList->c->string);
	    subst_free(headList->c->sub);
	    CommFree(headList->c);
	}
	h = headList;
	headList = headList->next;
	HeadFree(h);
    }
    commList = NULL;
}

void MakeHeadList(p)
struct process *p;
{
    DestroyHeadList();
/*
fprintf(stderr, "comm    : %11d %11d %11d %11d\n", commNew, commOld, commFree, commNew + commOld - commFree);
fprintf(stderr, "head    : %11d %11d %11d %11d\n", headNew, headOld, headFree, headNew + headOld - headFree);
*/

    headList = HeadListCompute(p, get_til_module(), get_eqm());

    MakeChooseList(headList);
}

struct communication *GetCommList()
{
    return(commList);
}

void GetChosenHead(nr, pp, pc)
int nr;
struct process **pp;
struct communication **pc;
{
    struct phead *hl;

    for(hl = headList; nr > 0; hl = hl->next, nr --);
    if (hl->tag == HEAD_ATOM) {
	*pp = hl->p;
	*pc = NULL;
    } else {
	*pp = NULL;
	*pc = hl->c;
    }
}

int GetHeadListNr(p, c)
struct process *p;
struct communication *c;
{
    int nr = 0;
    struct phead *hl;

    for(hl = headList; ; hl = hl->next, nr ++) {
	if (hl->tag == HEAD_ATOM) {
	    if (hl->p == p)
		return(nr);
	} else {
	    if (hl->c == c)
		return(nr);
	}
    }
}

int HeadListSum()
{
    struct phead *hl;

    for(hl = headList; hl != NULL; hl = hl->next) {
	if (hl->tag == HEAD_ATOM) {
	    if ((hl->p->flag & PORT_PROCESS) && hl->p->priority == MAX_PRIO)
		return(1);
	    if (hl->p->h_t->head->fun == SUM)
		return(1);
	}
    }
    return(0);
}
