
# package Anim - animation routines for simanim
#
# Author: Bob Diertens
# Version: 0.17
# Date: Januari 7, 2022
#    Set have_text to 0 in the evaluation of namespace to make it possible to
#    having only an inof window.
# Version: 0.16
# Date: April 9, 2019
#    Complaint about using geometry manager pack in grid.
#    So, changed BoxInit to use full paths ($W.frtext.fri.info).
# Version: 0.15
# Date: November 27, 2012
#    Set font to TkDefaultFont using TIP145
# Version: 0.14
# Date: September 26, 2012
#    Enclosed font in \" in CreateText to allow for spaces in fontname.
# Version: 0.13
# Date: November 02, 2006
#    removed ToolBus control (to tbanim.tcl) and changed initialization
#    so it can be used inside another application with only the animation
#    window.
#    CreateText now first does a DeleteText.
# Version: 0.12
# Date: July 05, 2006
#    Added option -offset to CreateItem and TextposItem for fine adjustment
#    of the text positions.
# Version: 0.11
# Date: June 30, 2006
#    Borders of items are now given the active color.
# Version: 0.10
# Date: March 02, 2006
#    Added CreatePLine using only positions, and Chop and Locate for
#    calculation of positions.
# Version: 0.9
# Date: March 01, 2006
#    Added coor in addition to item and pos for CreateLine
# Version: 0.8
# Date: February 10, 2005
#    Changed the behavior of the popups. Now you hav to hold the mouse-button,
#    and with a release the popup dissappears. This also works under
#    different window systems.
# Version: 0.7
# Date: November 22, 2001
#    Added option -speed for Anim::Windows
# Version: 0.6
# Date: March 14, 2001
#    Added function ReoriginItem in order to get the right dimensions when running.
# Version: 0.5
# Date: August 31, 2000
#    Function names are changed to Hungarian notation.
#    ( CreateLine instead of create_line )
# Version: 0.4
# Date: July 28, 2000
#    Beneath the window of a queue, a rectangle is drawn so that
#    dumping the canvas window at least shows something here.
#    (The canvas is not capable of printing PostScript for sub-windows).
# Version: 0.3
# Date: June 29, 2000
#    Added resizing when window does not fit on screen.
#    Added enlargement and centering of items on canvas when canvas is
#    smaller than its frame.
#    Added raising of lines when activated.
# Version: 0.2
# Date: April 5, 2000
#    Adjusted for use with simanim.
#    Use of a standard font for canvas.
#    Added adjustable move speed.
# Version: 0.1
# Date: September 30, 1999
#    Turned it into a package for use in the Software Renovation Factory.
# version: 0.0.10  date: 07/05/1999
# version: 0.0.9  date: 25/11/1997
# version: 0.0.8  date: 29/10/1997
# version: 0.0.7  date: 22/09/1997
# version: 0.0.6  date: 12/08/1997
# version: 0.0.5  date: 06/08/1997
# version: 0.0.4  date: 25/07/1997
# version: 0.0.3  date: 24/07/1997
# version: 0.0.2  date: 24/07/1997
# version: 0.0.1  date: 25/03/1997

namespace eval Anim {
    variable anim
    variable W
    variable Wca
    variable Wte
    variable color
    variable move
    variable indexvar
    variable item
    variable index
    variable dimlist
    variable dimllist
    variable line
    variable textpos
    variable queue
    variable box
    variable var
    variable list
    variable have_text
    variable print
    variable speed
    variable callback
    variable registerchoice
    variable registerindex

#    namespace export Init Windows
    namespace export Windows Colorset CreateItem DestroyItem DestroyFreeItems \
	ChangeTextItem Chop Locate Dim DimL DimQ DimS CreatePLine CreateLine \
	DestroyLine DestroyFreeLines ActivateLine DeactivateLine ClearLines \
	AddClear ClearInd Clear ClearItems AdjustTextpos TextposItem \
	TextposLine Textpos CreateText DeleteText ActivateItem ActivateItemInd \
	DeactivateItem CreateQueue AddQueue SubQueue DeleteQueue ClearQueues \
	CreateStack PushStack PopStack DeleteStack CreateBox CreateLabel \
	InitVar InitArray ClearVars ActivateList DestroyList AddList EmptyList \
	ClearLists \
	MoveDelay MoveFast MoveLeft MoveRight MoveUp MoveDown \
	ChangeTag CombineTag Move
    package provide Anim 0.3

    set anim(prog_name) [info script]

    # line on top of all other lines
    set anim(topline) 0
    set anim(notext) 0
    set registerchoice 0
    set registerindex -1

    set dimlist [list x y ce n e s w nw ne se sw n,y s,y e,x w,x wid ht]
    set dimllist [list start end start,x start,y end,x end,y]

    set W ""
    set have_text 0
}

# error

proc Anim::StackDump {} {
    for {set i [info level]; incr i -2} {$i > 0} {incr i -1} {
        puts stderr "$i: [info level $i]"
    }
}

proc Anim::Error {func code message} {
    variable anim

    puts stderr "$anim(prog_name): $func: $message"
    if {$code} {
	Anim::StackDump
	TBsend "snd-value(error)"
	exit $code
    }
}

# colorsets

proc Anim::Color {} {
    variable color

    return $color(screen)
}

proc Anim::Colorset {set type c1 c2} {
    variable color

    if {[array names color normal,$set,$type] == ""} {
	# fill new set with values from set 0
	set cl [array names color *,0,*]
	foreach el $cl {
	    regexp {^(.*),0,(.*)$} $el match arg1 arg2
	    set color($arg1,$set,$arg2) $color($arg1,0,$arg2)
	}
    }
    if {[lsearch -exact [list rect oval line text] $type] == -1} {
	Anim::Error Anim::Colorset 1 "unkown type '$type'"
    } else {
	set color(normal,$set,$type) $c1
	set color(active,$set,$type) $c2
    }
}

proc Anim::ColorListbox {c1 c2} {
    variable color

    set color(listbox) $c1
    set color(listboxselect) $c2
}

proc Anim::Init {args} {
    variable W
    variable Wca
    variable Wte
    variable color
    variable anim
    variable move
    variable callback
    variable registerchoice

    if {$args != ""} {
	set W [lindex $args 0]
	set callback [lindex $args 1]
	for {set i 2; set opt [lindex $args $i]} { $opt != ""} \
	    {incr i; set opt [lindex $args $i]} {
	    switch -exact -- $opt {
		-notext {
		    set anim(notext) 1
		}
		-register {
		    set registerchoice 1
		}
		default {
		    Anim::Error Anim::Init 1 "unknown option '$opt'"
		}
	    }
	}
	return
    }
    if {$W == ""} {
	set W .anim
    }
    set Wca $W.frcanvas.frc.ca
    set Wte $W.frtext.t.te

    # default settings
    set color(set) 0
    if {[winfo depth .] == 1} { # monochrome
	set color(screen) 0
	set color(normal,0,rect) #ffffff
	set color(normal,0,oval) #ffffff
	set color(normal,0,line) gray50
	set color(active,0,rect) gray25
	set color(active,0,oval) gray25
	set color(active,0,line) ""
	set color(normal,0,text) #000000
	set color(active,0,text) #000000
	set color(normal,1,rect) #000000
	set color(normal,1,oval) #000000
	set color(normal,1,line) gray50
	set color(active,1,rect) gray50
	set color(active,1,oval) gray50
	set color(active,1,line) ""
	set color(normal,1,text) #ffffff
	set color(active,1,text) #ffffff
	Anim::Colorset 2 rect #000000 #000000
    } else { # color
	set color(screen) [winfo depth .]
	set color(normal,0,rect) lightskyblue
	set color(normal,0,oval) lightskyblue
	set color(normal,0,line) gray50
	set color(active,0,rect) royalblue
	set color(active,0,oval) royalblue
	set color(active,0,line) red
	set color(normal,0,text) #000030
	set color(active,0,text) #ffffff
	set color(normal,1,rect) green
	set color(normal,1,oval) green
	set color(normal,1,line) gray50
	set color(active,1,rect) darkgreen
	set color(active,1,oval) darkgreen
	set color(active,1,line) red
	set color(normal,1,text) #003000
	set color(active,1,text) #ffffff
	set color(listbox) yellow1
	set color(listboxselect) yellow3
    }

    # delay before a move
    set move(delay) 5
    set move(fast) 0

    #set anim(cafont) "7x13bold"
    set tip145 [catch {font create TkDefaultFont}]
    if {!$tip145} {
        if {![catch {tk::pkgconfig get fontsystem} F(fs)] && $F(fs) eq "xft"} {
            set F(family) "sans-serif"
            set F(fixed)  "monospace"
        } else {
            set F(family) "Helvetica"
            set F(fixed)  "courier"
        }
        set F(size) -12
        font configure TkDefaultFont -family $F(family) -size $F(size)
    }
    set anim(cafont) TkDefaultFont

    # other initialisations
    Anim::IndexInit
}

# setup windows
proc Anim::InfoInit {} {
    variable W

    if {! [winfo exists $W.frtext]} {
	frame $W.frtext
	# pack $W.frtext
	grid $W.frtext -row 1 -column 0 -rowspan 1 -columnspan 1 \
	    -sticky news -padx 0 -pady 0
    }
}

proc Anim::BoxInit {} {
    variable W
    variable box

    if {! [winfo exists $W.frtext.fri.info]} {
	Anim::InfoInit

	frame $W.frtext.fri -borderwidth 1 -relief raised
	pack $W.frtext.fri -expand 1 -fill both
	pack $W.frtext.fri -in $W.frtext -side left
	frame $W.frtext.fri.info
	pack $W.frtext.fri.info -padx 2 -pady 2
	set box(info,w) $W.frtext.fri.info
	set box(info,side) top
	set box(info,fill) both
	set box(info,expand) 1
	pack $box(info,w) -in $W.frtext.fri -side $box(info,side) \
	    -fill $box(info,fill) -expand $box(info,expand)
    }
}

proc Anim::InitPrint {} {
    variable print

    toplevel .print
    wm withdraw .print
    wm transient .print .

    set f .print.file
    frame $f -relief raised -bd 1
    pack $f
    label $f.l -text "File" -width 5 -anchor w
    entry $f.e -width 20 -textvariable Anim::print(file)
    set Anim::print(file) "canvas.epsf"
    pack $f.l $f.e -in $f -side left -pady 1
    pack $f -in .print -fill both -expand 1
    set f .print.orient
    frame $f -relief raised -bd 1
    pack $f
    label $f.l -text "Orientation" -width 12 -anchor w
    frame $f.f -bd 2
    radiobutton $f.f.p -text "Portrait" -variable Anim::print(orient) -value 0
    radiobutton $f.f.l -text "Landscape" -variable Anim::print(orient) -value 1
    $f.f.p select
    pack $f.f.p $f.f.l -in $f.f -anchor w
    pack $f.l $f.f -in $f -side left -anchor n
    pack $f -in .print -fill both -expand 1
    set f .print.color
    frame $f -relief raised -bd 1
    pack $f
    label $f.l -text "Colormode" -width 12 -anchor w
    frame $f.f -bd 2
    radiobutton $f.f.c -text "Color" -variable Anim::print(color) -value "color"
    radiobutton $f.f.g -text "Gray" -variable Anim::print(color) -value "gray"
    radiobutton $f.f.m -text "Mono" -variable Anim::print(color) -value "mono"
    $f.f.c select
    pack $f.f.c $f.f.g $f.f.m -in $f.f -anchor w
    pack $f.l $f.f -in $f -side left -anchor n
    pack $f -in .print -fill both -expand 1
    set f .print.end
    frame $f -relief raised -bd 1
    pack $f
    button $f.print -text "Print" \
	-command {
	    Anim::CanvasPS $Anim::Wca
	    Anim::PopdownPrint
	}
    button $f.cancel -text "Cancel" -command { Anim::PopdownPrint }
    pack $f.print $f.cancel -in $f -side left -fill none -expand 1
    pack $f -in .print -fill both -expand 1
}

proc Anim::PopupPrint {} {
    set x [expr ([winfo pointerx .] % [winfo screenwidth .]) - 30]
    set y [expr ([winfo pointery .] % [winfo screenheight .]) - 30]
    set w [expr [winfo screenwidth .] - [winfo reqwidth .print] - 2]
    set h [expr [winfo screenheight .] - [winfo reqheight .print] - 2]
    if {$x > $w} {
	set x $w
    }
    if {$y > $h} {
	set y $h
    }
    wm geometry .print +$x+$y
    wm deiconify .print
    tkwait visibility .print
    raise .print
    grab .print
}

proc Anim::PopdownPrint {} {
    wm withdraw .print
    grab release .print
}

proc Anim::CanvasPS {ca} {
    variable Wca
    variable print

    set bbox [$Wca bbox all]  
    puts stderr "dump $Wca"
    set x [lindex $bbox 0]
    set y [lindex $bbox 1]
    set w [expr [lindex $bbox 2] - $x]
    set h [expr [lindex $bbox 3] - $y]
    $Wca postscript -width $w -height $h -x $x -y $y \
	-rotate $Anim::print(orient) -colormode $Anim::print(color) \
	-file "$Anim::print(file)"
}

proc Anim::Resize {} {
    variable Wca

    bind $Wca <Visibility> ""
    set W [winfo screenwidth .]
    set H [winfo screenheight .]
    set scrh [expr [winfo height .] - $H]
    set scrw [expr [winfo width .] - $W]
    if {$scrh > 0 || $scrw > 0} {
	if {$scrh > 0} {
	    set canvash [expr [winfo height $Wca] - $scrh - 30]
	    $Wca configure -height $canvash
	}
	if {$scrw > 0} {
	    set canvash [expr [winfo width $Wca] - $scrw]
	    $Wca configure -width $canvash
	}
	set x [expr ([winfo pointerx .] / $W) * $W + 1]
	set y [expr ([winfo pointery .] / $H) * $H + 1]
	wm geometry . +$x+$y
    }
    update
    Anim::Replace
}

proc Anim::Replace {} {
    variable Wca

    # if the frame of the canvas is larger than the canvas, enlarge
    # the canvas and center the items on the canvas
    set cw [winfo width $Wca]
    set ch [winfo height $Wca]
    set p [winfo parent $Wca]
    # the borderwidth of the frame is 1
    set pw [expr [winfo width $p] - 2]
    set ph [expr [winfo height $p] - 2]
    if {$cw < $pw} {
	set w $pw
	$Wca configure -width $w
	$Wca move all [expr ($pw - $cw) / 2] 0
	Anim::ReoriginText [expr ($pw - $cw) / 2] 0
	Anim::ReoriginItem [expr ($pw - $cw) / 2] 0
    } else {
	set w $cw
    }
    if {$ch < $ph} {
	set h $ph
	$Wca configure -height $h
	$Wca move all [expr ($ph - $ch) / 2] 0
	Anim::ReoriginText 0 [expr ($ph - $ch) / 2]
	Anim::ReoriginItem 0 [expr ($ph - $ch) / 2]
    } else {
	set h $ch
    }
    Anim::UpdateScrollregion
#    update
#    set cw [winfo width $Wca]
#    set ch [winfo height $Wca]
#    set pw [winfo width $p]
#    set ph [winfo height $p]
#    puts stderr "$cw x $ch  $pw x $ph"
}

proc Anim::TextDisplay {s} {
    variable Wte
    variable have_text

    if {$have_text} {
	$Wte insert insert $s
	$Wte yview moveto 1
    }
}

proc Anim::Windows {canvasw canvash args} {
    variable color
    variable anim
    variable move
    variable W
    variable Wca
    variable Wte
    variable box
    variable have_text
    variable speed

    set have_text 0
    set speed 0
    for {set i 0; set opt [lindex $args $i]} { $opt != ""} \
	{incr i; set opt [lindex $args $i]} {
	switch -exact -- $opt {
	    -text {
		if {! $anim(notext)} {
		    set have_text 1
		}
		incr i
		set textw [lindex $args $i]
		incr i
		set texth [lindex $args $i]
	    }
	    -speed {
		set speed 1
	    }
	    default {
		Anim::Error Anim::Windows 1 "unknown option '$opt'"
	    }
	}
    }

    Anim::Init

    # build windows

    ###
    frame $W.frcanvas -borderwidth 1 -relief raised
    pack $W.frcanvas -ipadx 2 -ipady 2
    frame $W.frcanvas.frc -borderwidth 1 -relief sunken
    pack $W.frcanvas.frc
#    set scrw [expr [winfo screenwidth .] - 16]
#    if {$canvasw > $scrw} {
#	set canvasw $scrw
#    }
#    set scrh [expr [winfo screenheight .] - 61]
#    if {$canvash > $scrh} {
#	set canvash $scrh
#    }
    canvas $Wca -scrollregion [list 0 0 $canvasw $canvash] \
	-width $canvasw -height $canvash -borderwidth 0 -relief sunken \
	-highlightthickness 0 -takefocus 0 \
	-xscrollcommand " $W.frcanvas.xs set " \
	-yscrollcommand " $W.frcanvas.ys set "
    set anim(canvas,width) $canvasw
    set anim(canvas,height) $canvash
    pack $Wca -in $W.frcanvas.frc
    scrollbar $W.frcanvas.xs -orient horizontal -width 8 \
	-highlightthickness 0 -takefocus 0 -command { $Anim::Wca xview } -bd 1
    scrollbar $W.frcanvas.ys -orient vertical -width 8 \
	-highlightthickness 0 -takefocus 0 -command { $Anim::Wca yview } -bd 1
    grid $W.frcanvas.frc  -row 0 -column 0 -rowspan 1 \
        -columnspan 1 -sticky news -padx 0 -pady 0
    grid $W.frcanvas.ys -row 0 -column 1 -rowspan 1 \
        -columnspan 1 -sticky ns -padx 0 -pady 0
    grid $W.frcanvas.xs -row 1 -column 0 -rowspan 1 \
        -columnspan 1 -sticky ew -padx 0 -pady 0
    grid rowconfig    $W.frcanvas 0 -weight 1 -minsize 0
    grid rowconfig    $W.frcanvas 1 -weight 0
    grid columnconfig $W.frcanvas 0 -weight 1 -minsize 0
    grid columnconfig $W.frcanvas 1 -weight 0

    ###
    grid $W.frcanvas -row 0 -column 0 -rowspan 1 -columnspan 1 -sticky news \
        -padx 0 -pady 0
    grid rowconfig    $W 0 -weight 1 -minsize 0
    grid columnconfig $W 0 -weight 1 -minsize 0

    if {$have_text} {
	Anim::InfoInit
	frame $W.frtext.t -borderwidth 1 -relief raised
	pack $W.frtext.t -ipadx 2 -ipady 2 -expand 1 -fill both
	text $Wte -width $textw -height $texth \
	    -yscrollcommand " $W.frtext.t.ys set " \
	    -highlightthickness 0 -takefocus 0 -bd 1
	scrollbar $W.frtext.t.ys -command { $Anim::Wte yview } -width 8 \
	    -highlightthickness 0 -takefocus 0 -bd 1
	grid $Wte  -row 0 -column 0 -rowspan 1 \
	    -columnspan 1 -sticky news -padx 0 -pady 0
	grid $W.frtext.t.ys  -row 0 -column 1 -rowspan 1 \
	    -columnspan 1 -sticky news -padx 0 -pady 0
	pack $W.frtext.t -in $W.frtext -side left
    }

    # bindings
    bind $Wca <ButtonPress-2> "$Anim::Wca scan mark %x %y"    
    bind $Wca <B2-Motion> "$Anim::Wca scan dragto %x %y"
#    bind all <ButtonPress-3> {+ Anim::CanvasPS %W }
    bind all <ButtonPress-3> {+ Anim::PopupPrint }
    bind $Wca <Visibility> { Anim::Resize }

    Anim::InitPrint
}

proc Anim::DisableButtons {} {
}

proc Anim::EnableButtons {} {
    variable anim

}

proc Anim::DisableControlbutton {} {
}

proc Anim::UpdateScrollregion {} {
    variable Wca

    set bbox [$Wca bbox all]  
    if {$bbox == {}} {
	set bbox {0 0 0 0}
    } else {
	set bbox "0 0 [expr [lindex $bbox 2] + [lindex $bbox 0]] [expr [lindex $bbox 3] + [lindex $bbox 1]]"
    }
    $Wca configure -scrollregion $bbox
}

proc Anim::View {x y} {
    variable anim
    variable Wca

    set bbox [$Wca bbox all]
    set x [expr $x - $anim(canvas,width) / 2]
    set y [expr $y - $anim(canvas,height) / 2]
    $Wca xview moveto [expr $x.0 / [lindex $bbox 2]]
    $Wca yview moveto [expr $y.0 / [lindex $bbox 3]]
}

proc Anim::ViewX {x} {
    variable anim
    variable Wca

    set bbox [$Wca bbox all]
    set x [expr $x - $anim(canvas,width) / 2]
    $Wca xview moveto [expr $x.0 / [lindex $bbox 2]]
}

proc Anim::ViewY {y} {
    variable anim
    variable Wca

    set bbox [$Wca bbox all]
    set y [expr $y - $anim(canvas,height) / 2]
    $Wca yview moveto [expr $y.0 / [lindex $bbox 3]]
}

# graphics routines

# index package for items
proc Anim::IndexInit {} {
    variable indexvar

    set indexvar(list) ""
    set indexvar(nr) 0
}

proc Anim::GetIndex {} {
    variable indexvar

    if {[llength $indexvar(list)]} {
	set ind [lindex $indexvar(list) 0]
	set indexvar(list) [lreplace $indexvar(list) 0 0]
	return $ind
    }
    set ind $indexvar(nr)
    incr indexvar(nr)
    return $ind
}

proc Anim::FreeIndex {ind} {
    variable indexvar

    lappend indexvar(list) $ind
}
# end index package

proc Anim::CreateItem {it type x y w h text args} {
    variable item
    variable index
    variable color
    variable Wca
    variable anim

    # options
    set list 1
    set free 0
    set colorset $color(set)
    set tags ""
    set anchor center
    set offsetx 0
    set offsety 0
    for {set i 0; set opt [lindex $args $i]} { $opt != ""} \
	{incr i; set opt [lindex $args $i]} {
	switch -exact -- $opt {
	    -nolist {
		set list 0
	    }
	    -free {
		set free 1
	    }
	    -color {
		incr i
		set colorset [lindex $args $i]
	    }
	    -tag {
		incr i
		set tags "$tags [lindex $args $i]"
	    }
	    -anchor {
		incr i
		set anchor [lindex $args $i]
	    }
	    -offset {
		incr i
		set offsetx [lindex $args $i]
		incr i
		set offsety [lindex $args $i]
	    }
	    default {
		Anim::Error Anim::CreateItem 1 "unknown option '$opt'"
	    }
	}
    }

    set ind [Anim::GetIndex]
    set index($it) $ind

    # register item
    set item($ind,type) $type
    set item($ind,x) $x
    set item($ind,y) $y
    set item($ind,wid) $w
    set item($ind,ht) $h
    set item($ind,ce) [concat $x $y]
    set item($ind,n)  [concat $x [expr $y - $h]]
    set item($ind,s)  [concat $x [expr $y + $h]]
    set item($ind,e)  [concat [expr $x + $w] $y]
    set item($ind,w)  [concat [expr $x - $w] $y]
    if {$type == "rect"} {
	set item($ind,ne)  [concat [expr $x + $w] [expr $y - $h]]
	set item($ind,nw)  [concat [expr $x - $w] [expr $y - $h]]
	set item($ind,se)  [concat [expr $x + $w] [expr $y + $h]]
	set item($ind,sw)  [concat [expr $x - $w] [expr $y + $h]]
    } elseif {$type == "oval"} {
	set corner [expr atan2($h, $w)]
	set dx [expr round($w*cos($corner))]
	set dy [expr round($h*sin($corner))]
	set item($ind,nw) [concat [expr $x - $dx] [expr $y - $dy] ]
	set corner [expr atan2($h, -$w)]
	set dx [expr round($w*cos($corner))]
	set dy [expr round($h*sin($corner))]
	set item($ind,ne) [concat [expr $x - $dx] [expr $y - $dy] ]
	set corner [expr atan2(-$h, $w)]
	set dx [expr round($w*cos($corner))]
	set dy [expr round($h*sin($corner))]
	set item($ind,sw) [concat [expr $x - $dx] [expr $y - $dy] ]
	set corner [expr atan2(-$h, -$w)]
	set dx [expr round($w*cos($corner))]
	set dy [expr round($h*sin($corner))]
	set item($ind,se) [concat [expr $x - $dx] [expr $y - $dy] ]
    } elseif {$type == "diamond"} {
	set item($ind,ne)  [concat [expr $x + $w / 2] [expr $y - $h / 2]]
	set item($ind,nw)  [concat [expr $x - $w / 2] [expr $y - $h / 2]]
	set item($ind,se)  [concat [expr $x + $w / 2] [expr $y + $h / 2]]
	set item($ind,sw)  [concat [expr $x - $w / 2] [expr $y + $h / 2]]
    }
    set item($ind,w,x) [expr $x - $w]
    set item($ind,e,x) [expr $x + $w]
    set item($ind,n,y) [expr $y - $h]
    set item($ind,s,y) [expr $y + $h]
    set item($ind,i) ANIMITEM$it
    set item($ind,a) 0
    set item($ind,at) "ANIMITEMat$it"
    set item($ind,clear) {}
    set item($ind,color) $colorset

    # draw item
    if {$type == "rect"} {
	$Wca create rectangle \
	    [expr $x-$w] \
	    [expr $y-$h] \
	    [expr $x+$w] \
	    [expr $y+$h] \
	    -fill $color(normal,$colorset,rect) \
	    -outline $color(active,$colorset,rect) \
	    -tags [concat ANIMITEM$it ANIMITEMat$it $tags]
    } elseif {$type == "oval"} {
	$Wca create oval \
	    [expr $x-$w] \
	    [expr $y-$h] \
	    [expr $x+$w] \
	    [expr $y+$h] \
	    -fill $color(normal,$colorset,oval) \
	    -outline $color(active,$colorset,oval) \
	    -tags [concat ANIMITEM$it ANIMITEMat$it $tags]
    } elseif {$type == "diamond"} {
	$Wca create polygon \
	    [expr $x-$w] $y \
	    $x [expr $y+$h] \
	    [expr $x+$w] $y \
	    $x [expr $y-$h] \
	    -fill $color(normal,$colorset,oval) \
	    -outline $color(active,$colorset,oval) \
#	    -outline black \
	    -tags [concat ANIMITEM$it ANIMITEMat$it $tags]
    } else {
	Anim::Error Anim::CreateItem 1 "unknown type '$type' for item '$it'"
    }
    $Wca create text [expr $x + $offsetx] [expr $y + $offsety] \
	-text $text -fill $color(normal,$colorset,text) \
	-justify center -tags [concat ANIMITEM$it ANIMITEMat$it \
	ANIMTEXT$ind $tags] -font $anim(cafont) -anchor $anchor

    # free on a reset
    if {$free} {
	set item($ind,free) 1
    } else {
	set item($ind,free) 0
    }

    # create a list for the item
    if {$list} {
	Anim::MakeList $ind
	set item($ind,list) 1
    } else {
	set item($ind,list) 0
    }

    Anim::UpdateScrollregion
}

proc Anim::DestroyItem {it} {
    variable index
    variable item
    variable Wca

    set ind $index($it)
    $Wca delete ANIMITEMat$it
    if {$item($ind,list)} {
	Anim::DestroyList $ind
    }
    Anim::FreeIndex $ind
    unset index($it)
}

proc Anim::DestroyFreeItems {} {
    variable index
    variable item

    foreach elem [array names index] {
	if {$item($index($elem),free)} {
	    Anim::DestroyItem $elem
	}
    }
}

proc Anim::ChangeTextItem {it text} {
    variable item
    variable color
    variable index
    variable Wca
    variable anim

    set ind $index($it)
    $Wca delete ANIMTEXT$ind
    set coor [$Wca coords ANIMITEM$it]
    set x [expr int([lindex $coor 0] + $item($ind,wid))]
    set y [expr int([lindex $coor 1] + $item($ind,ht))]
    if {$item($ind,a)} {
	/* one would expect an item not to be active at this time, but ... */
	$Wca create text $x $y \
	    -text $text -fill $color(active,$item($ind,color),text) \
	    -justify center -tags [concat ANIMITEM$it ANIMITEMat$it \
	    ANIMTEXT$ind] -font $anim(cafont)
    } else {
	$Wca create text $x $y \
	    -text $text -fill $color(normal,$item($ind,color),text) \
	    -justify center -tags [concat ANIMITEM$it ANIMITEMat$it \
	    ANIMTEXT$ind] -font $anim(cafont)
    }
}

proc Anim::CalcPos {ind corner {ex 0} {ey 0}} {
    variable item
    variable dimlist

    if {[lsearch -exact $dimlist $corner] >= 0} {
	return $item($ind,$corner)
    } else {
	if {$item($ind,type) == "rect"} {
	    switch $corner {
		chop {
		    set ay [expr $ey-$item($ind,y)]
		    set ax [expr $ex-$item($ind,x)]
		    if {$ax == 0} {
			set ta 1
		    } else {
			set ta [expr $ay.0/$ax]
		    }
		    if {$ta < 0} {
			set ta [expr -$ta]
		    }
		    set dy [expr round($item($ind,wid)*$ta)]
		    if {$dy > $item($ind,ht) || \
			$dy < -$item($ind,ht)} {
			set dy $item($ind,ht)
			if {$ax == 0} {
			    set dx 0
			} else {
			    set dx [expr round($item($ind,ht)/$ta)]
			}
		    } else {
			set dx $item($ind,wid)
		    }
		    if {$ay < 0} {
			set dy [expr -$dy]
		    }
		    if {$ax < 0} {
			set dx [expr -$dx]
		    }
		    return [concat [expr $item($ind,x) + $dx] \
			[expr $item($ind,y) + $dy] ]
		}
		default {
		    Anim::Error Anim::CalcPos 1 \
			"unknown corner-type '$corner' for item '$item'"
		}
	    }
	} else { # oval
	    switch $corner {
		chop {
		    set corner [expr atan2($item($ind,y) - \
			$ey,$item($ind,x)-$ex)]
		    set dx [expr round($item($ind,wid)*cos($corner))]
		    set dy [expr round($item($ind,ht)*sin($corner))]
		    return [concat [expr $item($ind,x) - $dx] \
			[expr $item($ind,y) - $dy] ]
		}
		default {
		    Anim::Error Anim::CalcPos 1 \
			"unknown corner-type '$corner' for item '$item'"
		}
	    }
	}
    }
}

proc Anim::Chop {id pos} {
    variable item
    variable dimlist
    variable index

    set ind $index($id)
    set ex [lindex $pos 0]
    set ey [lindex $pos 1]
    if {$item($ind,type) == "rect"} {
	set ay [expr $ey-$item($ind,y)]
	set ax [expr $ex-$item($ind,x)]
	if {$ax == 0} {
	    set ta 1
	} else {
	    set ta [expr $ay.0/$ax]
	}
	if {$ta < 0} {
	    set ta [expr -$ta]
	}
	set dy [expr round($item($ind,wid)*$ta)]
	if {$dy > $item($ind,ht) || \
	    $dy < -$item($ind,ht)} {
	    set dy $item($ind,ht)
	    if {$ax == 0} {
		set dx 0
	    } else {
		set dx [expr round($item($ind,ht)/$ta)]
	    }
	} else {
	    set dx $item($ind,wid)
	}
	if {$ay < 0} {
	    set dy [expr -$dy]
	}
	if {$ax < 0} {
	    set dx [expr -$dx]
	}
	return [concat [expr $item($ind,x) + $dx] \
	    [expr $item($ind,y) + $dy] ]
    } else { # oval
	set corner [expr atan2($item($ind,y) - \
	    $ey,$item($ind,x)-$ex)]
	set dx [expr round($item($ind,wid)*cos($corner))]
	set dy [expr round($item($ind,ht)*sin($corner))]
	return [concat [expr $item($ind,x) - $dx] \
	    [expr $item($ind,y) - $dy] ]
    }
}

proc Anim::Locate {item p {dx 0} {dy 0}} {
    set posxy [Anim::Dim $item $p]
    if {[llength $posxy] == 2} {
    set posxy [concat [expr [lindex $posxy 0] + $dx] [expr [lindex $posxy 1] + $dy]]
    return $posxy
    } else {
        return [expr $posxy + $dx]
    }
}

proc Anim::Pos {type ta} {
    variable item
    variable index

    if {$type == "item"} {
	set i $index([lindex $ta 0])
	set c [lindex $ta 1]
	if {$c == "chop"} {
	    set c  ce
	}
	return [Anim::CalcPos $i $c]
    } else {
	return $ta
    }
}

proc Anim::Dim {id co} {
    variable item
    variable index
    variable dimlist

    set ind $index($id)
    if {[lsearch -exact $dimlist $co] == -1} {
	Anim::Error Anim::Dim 1 \
	    "unkown dimension indicator '$co' for item '$id'"
    }
    if {[array names item $ind,$co] == ""} {
	Anim::Error Anim::Dim 1 "unknown item '$id'"
    }
    return $item($ind,$co)
}

proc Anim::DimL {id co} {
    variable line
    variable dimllist

    if {[lsearch -exact $dimllist $co] == -1} {
	Anim::Error Anim::DimL 1 \
	    "unkown dimension indicator '$co' for line '$id'"
    }
    if {[array names line $id,$co] == ""} {
	Anim::Error Anim::DimL 1 "unknown line '$id'"
    }
    return $line($id,$co)
}

proc Anim::DimQ {id co} {
    variable queue
    variable dimlist

    if {[lsearch -exact $dimlist $co] == -1} {
	Anim::Error Anim::DimQ 1 \
	    "unkown dimension indicator '$co' for queue '$id'"
    }
    if {[array names queue $id,$co] == ""} {
	Anim::Error Anim::DimQ 1 "unknown queue '$id'"
    }
    return $queue($id,$co)
}

proc Anim::DimS {id co} {
    variable queue
    variable dimlist

    if {[lsearch -exact $dimlist $co] == -1} {
	Anim::Error Anim::DimS 1 \
	    "unkown dimension indicator '$co' for stack '$id'"
    }
    if {[array names queue $id,$co] == ""} {
	Anim::Error Anim::DimS 1 "unknown stack '$id'"
    }
    return $queue($id,$co)
}

proc Anim::CreatePLine {li args } {
    variable line
    variable index
    variable color
    variable Wca
    variable anim

    # init while
    set plist $args
    set i 0
    set poslist ""
    # loop until end of plist
    while {[lindex $plist $i] != ""} {
	# or until options
	if {[string match -* [lindex $plist $i]]} {
	    break
	}
	set pos [lindex $plist $i]
	incr i
	if {[llength $pos] != 2} {
	    Anim::Error Anim::CreatePLine 1 \
		"not a position '$pos'"
	}
	set poslist [concat $poslist $pos]
    }
    # default options
    set width 3
    set arrow none
    set colorset 0
    set lower 1
    set free 0
    set smooth 0
    set tags ""
    # handle options
    while {[lindex $plist $i] != ""} {
	switch -exact -- [lindex $plist $i] {
	    -width {
		incr i
		set width [lindex $plist $i]
	    }
	    -arrow {
		incr i
		set arrow [lindex $plist $i]
	    }
	    -color {
		incr i
		set colorset [lindex $plist $i]
	    }
	    -nolower {
		set lower 0
	    }
	    -free {
		set free 1
	    }
	    -smooth {
		set smooth 1
	    }
	    -tag {
		incr i
		set tags "$tags [lindex $plist $i]"
	    }
	    default {
		Anim::Error Anim::CreateLine 1 \
		    "unknown option '[lindex $plist $i]' for line '$li'"
	    }
	}
	incr i
    }
    # register line
    set line($li,i) ANIMLINE$li
    set line($li,a) 0
    set line($li,pos) $poslist
    set line($li,start,x) [lindex $poslist 0]
    set line($li,start,y) [lindex $poslist 1]
    set line($li,start) [concat $line($li,start,x) \
	$line($li,start,y)]
    set i [llength $poslist]
    incr i -1
    set line($li,end,y) [lindex $poslist $i]
    incr i -1
    set line($li,end,x) [lindex $poslist $i]
    set line($li,end) [concat $line($li,end,x) \
	$line($li,end,y)]
    set line($li,color) $colorset
    set line($li,free) $free
    # draw line
    if {$color(screen)} {
	eval $Wca create line $line($li,pos) \
	    -joinstyle miter -fill $color(normal,$colorset,line) \
	    -tags {[concat $line($li,i) $tags]} -width $width -arrow $arrow \
	    -smooth $smooth
    } else {
	eval $Wca create line $line($li,pos) \
	    -joinstyle miter -stipple $color(normal,$colorset,line) \
	    -tags {[concat $line($li,i) $tags]} -width $width -arrow $arrow \
            -fill $color(normal,$colorset,text) -smooth $smooth
    }
    # lower it so that it doesn't obscure the items
    if {$lower} {
	$Wca lower $line($li,i)
    }
    # keep track of the line on top of all other lines (see activate_line)
    if {$anim(topline) == 0} {
	set anim(topline) $line($li,i)
    }

    Anim::UpdateScrollregion
}

proc Anim::CreateLine {li args } {
    variable line
    variable index
    variable color
    variable Wca
    variable anim

    # init while
    set plist $args
    set i 0
    set p2 [lindex $plist $i]
    incr i
    set p2a1 [lindex $plist $i]
    incr i
    set p2a2 [lindex $plist $i]
    incr i
    set poslist ""
    # loop until end of plist
    while {[lindex $plist $i] != ""} {
	# or until options
	if {[string match -* [lindex $plist $i]]} {
	    break
	}
	set p1 $p2; set p1a1 $p2a1; set p1a2 $p2a2
	if {[string match -* $p1a2 ]} {
	    incr i -1
	    break
	}
	set p2 [lindex $plist $i]
	incr i
	set p2a1 [lindex $plist $i]
	incr i
	set p2a2 [lindex $plist $i]
	incr i
	if {$p1 == "item"} {
	    set tpos [Anim::Pos $p2 "$p2a1 $p2a2"]
	    set ind $index($p1a1)
	    set pos [Anim::CalcPos $ind $p1a2 [lindex $tpos 0] \
		    [lindex $tpos 1]]
	    lappend poslist [lindex $pos 0] [lindex $pos 1]
	} elseif {$p1 == "coor"} {
	    lappend poslist [lindex $p1a1 0] [lindex $p1a1 1]
	    incr i -1
	    set p2a2 $p2a1
	    set p2a1 $p2
	    set p2 $p1a2
	} else {
	    lappend poslist $p1a1 $p1a2
	}
    }
    # add the last position
    if {$p2 == "item"} {
	set tpos [Anim::Pos $p1 "$p1a1 $p1a2"]
	    set ind $index($p2a1)
	set pos [Anim::CalcPos $ind $p2a2 [lindex $tpos 0] [lindex $tpos 1]]
	lappend poslist [lindex $pos 0] [lindex $pos 1]
    } elseif {$p2 == "coor"} {
	lappend poslist [lindex $p2a1 0] [lindex $p2a1 1]
    } else {
	lappend poslist $p2a1 $p2a2
    }
    # default options
    set width 3
    set arrow none
    set colorset 0
    set lower 1
    set free 0
    set smooth 0
    set tags ""
    # handle options
    while {[lindex $plist $i] != ""} {
	switch -exact -- [lindex $plist $i] {
	    -width {
		incr i
		set width [lindex $plist $i]
	    }
	    -arrow {
		incr i
		set arrow [lindex $plist $i]
	    }
	    -color {
		incr i
		set colorset [lindex $plist $i]
	    }
	    -nolower {
		set lower 0
	    }
	    -free {
		set free 1
	    }
	    -smooth {
		set smooth 1
	    }
	    -tag {
		incr i
		set tags "$tags [lindex $plist $i]"
	    }
	    default {
		Anim::Error Anim::CreateLine 1 \
		    "unknown option '[lindex $plist $i]' for line '$li'"
	    }
	}
	incr i
    }
    # register line
    set line($li,i) ANIMLINE$li
    set line($li,a) 0
    set line($li,pos) $poslist
    set line($li,start,x) [lindex $poslist 0]
    set line($li,start,y) [lindex $poslist 1]
    set line($li,start) [concat $line($li,start,x) \
	$line($li,start,y)]
    set i [llength $poslist]
    incr i -1
    set line($li,end,y) [lindex $poslist $i]
    incr i -1
    set line($li,end,x) [lindex $poslist $i]
    set line($li,end) [concat $line($li,end,x) \
	$line($li,end,y)]
    set line($li,color) $colorset
    set line($li,free) $free
    # draw line
    if {$color(screen)} {
	eval $Wca create line $line($li,pos) \
	    -joinstyle miter -fill $color(normal,$colorset,line) \
	    -tags {[concat $line($li,i) $tags]} -width $width -arrow $arrow \
	    -smooth $smooth
    } else {
	eval $Wca create line $line($li,pos) \
	    -joinstyle miter -stipple $color(normal,$colorset,line) \
	    -tags {[concat $line($li,i) $tags]} -width $width -arrow $arrow \
            -fill $color(normal,$colorset,text) -smooth $smooth
    }
    # lower it so that it doesn't obscure the items
    if {$lower} {
	$Wca lower $line($li,i)
    }
    # keep track of the line on top of all other lines (see activate_line)
    if {$anim(topline) == 0} {
	set anim(topline) $line($li,i)
    }

    Anim::UpdateScrollregion
}

proc Anim::DestroyLine {li} {
    variable line
    variable Wca

    $Wca delete $line($li,i)
    set line($li,free) 0
}

proc Anim::DestroyFreeLines {} {
    variable line

    foreach elem [array names line *,a] {
	regexp {^(.*),a$} $elem match name
	if {$line($name,free)} {
	    Anim::DestroyLine $name
	}
    }
}

proc Anim::ActivateLine {li} {
    variable line
    variable color
    variable Wca
    variable anim

    if {! $line($li,a)} {
	if {$color(screen)} {
	    $Wca itemconfigure $line($li,i) \
		-fill $color(active,$line($li,color),line)
	} else {
	    $Wca itemconfigure $line($li,i) \
		-stipple $color(active,$line($li,color),line)
	}
	set line($li,a) 1
    }
    # put this line on top of all other lines
    $Wca raise $line($li,i) $anim(topline)
    set anim(topline) $line($li,i)
}

proc Anim::DeactivateLine {li} {
    variable line
    variable color
    variable Wca

    if {$line($li,a)} {
	if {$color(screen)} {
	    $Wca itemconfigure $line($li,i) \
		-fill $color(normal,$line($li,color),line)
	} else {
	    $Wca itemconfigure $line($li,i) \
		-stipple $color(normal,$line($li,color),line)
	}
	set line($li,a) 0
    }
}

proc Anim::ClearLines {} {
    variable line

    foreach elem [array names line *,a] {
	regexp {^(.*),a$} $elem match name
	Anim::DeactivateLine $name
    }
}

proc Anim::AddClear {it args} {
    variable item
    variable index

    set ind $index($it)
    set item($ind,clear) [concat $item($ind,clear) $args]
}

proc Anim::ClearInd {ind} {
    variable item

    foreach el $item($ind,clear) {
	if {[lindex $el 0] == "line"} {
	    Anim::DeactivateLine [lindex $el 1]
	} else { # text
	    Anim::DeleteText [lindex $el 1]
	}
    }
    set item($ind,clear) {}
}

proc Anim::Clear {it} {
    variable index

    Anim::ClearInd $index($it)
}

proc Anim::ClearItems {} {
    variable item

    foreach elem [array names item *,clear] {
	regexp {^(.*),clear$} $elem match itemind
	Anim::ClearInd $itemind
    }
}

proc Anim::AdjustTextpos {id} {
    variable textpos

    switch $textpos($id,anchor) {
	n {
	    incr textpos($id,y) 1
	}
	e {
	    incr textpos($id,x) -3
	}
	s {
	    incr textpos($id,y) -2
	}
	w {
	    incr textpos($id,x) 3
	}
	ne {
	    incr textpos($id,x) -1
	    incr textpos($id,y) 1
	}
	se {
	    incr textpos($id,x) -1
	    incr textpos($id,y) -2
	}
	nw {
	    incr textpos($id,x) 3
	    incr textpos($id,y) 1
	}
	sw {
	    incr textpos($id,x) 3
	    incr textpos($id,y) -2
	}
    }
}

proc Anim::TextposItem {id it corner anchor args} {
    variable textpos
    variable index

    set i 0
    set tags ""
    set offsetx 0
    set offsety 0
    while {[lindex $args $i] != ""} {
	switch -- [lindex $args $i] {
	    -tag {
		incr i
		set tags "$tags [lindex $args $i]"
	    }
	    -offset {
		incr i
		set offsetx [lindex $args $i]
		incr i
		set offsety [lindex $args $i]
	    }
	    default {
		Anim::Error Anim::TextposItem 1 \
		    "unknown option '[lindex $args $i]' for text-position '$id'"
	    }
	}
	incr i
    }
    set ind $index($it)
    set pos [Anim::CalcPos $ind $corner]
    set textpos($id,x) [expr [lindex $pos 0] + $offsetx]
    set textpos($id,y) [expr [lindex $pos 1] + $offsety]
    set textpos($id,anchor) $anchor
    set textpos($id,i) [concat ANIMTEXT$id $tags]
    set textpos($id,a) 0
    set textpos($id,reset) 1
    Anim::AdjustTextpos $id
}

proc Anim::TextposLine {id li anchor args} {
    variable line
    variable textpos

    set d 0.5
    set s 1
    set i 0
    set tags ""
    while {[lindex $args $i] != ""} {
	switch -- [lindex $args $i] {
	    -d {
		incr i
		set d [lindex $args $i]
	    }
	    -s {
		incr i
		set s [lindex $args $i]
	    }
	    -tag {
		incr i
		set tags "$tags [lindex $args $i]"
	    }
	    default {
		Anim::Error Anim::TextposLine 1 \
		    "unknown option '[lindex $args $i]' for text-position '$id'"
	    }
	}
	incr i
    }
    set x1 [lindex $line($li,pos) [expr ($s - 1) * 2]]
    set y1 [lindex $line($li,pos) [expr ($s - 1) * 2 + 1]]
    set x2 [lindex $line($li,pos) [expr $s * 2]]
    set y2 [lindex $line($li,pos) [expr $s * 2 + 1]]
    set dx [expr round(($x2-$x1)*$d)]
    set dy [expr round(($y2-$y1)*$d)]
    set textpos($id,x) [expr $x1+$dx]
    set textpos($id,y) [expr $y1+$dy]
    set textpos($id,anchor) $anchor
    set textpos($id,i) [concat ANIMTEXT$id $args]
    set textpos($id,a) 0
    set textpos($id,reset) 1
    Anim::AdjustTextpos $id
}

proc Anim::Textpos {id x y anchor args} {
    variable textpos

    set reset 1
    set i 0
    set tags ""
    while {[lindex $args $i] != ""} {
	switch -- [lindex $args $i] {
	    -noreset {
		incr i
		set reset 0
	    }
	    -tag {
		incr i
		set tags "$tags [lindex $args $i]"
	    }
	    default {
		Anim::Error Anim::Textpos 1 \
		    "unknown option '[lindex $args $i]' for text-position '$id'"
	    }
	}
	incr i
    }
    set textpos($id,x) $x
    set textpos($id,y) $y
    set textpos($id,anchor) $anchor
    set textpos($id,i) [concat ANIMTEXT$id $tags]
    set textpos($id,a) 0
    set textpos($id,reset) $reset
}

proc Anim::CreateText {id text args} {
    variable textpos
    variable Wca
    variable anim

    Anim::DeleteText $id
    eval $Wca create text $textpos($id,x) $textpos($id,y) \
	-text \"$text\" -anchor $textpos($id,anchor) \
	-tags {$textpos($id,i)} $args -font \"$anim(cafont)\"
    set textpos($id,a) 1
}

proc Anim::DeleteText {id} {
    variable textpos
    variable Wca

    if {$textpos($id,a)} {
	$Wca delete [lindex $textpos($id,i) 0]
	set textpos($id,a) 0
    }
}

proc Anim::ClearTextposs {} {
    variable textpos

    foreach elem [array names textpos *,a] {
	regexp {^(.*),a$} $elem match name
	if {$textpos($name,reset)} {
	    Anim::DeleteText $name
	}
    }
}

proc Anim::ActivateItem {it} {
    variable item
    variable index
    variable color
    variable Wca

    set ind $index($it)
    if {!$item($ind,a)} {
	set item($ind,a) 1
	if {$color(screen)} {
	    $Wca itemconfigure $item($ind,at) \
		-fill $color(active,$item($ind,color),$item($ind,type))
	} else {
	    $Wca itemconfigure $item($ind,at) -stipple \
		$color(active,$item($ind,color),$item($ind,type)) \
		-fill #000000
	}
	$Wca itemconfigure ANIMTEXT$ind \
	    -fill $color(active,$item($ind,color),text)
    }
}

proc Anim::ActivateItemInd {ind} {
    variable item
    variable color
    variable Wca

    if {$item($ind,a)} {
	set item($ind,a) 0
	if {$color(screen)} {
	    $Wca itemconfigure $item($ind,at) \
		-fill $color(normal,$item($ind,color),$item($ind,type))
	} else {
	    $Wca itemconfigure $item($ind,at) -stipple "" \
		-fill $color(normal,$item($ind,color),$item($ind,type))
	}
	$Wca itemconfigure ANIMTEXT$ind \
	    -fill $color(normal,$item($ind,color),text)
    }
}

proc Anim::DeactivateItem {it} {
    variable index

    set ind $index($it)
    Anim::ActivateItemInd $ind
}

proc Anim::CreateQueue {qu x y w h args} {
    variable queue
    variable Wca

    set orient horizontal
    set anchor center
    set input_button 0
    set output_button 0
    set tags ""
    for {set i 0} {[lindex $args $i] != ""} {incr i} {
	switch -- [lindex $args $i] {
	    -orient {
		incr i
		set orient [lindex $args $i]
	    }
	    -anchor {
		incr i
		set anchor [lindex $args $i]
	    }
	    -tag {
		incr i
		set tags "$tags [lindex $args $i]"
	    }
	}
    }
    frame $Wca.queuefr$qu
    text $Wca.queuete$qu -width $w -height $h \
	-yscrollcommand "$Anim::Wca.queuesb$qu set" -highlightthickness 0 -bd 1
    scrollbar $Wca.queuesb$qu -command "$Anim::Wca.queuete$qu yview" \
	-orient $orient -width 8 -highlightthickness 0 -bd 1
    if {$orient == "horizontal"} {
	pack $Wca.queuesb$qu $Wca.queuete$qu \
	    -in $Wca.queuefr$qu -side top -fill x
    } else {
	pack $Wca.queuesb$qu $Wca.queuete$qu \
	    -in $Wca.queuefr$qu -side left -fill y
    }
    $Wca create window $x $y -window $Wca.queuefr$qu \
	-anchor $anchor -tags [concat ANIMQUEUE$qu $tags]

    # register queue
    set queue($qu,x) $x
    set queue($qu,y) $y
    set queue($qu,anchor) $anchor
    set queue($qu,orient) $orient
    set queue($qu,i) ANIMQUEUE$qu
    set queue($qu,window) $Wca.queuefr$qu
#    tkwait visibility $Wca.queuefr$qu
#    set wx [expr int([winfo width $queue($qu,window)]/2)]
#    set hx [expr int([winfo height $queue($qu,window)]/2)]
#    if {$anchor != "center"} {
#	set x [winfo x $Wca.queuefr$qu]
#	incr x $w
#	set y [winfo y $Wca.queuefr$qu]
#	incr y $h
#    }
    set F [$Wca.queuete$qu cget -font]
    set fh [font metrics $F -linespace]
    set fw [font measure $F 0]
    set w [expr $w*$fw + 4]
    set h [expr $h*$fh + 4]
    if {$orient == "horizontal"} {
	incr h 10
    }
    if {$orient == "vertical"} {
	incr w 10
    }
    set w [expr int($w/2)]
    set h [expr int($h/2)]
    if {$anchor != "center"} {
	if {[regexp {n} $anchor match]} {
	    set y [expr $y + $h]
	}
	if {[regexp {s} $anchor match]} {
	    set y [expr $y - $h]
	}
	if {[regexp {w} $anchor match]} {
	    set x [expr $x + $w]
	}
	if {[regexp {e} $anchor match]} {
	    set x [expr $x - $w]
	}
    }
    set queue($qu,x) $x
    set queue($qu,y) $y
    set queue($qu,wid) $w
    set queue($qu,ht) $h
    set queue($qu,ce) [concat $x $y]
    set queue($qu,n)  [concat $x [expr $y - $h]]
    set queue($qu,s)  [concat $x [expr $y + $h]]
    set queue($qu,e)  [concat [expr $x + $w] $y]
    set queue($qu,w)  [concat [expr $x - $w] $y]
    set queue($qu,ne)  [concat [expr $x + $w] [expr $y - $h]]
    set queue($qu,nw)  [concat [expr $x - $w] [expr $y - $h]]
    set queue($qu,se)  [concat [expr $x + $w] [expr $y + $h]]
    set queue($qu,sw)  [concat [expr $x - $w] [expr $y + $h]]
    set queue($qu,w,x) [expr $x - $w]
    set queue($qu,e,x) [expr $x + $w]
    set queue($qu,n,y) [expr $y - $h]
    set queue($qu,s,y) [expr $y + $h]
    $Wca create line [expr $x-$w] [expr $y-$h] [expr $x+$w-1] [expr $y-$h] [expr $x+$w-1] [expr $y+$h-1] [expr $x-$w] [expr $y+$h-1] [expr $x-$w] [expr $y-$h]

    Anim::UpdateScrollregion
}

proc Anim::AddQueue {qu text} {
    variable queue
    variable Wca

    if {$queue($qu,orient) == "horizontal"} {
	$Wca.queuete$qu insert insert "$text "
    } else {
	$Wca.queuete$qu insert insert "$text\n"
    }
    $Wca.queuete$qu yview moveto 0
}
    
proc Anim::SubQueue {qu} {
    variable queue
    variable Wca

    if {$queue($qu,orient) == "horizontal"} {
	set index [$Wca.queuete$qu search " " 0.0]
    } else {
	set index [$Wca.queuete$qu search "\n" 0.0]
    }
    $Wca.queuete$qu delete 0.0 $index+1chars
    $Wca.queuete$qu yview moveto 0
}

proc Anim::DeleteQueue {qu text} {
    variable queue
    variable Wca

    if {$queue($qu,orient) == "horizontal"} {
	set index [$Wca.queuete$qu search -regexp "(^| )$text " 0.0]
    } else {
	set index [$Wca.queuete$qu search -regexp "^$text$" 0.0]
    }
    if {$index != ""} {
	$Wca.queuete$qu delete $index $index+[expr [string length $text] + 1]chars
	$Wca.queuete$qu yview moveto 0
    }
}

proc Anim::ClearQueues {} {
    variable queue
    variable Wca

    foreach elem [array names queue *,i] {
	regexp {^(.*),i$} $elem match name
	$Wca.queuete$name delete 0.0 end
    }
}

proc Anim::CreateStack {stack x y w h args} {
    # implemented as a queue
    eval Anim::CreateQueue $stack $x $y $w $h $args
}

proc Anim::PushStack {stack text} {
    variable queue
    variable Wca

    if {$queue($stack,orient) == "horizontal"} {
	$Wca.queuete$stack insert 0.0 "$text "
    } else {
	$Wca.queuete$stack insert 0.0 "$text\n"
    }
    $Wca.queuete$stack yview moveto 0
}

proc Anim::PopStack {stack} {
    Anim::SubQueue $stack
}

proc Anim::DeleteStack {stack text} {
    Anim::DeleteQueue $stack $text
}

#info

proc Anim::CreateBox {pbox name args} {
    variable box

    Anim::BoxInit
    set side top
    set fill none
    set relief flat
    set borderwidth 0
    set ipadx 0
    set ipady 0
    set expand 0
    for {set i 0} {[lindex $args $i] != ""} {incr i} {
	switch -- [lindex $args $i] {
	    -side {
		incr i
		set side [lindex $args $i]
	    }
	    -fill {
		incr i
		set fill [lindex $args $i]
	    }
	    -relief {
		incr i
		set relief [lindex $args $i]
	    }
	    -bw {
		incr i
		set borderwidth [lindex $args $i]
	    }
	    -ipadx {
		incr i
		set ipadx [lindex $args $i]
	    }
	    -ipady {
		incr i
		set ipady [lindex $args $i]
	    }
	    -expand {
		set expand 1
	    }
	}
    }
    set box($name,w) $box($pbox,w).w$name
    set box($name,side) $side
    set box($name,fill) $fill
    set box($name,expand) $expand
    frame $box($name,w) -borderwidth $borderwidth -relief $relief
    pack $box($name,w) -in $box($pbox,w) \
	-side $box($pbox,side) -fill $box($pbox,fill) \
	-ipadx $ipadx -ipady $ipady -expand $box($pbox,expand)
}

proc Anim::CreateLabel {bo name text args} {
    variable box

    Anim::BoxInit
    set var 0
    set borderwidth 2
    set relief flat
    set width 0
    set anchor center
    set padx 1
    for {set i 0} {[lindex $args $i] != ""} {incr i} {
	switch -- [lindex $args $i] {
	    -var {
		    set var 1
	    }
	    -relief {
		incr i
		set relief [lindex $args $i]
	    }
	    -bw {
		incr i
		set borderwidth [lindex $args $i]
	    }
	    -width {
		incr i
		set width [lindex $args $i]
	    }
	    -anchor {
		incr i
		set anchor [lindex $args $i]
	    }
	    -padx {
		incr i
		set padx [lindex $args $i]
	    }
	}
    }
    if {$var} {
	label $box($bo,w).w$name -textvariable $text \
	    -borderwidth $borderwidth -relief $relief -width $width \
	    -anchor $anchor -padx $padx
    } else {
	label $box($bo,w).w$name -text $text \
	    -borderwidth $borderwidth -relief $relief -width $width \
	    -anchor $anchor -padx $padx
    }
    pack $box($bo,w).w$name -in $box($bo,w) \
	-side $box($bo,side) -fill $box($bo,fill)
}

proc Anim::InitVar {name value} {
    variable var

    set var($name,init) $value
    set var($name,type) "var"
    variable $name
    set ::$name $value
}

proc Anim::InitArray {name list} {
    variable var

    if {[info exist var($name,init)]} {
        set var($name,init) [concat $var($name,init) $list]
    } else {
        set var($name,init) $list
    }
    set var($name,type) "array"
    variable $name
    array set ::$name $list
}

proc Anim::ClearVars {} {
    variable var

    foreach elem [array names var *,init] {
	regexp {^(.*),init$} $elem match name
	variable $name
	if {$var($name,type) == "var"} {
	    set ::$name $var($name,init)
	} else { # array
	    array set ::$name $var($name,init)
	}
    }
}

# list routines

proc Anim::MakeList {ind} {
    variable list
    variable item
    variable color
    variable Wca

    set parentwindow $Wca
    set list($ind,p) $parentwindow
    set list($ind,window) $parentwindow.animlist$ind
    set list($ind,t) ANIMLIST$ind
    set list($ind,i) $item($ind,i)
    set list($ind,a) 0
    if {$color(screen)} {
	listbox $list($ind,window) -selectmode browse -width 0 -height 0 \
	    -relief raised -bg $color(listbox) -selectbackground \
	    $color(listboxselect) -highlightthickness 0 -bd 1
    } else {
	listbox $list($ind,window) -selectmode browse -width 0 -height 0 \
	    -relief raised -highlightthickness 0 -bd 1
    }
    bind $list($ind,window) \
	<ButtonRelease-1> "$list($ind,p) delete $list($ind,t); grab release $list($ind,window)"
    bind $list($ind,window) \
	<ButtonRelease-1> {+ Anim::ChoiceMade %W [%W curselection] }
    bind $list($ind,window) \
	<Leave> "$list($ind,p) delete $list($ind,t); grab release $list($ind,window)"
}

proc Anim::PopupList {W x y li} {
    variable list
    variable index
    variable queue

    set ind $index($li)
    set wh [winfo height $W] 
    set ww [winfo width $W] 
    set x [expr $x - 8]
    set y [expr $y - 8]
    # real x,y on canvas
    set rx [$W canvasx $x 1]
    set ry [$W canvasy $y 1]
    $W create window $rx $ry \
	-window $list($ind,window) -anchor nw -tags $list($ind,t)
    # wait for appearance
    tkwait visibility $list($ind,window)
    # try to keep it on screen totally by moving it
    set lh [winfo height $list($ind,window)] 
    set lw [winfo width $list($ind,window)] 
    if {$x < 0} {
	set newx [expr - $x]
    } else {
	if {$x + $lw > $ww} { 
	    set newx [expr $ww - ($lw + $x)] 
	} else {
	    set newx 0
	}
    }
    if {$y < 0} {
	set newy [expr - $y]
    } else {
	if {$y + $lh > $wh} { 
	    set newy [expr $wh - ($lh + $y)] 
	} else {
	    set newy 0
	}
    }
    $W move $list($ind,t) $newx $newy
    raise $list($ind,window)
grab $list($ind,window)
$list($ind,window) selection clear 0 end
$list($ind,window) selection set 0
}

proc Anim::ActivateList {li} {
    variable list
    variable index

    set ind $index($li)
    if {! $list($ind,a)} {
	set list($ind,a) 1
	# x and y should be variable of Anim !!!!!
#	$list($ind,p) bind $list($ind,i) \
#	    "<ButtonPress-1>" {set x %x; set y %y}
#	$list($ind,p) bind $list($ind,i) \
#	    "<ButtonRelease-1>" [subst -nocommands {if {\$x==%x&&\$y==%y} \
#	    {Anim::PopupList %W %x %y $li} }]
	$list($ind,p) bind $list($ind,i) \
	    "<ButtonPress-1>" [subst -nocommands \
	    {Anim::PopupList %W %x %y $li} ]
    }
}

proc Anim::DeactivateList {ind} {
    variable list

    if {$list($ind,a)} {
	set list($ind,a) 0
	$list($ind,p) bind $list($ind,i) <ButtonPress-1> ""
    }
}

proc Anim::DestroyList {ind} {
    variable list

    bind $list($ind,window) <ButtonRelease-1> ""
    bind $list($ind,window) <Leave> ""
    destroy $list($ind,window)
    unset list($ind,a)
}

proc Anim::AddList {li item} {
    variable list
    variable index
    variable registerchoice

    set ind $index($li)
    $list($ind,window) insert end $item
    if {! $list($ind,a)} {
	Anim::ActivateList $li
	Anim::ActivateItem $li
    }
    if {$registerchoice} {
	Anim::RegisterSet $list($ind,window) [expr [$list($ind,window) size] - 1]
    }
}

proc Anim::EmptyList {ind} {
    variable list

    if {$list($ind,a)} {
	Anim::DeactivateList $ind
    }
    $list($ind,window) delete 0 [expr [$list($ind,window) size] - 1]
}

proc Anim::ClearLists {} {
    variable list
    variable item

    foreach elem [array names list *,a] {
	regexp {^(.*),a$} $elem match itemind
	Anim::ActivateItemInd $itemind
	if {$item($itemind,list)} {
	    Anim::EmptyList $itemind
	}
    }
}

proc Anim::ChoiceMade {w wi} {
    variable callback
    variable registerchoice

    set str [$w get $wi]
    Anim::ClearLists
    if {$registerchoice} {
	set i [Anim::RegisterGet $w $wi]
	$callback $i $str
    } else {
	$callback $str
    }
}

# move routines

proc Anim::MoveDelay {delay} {
    variable move

    set move(delay) $delay
}

proc Anim::MoveFast {b} {
    variable move

    set move(fast) $b
}

proc Anim::MoveLeft {id pos} {
    variable move
    variable Wca

    if {$move(fast)} {
	$Wca move $id [expr -$pos] 0
    } else {
	for {set i 0} {$i < $pos} {incr i} {
	    after $move(delay)
	    $Wca move $id -1 0
	    update
	}
    }
}
    
proc Anim::MoveRight {id pos} {
    variable move
    variable Wca

    if {$move(fast)} {
	$Wca move $id $pos 0
    } else {
	for {set i 0} {$i < $pos} {incr i} {
	    after $move(delay)
	    $Wca move $id 1 0
	    update
	}
    }
}
    
proc Anim::MoveUp {id pos} {
    variable move
    variable Wca

    if {$move(fast)} {
	$Wca move $id 0 [expr -$pos]
    } else {
	for {set i 0} {$i < $pos} {incr i} {
	    after $move(delay)
	    $Wca move $id 0 -1
	    update
	}
    }
}
    
proc Anim::MoveDown {id pos} {
    variable move
    variable Wca

    if {$move(fast)} {
	$Wca move $id 0 $pos
    } else {
	for {set i 0} {$i < $pos} {incr i} {
	    after $move(delay)
	    $Wca move $id 0 1
	    update
	}
    }
}
    
proc Anim::ChangeTag {old new} {
    variable Wca

    $Wca addtag ANIMITEM$new withtag ANIMITEM$old
    $Wca dtag ANIMITEM$new ANIMITEM$old
    $Wca addtag ANIMITEMat$new withtag ANIMITEMat$old
    $Wca dtag ANIMITEMat$new ANIMITEMat$old
}

proc Anim::CombineTag {item type obj} {
    variable Wca

    if {$type == "line"} {
        $Wca addtag ANIMITEM$item withtag ANIMLINE$obj
    } elseif {$type == "item"} {
        $Wca addtag ANIMITEM$item withtag ANIMLINE$obj
    } else {
        Anim::Error Anim::CombineTag 1 "unkown type '$type'"
    }
}

proc Anim::Move {it args} {
    variable item
    variable index
    variable list
    variable Wca

    set ind $index($it)
    set coor [$Wca coords ANIMITEM$it]
    if {$coor == ""} {
	Anim::Error Anim::Move 1 "no item with id '$it' found"
    }
    for {set i 0; set com [lindex $args $i]} {$com != "" && \
	[string match -* $com] == 0} {incr i; set com [lindex $args $i]} {
	incr i
	set arg [lindex $args $i]
	switch -exact $com {
	    left {
		Anim::MoveLeft ANIMITEM$it $arg
	    }
	    right {
		Anim::MoveRight ANIMITEM$it $arg
	    }
	    up {
		Anim::MoveUp ANIMITEM$it $arg
	    }
	    down {
		Anim::MoveDown ANIMITEM$it $arg
	    }
	    leftto {
		set coor [$Wca coords ANIMITEM$it]
		set a [lindex $coor 0]
		set a [expr int($a + ([lindex $coor 2] - $a)/2)]
		Anim::MoveLeft ANIMITEM$it [expr $a - $arg]
	    }
	    rightto {
		set coor [$Wca coords ANIMITEM$it]
		set a [lindex $coor 0]
		set a [expr int($a + ([lindex $coor 2] - $a)/2)]
		Anim::MoveRight ANIMITEM$it [expr $arg - $a]
	    }
	    upto {
		set coor [$Wca coords ANIMITEM$it]
		set a [lindex $coor 1]
		set a [expr int($a + ([lindex $coor 3] - $a)/2)]
		Anim::MoveUp ANIMITEM$it [expr $a - $arg]
	    }
	    downto {
		set coor [$Wca coords ANIMITEM$it]
		set a [lindex $coor 1]
		set a [expr int($a + ([lindex $coor 3] - $a)/2)]
		Anim::MoveDown ANIMITEM$it [expr $arg - $a]
	    }
	    default {
		Anim::Error Anim::Move 1 "unknown direction '$com' for move"
	    }
	}
    }
    for {} {$com != ""} {incr i; set com [lindex $args $i]} {
	switch -exact -- $com {
	    -newid {
		incr i
		set arg [lindex $args $i]
		Anim::ChangeTag $it $arg
		set index($arg) $index($it)
		unset index($it)
		set list($ind,i) ANIMITEM$arg
		set item($ind,i) ANIMITEM$arg
		set item($ind,at) ANIMITEMat$arg
	    }
	    default {
		Anim::Error Anim::Move 1 "unknown option '$com' for move"
	    }
	}
    }
}

proc Anim::Reorigin {id x y} {
    variable Wca
    variable textpos

    $Wca move $id $x $y
    foreach elem [array names textpos *,i] {
	if {[lsearch -exact $textpos($elem) $id] >= 0} {
	    regexp {^(.*),i$} $elem match name
	    incr textpos($name,x) $x
	    incr textpos($name,y) $y
	}
    }
    Anim::UpdateScrollregion
}

proc Anim::ReoriginText {x y} {
    variable Wca
    variable textpos

    foreach elem [array names textpos *,i] {
	regexp {^(.*),i$} $elem match name
	incr textpos($name,x) $x
	incr textpos($name,y) $y
    }
    Anim::UpdateScrollregion
}

proc Anim::ReoriginItem {x y} {
    variable Wca
    variable item

    foreach elem [array names item *,i] {
	regexp {^(.*),i$} $elem match name
	incr item($name,x) $x
	incr item($name,y) $y
    }
    Anim::UpdateScrollregion
}

proc Anim::Reset {} {
    Anim::ClearLists
    Anim::ClearItems
    Anim::DestroyFreeItems
    Anim::DestroyFreeLines
    Anim::ClearTextposs
    Anim::ClearLines
    Anim::ClearQueues
    Anim::ClearVars
    Anim::TextDisplay "-- reset --\n"
}

proc Anim::RegisterIndex {i} {
    variable registerindex

    set registerindex $i
}

proc Anim::RegisterSet {w wi} {
    variable register
    variable registerindex

    set register($w,$wi) $registerindex
}

proc Anim::RegisterGet {w wi} {
    variable register
    variable registerindex

    return $register($w,$wi)
}

#namespace eval Anim {
#    set funcs [info procs *]
#    set sfuncs [lsort $funcs]
#    foreach func $sfuncs {
#	set args [info args $func]
#	puts -nonewline "$func"
#	foreach arg $args {
#	    puts -nonewline " $arg"
#	    if {[info default $func $arg def]} {
#		puts -nonewline " {$def}"
#	    }
#	}
#	puts ""
#    }
#}

#source $env(ANIM_SOURCE)
