#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <unistd.h>
#include <errno.h>
#include "psf_prototype.h"
#include "statecontrol.h"
#include "functionwid.h"
#include "process.h"
#include "widgets.h"
#include "choosewid.h"
#include "history.h"

int Option_ToolBus = 0;

static FILE *TB_in;
static FILE *TB_out;

static int reset_same_process = 0;

/*
 * Read a line from the input, and execute events while waiting.
 */
static char line[10240];

static void tb_readline()
{
    int n;
    char *p;
    int fd;

    p = line;
    fd = fileno(TB_in);
    while (1) {
	while (read(fd, p, 1) == -1) {
	    if (errno != EAGAIN) {
		fprintf(stderr, "error\n");
		exit(1);
	    }
	    execute_events();
	}
	if (*p == '\n') {
	    *p = '\0';
	    return;
	}
	p ++;
    }
}

static int control; /* 1 => sim in control, 0 => anim in control */
static int keep_control;

/*
 * Initialization of the ToolBus interface.
 */
void tb_init(in, out)
char *in;
char *out;
{
    char tool[16];

    /* open the pipes */
    TB_in = fdopen(open(in, O_RDONLY | O_NONBLOCK), "r");
    TB_out = fdopen(open(out, O_WRONLY), "w");

    Option_ToolBus = 1;

    /* get info from anim about who gets the control */
    tb_readline();
    sscanf(line, " control(%[^,], %d)", tool, &keep_control);

    if (!strcmp("anim", tool)) {
	control = 0;
	init_active_tbcontrol(0);
    } else if (!strcmp("sim", tool)) {
	control = 1;
	init_active_tbcontrol(1);
    } else if (!strcmp("gen", tool)) {
	control = 0;
	init_active_tbcontrol(0);
	reset_same_process = 1;
    }
}

/*
 * Send string with format f and argument s.
 */
void tb_send(f, s)
char *f;
char *s;
{
    fprintf(TB_out, f, s);
    fflush(TB_out);
}

/*
 * Receive the acknowledgement (indicating that anim is ready).
 */
void tb_receive_ack()
{
    tb_readline();
    if (strcmp(line, "ack")) {
	fprintf(stderr, "error reading ack: %s\n", line);
    }
}

/*
 * Give info about who is in control.
 */
int tb_control()
{
    return(control);
}

/*
 * Give info about keeping control.
 */
int tb_keep_control()
{
    return(keep_control);
}

/*
 * Give the control away.
 */
void tb_give_control()
{
    tb_send("control(%s)\n", "anim");
    control = 0;
    if (get_state() == STATE_CHOOSE)
	force_state(STATE_CHOOSE_INIT);
}

/*
 * Send a reset and wait for acknowledgement.
 */
void tb_reset()
{
    if (control) {
	tb_send("reset\n", NULL);
	tb_receive_ack();
    }
}

/*
 * Send a quit.
 */
void tb_quit()
{
    tb_send("quit\n", NULL);
}

/*
 * Send an atom and wait for acknowledgement.
 */
void tb_atom(s)
char * s;
{
    if (control) {
	tb_send("atom(%s)\n", s);
	tb_receive_ack();
    }
}

/*
 * Send choose-list to ToolBus and wait for reply.
 * The reply can be:
 *     - choice from the choose-list
 *     - reset
 *     - quit
 *     - take control
 * Note: input must be buffered.
 */
void tb_choice(chooselist)
char *chooselist;
{
    unsigned int markid;

    /* send the choose-list */
/*
    if (tb_gen_state()) {
	(void) strcat(chooselist, "#");
	(void) strcat(chooselist, not_list());
    }
*/
    tb_send("choose(%s)\n", chooselist);

    tb_readline();

    /* find out the command and act on it */
    if (!strcmp("<reset>", line)) {
	do_reset();
	if (reset_same_process)
	    force_chosen_item(start_process);
    } else if (!strcmp("<quit>", line)) {
	tb_quit();
	end_widgets();
    } else if (!strcmp("<take_control>", line)) {
	control = 1;
	activate_tbcontrol();
	force_state(STATE_CHOOSE_INIT);
    } else if (!strncmp("<gotomark", line, 9)) {
	(void) sscanf(line, "<gotomark(%u)>", &markid);
	for (; markid; markid --)
	    history_backward();
	force_state(STATE_CHOOSE_INIT);
    } else {
	make_choice(line);
	force_state(STATE_SIMULATE);
    }
}

/*
 * Send a end and wait for acknowledgement.
 */
void tb_end()
{
    if (!control) {
	tb_send("end\n", NULL);
	tb_receive_ack();

	tb_readline();

	/* find out the command and act on it */
	if (!strcmp("<reset>", line)) {
	    do_reset();
	} else if (!strcmp("<quit>", line)) {
	    exit(0);
	} else {
	    exit(1);
	}
    }
}

/*
 * Send a deadlock and wait for acknowledgement.
 */
void tb_deadlock()
{
    tb_end();
}

int tb_gen_state()
{
    return(reset_same_process);
}
