#include <stdio.h>
#include <string.h>
#include "psf_prototype.h"
#include "tiltype.h"
#include "prtilparts.h"
#include "fieldex.h"
#include "tilutil.h"
#include "msprint.h"
#include "simutil.h"
#include "psf_malloc.h"


static char **module_name_table;
static int nr_modules;

void add_module_name(index, name, mod)
    unsigned int index;
    char *name;
    struct module *mod;
{
    if (index > nr_modules) {
	nr_modules = index;
	module_name_table = PSF_REALLOC(module_name_table,char *, 
					nr_modules + 1) ;
    }
    module_name_table[index] = psf_strdup(name);
}

static void make_module_name_table()
{
    struct module *mod;
    unsigned int nr_adm;
    adm_tuple *adm_tuples;
    unsigned int i;
    char *cp;
    unsigned int index;
    char name[128];

    mod = get_til_module();

    module_name_table = PSF_MALLOC(char *);
    nr_modules = 0;

    nr_adm = mod->entries_table[ADM];
    adm_tuples = mod->adm;

    for (i = 1; i <= nr_adm; i++) {
	cp = field_extract("m", adm_tuples[i].ff, ADM, i);
	if (sscanf(cp, "%u %s ", &index, name) != 2)
	    continue;
	add_module_name(index, name, mod);
    }
}

void print_module_name_table()
{
    int i;

    for (i = 1; i < nr_modules; i++)
	printf("#%s#\n", module_name_table[i]);
}

struct simulate_information {
    int breakpoint;
    int trace;
    unsigned int nr_visits;
    char *name;
    unsigned int mod_nr;
};
typedef struct simulate_information simulate_info;

#define	SIM_ATM		0
#define	SIM_FUN		1
#define	SIM_PRO		2
#define	SIM_VAR		3

#define	MAX_SIM_TABLE	3

static simulate_info *sim_info[MAX_SIM_TABLE];
static unsigned int sim_info_ent[MAX_SIM_TABLE];

void add_sim_info(i, j, mod)
    unsigned int i;
    unsigned int j;
    struct module *mod;
{
    char *field;

    if (j > sim_info_ent[i]) {
	sim_info_ent[i]++; 
	sim_info[i] = PSF_NMALLOC(simulate_info ,sim_info_ent[i] + 1);
    }
    sim_info[i][j].breakpoint = 0;
    sim_info[i][j].trace = 0;
    sim_info[i][j].nr_visits = 0;
    switch (i) {
    case SIM_ATM:
	field = field_extract("n",
			      mod->atm[j].ff, ATM, j);
	msprintf("%s", field);
	msprint_ind_list(mod->atm[j].sor_indlist.indlist,
			 mod->atm[j].sor_indlist.a);
	break;
    case SIM_FUN:
	field = field_extract("n",
			      mod->fun[j].ff, FUN, j);
	msprintf("%s", field);
	msprint_ind_list(mod->fun[j].sor_indlist.indlist,
			 mod->fun[j].sor_indlist.a);
	break;
    case SIM_PRO:
	field = field_extract("n",
			      mod->pro[j].ff, PRO, j);
	msprintf("%s", field);
	msprint_ind_list(mod->pro[j].sor_indlist.indlist,
			 mod->pro[j].sor_indlist.a);
	break;
    case SIM_VAR:
	field = field_extract("n",
			      mod->var[j].ff, VAR, j);
	msprintf("%s", field);
	break;
    }
    sim_info[i][j].name = get_ms();
    switch (i) {
    case SIM_ATM:
	field = field_extract("o",
			      mod->atm[j].ff, ATM, j);
	break;
    case SIM_FUN:
	field = field_extract("o",
			      mod->fun[j].ff, FUN, j);
	break;
    case SIM_PRO:
	field = field_extract("o",
			      mod->pro[j].ff, PRO, j);
	break;
    case SIM_VAR:
	field = field_extract("o",
			      mod->var[j].ff, VAR, j);
	break;
    }
    if (sscanf(field, "%u", &sim_info[i][j].mod_nr) != 1)
	sim_info[i][j].mod_nr = 0;
}

static void make_sim_info_tables()
{
    unsigned int i, j;
    unsigned int til_tnr;
    struct module *mod;

    mod = get_til_module();
    for (i = 0; i < MAX_SIM_TABLE; i++) {
	/* we don't really want info on functions and variables, so */
	if (i == SIM_FUN || i == SIM_VAR)
	    continue;
	switch (i) {
	case SIM_ATM:
	    til_tnr = ATM;
	    break;
	case SIM_FUN:
	    til_tnr = FUN;
	    break;
	case SIM_PRO:
	    til_tnr = PRO;
	    break;
	case SIM_VAR:
	    til_tnr = VAR;
	    break;
	};
	sim_info_ent[i] = mod->entries_table[til_tnr];
	sim_info[i] = PSF_NMALLOC(simulate_info , sim_info_ent[i] + 1) ;
	for (j = 1; j <= sim_info_ent[i]; j++) {
	    add_sim_info(i, j, mod);
	}
    }
}

void print_sim_info_tables()
{
    int i, j;

    printf("------------------------------------------------\n");
    for (i = 0; i < MAX_SIM_TABLE; i++)
	for (j = 1; j <= sim_info_ent[i]; j++)
	    printf("%u - %-15s   %d   %d\n", sim_info[i][j].mod_nr,
		   sim_info[i][j].name, sim_info[i][j].breakpoint,
		   sim_info[i][j].trace);
}

static char ***module_item_name;
static int ***module_item_breakpoint;
static int ***module_item_trace;
static int *module_nr_items;
static int *mind;
static int nr_m;

void add_module_item(i, j)
    unsigned int i;
    unsigned int j;
{
    int mnr;

    mnr = sim_info[i][j].mod_nr;
    if (mnr > nr_m) {
	nr_m++;
	module_nr_items = PSF_NMALLOC(int ,nr_m + 1);
	module_nr_items[mnr] = 1;
	mind = PSF_NMALLOC(int , nr_m + 1) ;
	mind[mnr] = 0;
	module_item_name = PSF_REALLOC(module_item_name,char**, nr_m + 1);
	module_item_breakpoint = PSF_REALLOC(module_item_breakpoint,int**, 
						nr_m + 1);
	module_item_trace = PSF_REALLOC(module_item_trace,int**, nr_m + 1);
	module_item_breakpoint[mnr] = PSF_NMALLOC(int *, module_nr_items[mnr]);
	module_item_trace[mnr] =  PSF_NMALLOC(int *, module_nr_items[mnr]);
	module_item_name[mnr] =  PSF_NMALLOC(char *, module_nr_items[mnr]);
    }
    if (mind[mnr] > module_nr_items[mnr]) {
	module_nr_items[mnr]++;
	module_item_breakpoint[mnr] =  PSF_NMALLOC(int *, module_nr_items[mnr]);
	module_item_trace[mnr] =  PSF_NMALLOC(int *, module_nr_items[mnr]);
	module_item_name[mnr] =  PSF_NMALLOC(char *, module_nr_items[mnr]);
    }
    module_item_breakpoint[mnr][mind[mnr]] =
	&sim_info[i][j].breakpoint;
    module_item_trace[mnr][mind[mnr]] =
	&sim_info[i][j].trace;
    module_item_name[mnr][mind[mnr]] = sim_info[i][j].name;
    mind[mnr]++;
}

static void make_module_item_lists()
{
    unsigned int i, j;

    module_nr_items = PSF_NMALLOC(int ,nr_modules + 1) ;
    mind =  PSF_NMALLOC(int ,nr_modules + 1) ;
    nr_m = nr_modules;
    for (i = 0; i <= nr_modules; i++) {
	module_nr_items[i] = 0;
	mind[i] = 0;
    }
    module_item_name = PSF_NMALLOC(char **, nr_modules + 1);
    module_item_breakpoint = PSF_NMALLOC(int **, nr_modules + 1);
    module_item_trace =  PSF_NMALLOC(int **, nr_modules + 1);

    for (i = 0; i < MAX_SIM_TABLE; i++) {
	for (j = 1; j <= sim_info_ent[i]; j++)
	    module_nr_items[sim_info[i][j].mod_nr]++;
    }

    for (i = 1; i <= nr_modules; i++) {
	module_item_breakpoint[i] = PSF_NMALLOC(int *, module_nr_items[i] );
	module_item_trace[i] = PSF_NMALLOC(int *, module_nr_items[i] );
	module_item_name[i] = PSF_NMALLOC(char *, module_nr_items[i] );
    }

    for (i = 0; i < MAX_SIM_TABLE; i++) {
	for (j = 1; j <= sim_info_ent[i]; j++) {
	    add_module_item(i, j);
	}
    }
}

void print_module_item_lists()
{
    int i, j;

    for (i = 1; i <= nr_modules; i++) {
	printf("%s\n", module_name_table[i]);
	for (j = 0; j < module_nr_items[i]; j++) {
	    printf("\t%d %d  %s\n",
		   *module_item_breakpoint[i][j],
		   *module_item_trace[i][j],
		   module_item_name[i][j]);
	}
    }
}

static char **definition_name;
static int nr_definitions;

static void make_definition_names()
{
    struct module *mod;
    int nr_def;
    int i;

    mod = get_til_module();
    nr_def = mod->entries_table[DEF];
    nr_definitions = nr_def;
    definition_name = PSF_NMALLOC(char *, nr_def + 1);
    for (i = 1; i <= nr_def; i++) {
	msprint_ae_term(&mod->def[i].ae_t);
	definition_name[i] = get_ms();
    }
}

char *get_definition_name(i)
    int i;
{
    return (definition_name[i]);
}

int get_nr_definitions()
{
    return (nr_definitions);
}

void init_simutil()
{
    if (get_til_module() == NULL)
	return;
    make_module_name_table();
    make_sim_info_tables();
    make_module_item_lists();
    make_definition_names();
}

void get_modules_list(ml, nr)
    char ***ml;
    unsigned int *nr;
{
    *ml = module_name_table;
    (*ml)++;
    *nr = nr_modules;
    if (nr_modules == 0)
	*ml = NULL;
}

void get_name_list(mnr, nl, nr)
    unsigned int mnr;
    char ***nl;
    unsigned int *nr;
{
    *nl = module_item_name[++mnr];
    *nr = module_nr_items[mnr];
}

void get_breakpoint_list(mnr, bl, nr)
    unsigned int mnr;
    int ***bl;
    unsigned int *nr;
{
    *bl = module_item_breakpoint[++mnr];
    *nr = module_nr_items[mnr];
}

void get_trace_list(mnr, tl, nr)
    unsigned int mnr;
    int ***tl;
    unsigned int *nr;
{
    *tl = module_item_trace[++mnr];
    *nr = module_nr_items[mnr];
}

char *get_item_name(ind)
    struct indextype *ind;
{
    int index;

    switch (ind->table) {
    case ATM:
	index = SIM_ATM;
	break;
    case FUN:
	index = SIM_FUN;
	break;
    case PRO:
	index = SIM_PRO;
	break;
    case VAR:
	index = SIM_VAR;
	break;
    }
    return (sim_info[index][ind->key].name);
}

int get_nr_process_var()
{
    return (sim_info_ent[SIM_PRO]);
}

char *get_process_var_name(index)
    int index;
{
    return (sim_info[SIM_PRO][index].name);
}

int break_on(ind)
    struct indextype *ind;
{
    int index;

    switch (ind->table) {
    case ATM:
	index = SIM_ATM;
	break;
    case FUN:
	index = SIM_FUN;
	break;
    case PRO:
	index = SIM_PRO;
	break;
    case VAR:
	index = SIM_VAR;
	break;
    default:
	return (0);
    }
    return (sim_info[index][ind->key].breakpoint);
}

int trace_on(ind)
    struct indextype *ind;
{
    int index;

    switch (ind->table) {
    case ATM:
	index = SIM_ATM;
	break;
    case FUN:
	index = SIM_FUN;
	break;
    case PRO:
	index = SIM_PRO;
	break;
    case VAR:
	index = SIM_VAR;
	break;
    default:
	return (0);
    }
    return (sim_info[index][ind->key].trace);
}

void free_simutil()
{
    int i, j;

    for (i = 1; i <= nr_modules; i++)
	free(module_name_table[i]);
    free(module_name_table);

    for (i = 0; i < MAX_SIM_TABLE; i++) {
	for (j = 1; j <= sim_info_ent[i]; j++)
	    free(sim_info[i][j].name);
	free(sim_info[i]);
    }

    for (i = 1; i <= nr_modules; i++) {
	free(module_item_breakpoint[i]);
	free(module_item_trace[i]);
	free(module_item_name[i]);
    }
    free(module_item_breakpoint);
    free(module_item_trace);
    free(module_item_name);
    free(module_nr_items);
    free(mind);
    for (i = 1; i <= nr_definitions; i++)
	free(definition_name[i]);
    free(definition_name);
}
