/*
 *  Pretty Print routines
 *  These routines form a base for pretty-printing, which is reduced
 *  to calling the right routine at the right time without even
 *  knowing the settings of the options.
 */

#include <stdio.h>
#include <string.h>
#include "psf_prototype.h"
#include "psf_fopen.h"
#include "psf_standards.h"
#include "psf_malloc.h"
#include "env_variables.h"
#include "getextopt.h"
#include "pretty.h"

extern FILE *yyin;		/* input-file of lex */
extern int yylineno;		/* linenumber from lex */
char *filename = "stdin";	/* name of the current file */
int startlineno = 1;		/* linenumber of the first line in a file */
int errorseen = 0;		/* flags grammatical errors *//* HM001 */

/* Options */
int Length;
int Indent;
int Place_comment;
int Blank_section;
int Blank_comma;
int Blank_line_comma;
int Cont_indent;
int Right_block;
int Numbers;
int Arg_indent;
int Tag_indent;
int Equ_indent;
int Exp_indent;
int Def_indent;
int Set_indent;
int Placeh_indent;
int BeginEnd_indent;
int Arg_sep;
int Def_format;
int Tag_generate;
int Start_right_block;
int Preserve_empty;
int Tag_length;
int Tag_start;
int Tag_continuous;
char *Keyword_pre;
char *Keyword_post;
int Set_format;
char *Identifier_pre;
char *Identifier_post;
int Dev_indep = 0;
int Keep_exp_format;

int Accept_newlines = 0;

/* Locals */
static int indent = 0;
static int position = 0;
static int line = 0;

static FILE *fpout;

static int spaceline = 0, spacepos = 0;

static int commline = 0, commpos = 0;

static int lastppindent = -1;

static int exp_format;

/* some comments are to be printed later then they are scanned */
struct comment {
    char *s;
    int type;
    int indent;
    struct comment *next;
};
static struct comment *cs = NULL;
static struct comment *cs_start = NULL;

void save_comment(s, type)
char *s;
int type;
{
    struct comment *c;

    c = PSF_MALLOC(struct comment);
    if (cs == NULL)
	cs_start = c;
    else
	cs->next = c;
    cs = c;
    cs->s = psf_strdup(s);
    cs->type = type;
    cs->indent = indent;
}

int comments_saved()
{
    if (cs_start == NULL)
	return(0);
    else
	return(1);
}

static struct comment *get_comm_str()
{
    struct comment *c;

    if (cs_start != NULL) {
	c = cs_start;
	cs_start = cs_start->next;
	if (cs_start == NULL) {
	    cs = NULL;
	    /* no more comments */
	}
	return(c);
    } else
	return(NULL);
}

/* Indentations are recorded, so that returning is simple */
static int nr_indent[64];
static int ind_ind = 0;

static void set_indent(i)
int i;
{
    nr_indent[ind_ind] = i;
    ind_ind ++;
    if (Dev_indep)
	fprintf(fpout, "<I +%d>", i);
}

static int get_indent()
{
    ind_ind --;
    if (Dev_indep)
	fprintf(fpout, "<I -%d>", nr_indent[ind_ind]);
    return(nr_indent[ind_ind]);
}

static void line_number()
{
    line ++;
    if (Numbers)
	fprintf(fpout, "%4d ", line);
}

static void spacing(n)
int n;
{
    int i;

    for (i = 0; i < n; i++) {
	fprintf(fpout, " ");
	position ++;
    }
}

void pp_flush()
{
    if (Dev_indep)
	fprintf(fpout, "<flush>");
}

void pp_indent()
{
    if (position)
	fprintf(fpout, "\n");
    if (Dev_indep)
	fprintf(fpout, "<indent>");
    else
	spacing(indent);
    lastppindent = line;
}

static int last_empty = 1;

void pp_force_newline()
{
    if (line) { /* first line is always forced! */
	if (position == 0) {
	    line_number();
	    last_empty = 1;
	} else
	    last_empty = 0;

	if (Dev_indep)
	    pp_flush();
	fprintf(fpout, "\n");
	position = 0;
    }
}

static empty_lines = 0;

static void pp_newlinex()
{
    if (position) {
	pp_force_newline();
    }
    while (empty_lines) {
	if (Dev_indep)
	    pp_flush();
	else
	    pp_force_newline();
	empty_lines --;
    }
}

void pp_newline()
{
    struct comment *c;
    int first = 1;
    int last_type = -1; /* indicates no comment */

    while ((c = get_comm_str()) != NULL) {
	if (first) {
	    if (c->type != COMMENT_RIGHT)
		pp_newlinex();
	    first = 0;
	}
	if (c->type == COMMENT_LEFT) {
	    set_indent(- indent);
	    indent = 0;
	} else if (c->type == COMMENT_SKIPCODE) {
	    set_indent(0);
	} else {
	    set_indent(c->indent - indent);
	    indent = c->indent;
	    if (c->type == COMMENT_RIGHT)
		pp_space();
	}
	pp_comment(c->s);
	indent -= get_indent();
	last_type = c->type;
    }
    if (last_type != COMMENT_LEFT)
	pp_newlinex();
}

#define STR_NORMAL 0
#define STR_KEYWORD 1
#define STR_IDENTIFIER 2

static void to_indent();

static int X = 0;

static void print_string(s, special)
char *s;
int special;
{
    if (position == 0) {
	line_number();
	if (!X)
	    pp_indent();
    } else if (line > lastppindent) {
	if (!(strlen(s) == 1 && *s == ' '))
	    if (!X)
		to_indent(0);
	lastppindent = line;
	/*
	 * We don't want a single indent on a space.
	 */
    }

    position += strlen(s);
    if (! Dev_indep) {
	if (position > Length) {
	    while (*s == ' ') /* Lets not begin with space */
		s ++;
	    if (*s == '\0')
		return;
		/*
		 * Since we updated 'position', the next item will be on a
		 * newline.
		 * This avoids the problem that position is still 0, so another
		 * indent is done.
		 */
	    pp_newline();
	    line_number();
	    if (Dev_indep)
		set_indent(Cont_indent);
	    else
		indent += Cont_indent;
	    pp_indent();
	    if (Dev_indep)
		indent -= get_indent();
	    else
		indent -= Cont_indent;
	    position += strlen(s);
	}
    }
    switch (special) {
    case STR_KEYWORD:
	if (Dev_indep) {
	    fprintf(fpout, "<keyword %s>", s);
	} else {
	    if (Keyword_pre != NULL)
		fprintf(fpout, "%s", Keyword_pre);
	    fprintf(fpout, "%s", s);
	    if (Keyword_post != NULL)
		fprintf(fpout, "%s", Keyword_post);
	}
	break;
    case STR_IDENTIFIER:
	if (Dev_indep) {
	    fprintf(fpout, "<identifier %s>", s);
	} else {
	    if (Identifier_pre != NULL)
		fprintf(fpout, "%s", Identifier_pre);
	    fprintf(fpout, "%s", s);
	    if (Identifier_post != NULL)
		fprintf(fpout, "%s", Identifier_post);
	}
	break;
    default:
	fprintf(fpout, "%s", s);
	break;
    }
}

void pprint(s)
char *s;
{
    print_string(s, STR_NORMAL);
}

void pp_keyword(s)
char *s;
{
    print_string(s, STR_KEYWORD);
}

void pp_identifier(s)
char *s;
{
    print_string(s, STR_IDENTIFIER);
}

void pp_operator(s)
char *s;
{
    if (Dev_indep && X) {
	if (exp_format)
	    fprintf(fpout, "<op %s>", s);
	else
	    fprintf(fpout, "<fitop %s>", s);
	position += strlen(s);
    } else
	    pprint(s);
}

void pp_space()
{
    if (line != spaceline || position - 1 != spacepos) {
	spaceline = line;
	spacepos = position;
	pprint(" ");
    }
}

static void to_indent(n)
int n;
{
    if (Dev_indep)
	fprintf(fpout, "<to %d>", n);
    else {
	if (position < indent + n) {
	    spacing(indent + n - position);
	} else {
	    spacing(1);
	}
    }
}

static void escape_hooks(c)
char c;
{
    if (Dev_indep) {
	if (c == '<')
	    fprintf(fpout, "<lhook>");
	else if (c == '>')
	    fprintf(fpout, "<rhook>");
	else
	    fprintf(fpout, "%c", c);
    } else
	fprintf(fpout, "%c", c);
}

/* What if comment exceeds line length? */
void pp_comment(s)
    char *s;
{
    char *i;
    int x;

    if (commline == line && commpos == position) { /* continuous comment */
	pp_newlinex();
    }


    for (; *s == ' ' || *s == '\t'; s ++);
    for (i = s + (strlen(s) - 1); *i == ' ' || *i == '\t'; i --);
    *(i + 1) = '\0';
    X = Dev_indep;
    pp_fit_start();
    for (i = s; *i != '\0'; i++) {
	if (*i == '\n') {
	    pp_fit_end();
	    X = 0;
	    if (commline == line && commpos == 0 && Preserve_empty)
		pp_force_newline();
	    else if (*(i + 1) == '\n' && Preserve_empty)
		pp_force_newline();
	    else
		pp_newlinex();
	    for (; *(i+1) == ' ' || *(i+1) == '\t'; i ++);
	    if (*(i+1) != '\0') {
		X = Dev_indep;
		pp_fit_start();
	    }
	} else {
	    if (!position) {
		line_number();
		if (! Dev_indep)
		    spacing(indent - Place_comment * Indent);
		else {
		    x = get_indent();
		    set_indent(x - Place_comment * Indent);
		    pp_indent();
		    get_indent();
	 	    set_indent(x);
		}
	    }
	    if (*i == '\t') {
		position += 8 - (position % 8);
		escape_hooks(*i);
	    } else {
		position++;
		escape_hooks(*i);
	    }
	}
    }
    if (*(i - 1) == '\n') { /* next comment must be on newline */
	commline = line;
	commpos = position;
    }
    if (X) {
	pp_fit_end();
	X = 0;
    }
}

/* in code comment: -- ... -- */
void pp_comment2(s)
char *s;
{
    char *i;

    for (;*s == ' ' || *s == '\t'; s ++);
    for (i = s + (strlen(s) - 1); *i == ' ' || *i == '\t'; i --);
    *(i + 1) = '\0';
    pp_space();
    for (; *s != '\0'; s ++)
	escape_hooks(*s);
    pp_space();
}

/* empty_line followed by comment */
void pp_comment3(s)
char *s;
{
    if (Preserve_empty) {
	if (*s == '\n') {
	    if (position)
		empty_lines ++;
	    else {
		pp_force_newline();
	    }
	}
	s ++;
    }
    pp_comment(s);
    pp_force_newline();
}

void pp_skipcode(s)
char *s;
{
    if (position)
	save_comment(s, COMMENT_SKIPCODE);
    else {
	save_comment(s, COMMENT_SKIPCODE);
	pp_newline();
    }
}

void pp_empty_codeline()
{
    if (! last_empty) {
	if (line != commline) {
	    if (Preserve_empty) {
		pp_force_newline();
	    }
	}
    }
}

void pp_inc()
{
    indent += Indent;
    set_indent(Indent);
}

void pp_dec()
{
    indent -= get_indent();
    if (indent < 0)
	indent = 0;
}

void pp_comma()
{
    pprint(",");
    if (Blank_comma)
	pp_space();
}

void pp_line_comma()
{
    if (Blank_line_comma) {
/*
	pprint(",");
*/
	pp_newline();
    } /* else
	pp_comma();
*/
}

void pp_begin_block(s)
char *s;
{
    if (Right_block) {
	pp_fit_start();
	pp_space();
	pp_keyword(s);
	pp_fit_end();
	pp_inc();
	pp_newline();
    } else {
	pp_newline();
	pp_fit_start();
	pp_keyword(s);
	pp_fit_end();
	if (Start_right_block && (*s == '{' || *s == '['))
	    to_indent(Indent);
	pp_inc();
	if (*s == '{' || *s == '[') {
	    if (!Start_right_block)
		pp_newline();
	} else /* begin */
	    pp_newline();
    }
}

void pp_end_block(s)
char *s;
{
    pp_dec();
    pp_fit_start();
    pp_keyword(s);
    pp_fit_end();
}

void pp_begin_section()
{
    if (Dev_indep)
	fprintf(fpout, "<section>");
}

void pp_end_section()
{
    if (Blank_section)
	if (! last_empty)
	    pp_force_newline();
}

void pp_beginend_indent()
{
    if (BeginEnd_indent)
	pp_inc();
    else
	set_indent(0);
}

void pp_arg_indent(s)
char *s;
{
    to_indent(Arg_indent);
    pprint(s);
}

void pp_arg_sep(s)
char *s;
{
    if (Arg_sep)
	pp_space();
    pprint(s);
    if (Arg_sep)
	pp_space();
}

void pp_tag_indent()
{
    indent += Tag_indent;
    set_indent(Tag_indent);
}

void pp_arg_start()
{
    X = Dev_indep;
    if (Dev_indep)
	set_indent(Arg_indent);
}

void pp_arg_end()
{
    X = 0;
    if (Dev_indep)
	indent -= get_indent(); /* argument indent */
}

void pp_list_start()
{
    X = Dev_indep;
}

void pp_list_end()
{
    X = 0;
}

void pp_equ_start()
{
    X = Dev_indep;
    if (Dev_indep)
	set_indent(Cont_indent);
}

void pp_equ_end()
{
    X = 0;
    if (Dev_indep)
	indent -= get_indent(); /* continuation indent */
}

void pp_equ_space()
{
    to_indent(1);
}

void pp_equ_toindent()
{
    if (Dev_indep)
	pp_space();
}

void pp_equ_indent()
{
    to_indent(Equ_indent);
    indent += Equ_indent;
    set_indent(Equ_indent);
}

void pp_placeholder_start()
{
    to_indent(Placeh_indent);
    if (Dev_indep)
	set_indent(Placeh_indent);
}

void pp_placeholder_end()
{
    if (Dev_indep)
	indent -= get_indent(); /* placeholder indent */
}

/* set-expression and process-expression routines */
static int def_ind;
static int startexp;

void pp_exp_start(s)
char *s;
{
    X = Dev_indep;
    startexp = 1;
    to_indent(def_ind);
    indent += def_ind;
    set_indent(def_ind);
    pp_fit_start();
    pprint(s);
    pp_fit_end();
    if (exp_format) {
	if (Dev_indep)
	    to_indent(def_ind + Exp_indent);
	else {
	    if (position >= indent + Exp_indent)
		pp_newline();
	    else
		to_indent(Exp_indent);
	}
    } else {
	pp_space();
	if (Dev_indep)
	    set_indent(Cont_indent);
    }
}

void pp_subexp_start()
{
    startexp = 1;
}

void pp_exp_end()
{
    X = 0;
    if (! exp_format)
	if (Dev_indep)
	    indent -= get_indent(); /* continuation indent */
    indent -= get_indent();
    if (!(Keep_exp_format && Accept_newlines))
	pp_newline();
}

void pp_exp_indent()
{
    if (exp_format) {
	indent += Exp_indent;
	set_indent(Exp_indent);
    } else {
	set_indent(0);
    }
}

static int lastspecindl = 0;
static int lastspecindp = 0;

void pp_spec_indent(s)
char *s;
{
    int i;

    for (i = 0; *s != '\0'; s ++) {
	if (*s == ' ')
	    i ++;
	if (*s == '\t')
	    i = (i / 8 + 1) * 8;
    }
    pp_newline();
    if (Dev_indep) {
	if (i > 2 * Indent)
	    to_indent(i - 2 * Indent);
    } else {
	if (i > 2*Indent)
	    to_indent(i - indent);
    }
    position ++; /* we don't want an indent on this line */
    lastspecindl = line;
    lastspecindp = position;
}

void pp_exp_newline()
{
    if (exp_format) {
	if (startexp)
	    startexp = 0;
	else {
	    if (!X)
		pp_newline();
	    else {
		pp_flush();
		pp_indent();
	    }
	}
    }
}

void pp_exp_preop()
{
    if (exp_format) {
	if (!X) {
	    pp_newline();
	} else {
	    pp_flush();
	    pp_indent();
	}
    } else {
	if ((lastspecindl != line) || (lastspecindp != position))
	    pp_space();
    }
}

void pp_exp_postop()
{
    if (exp_format) {
	to_indent(Exp_indent);
    } else
	pp_space();
}

void pp_start_set()
{
    if (Set_format)
	exp_format = 1;
    else
	exp_format = 0;
    def_ind = Set_indent;
}

void pp_start_def()
{
    if (Def_format)
	exp_format = 1;
    else
	exp_format = 0;
    def_ind = Def_indent;
}

/* Tags can be numbered */
static int tagnr;

void pp_init_tag()
{
    if (!Tag_continuous)
	tagnr = Tag_start - 1;
}

void pp_tag(s)
char *s;
{
    char buf[3];

    if (Tag_generate) {
	tagnr ++;
	sprintf(buf, "%0*d", Tag_length, tagnr);
	pprint(buf);
    } else {
	pprint(s);
    }
}

void pp_empty(s)
char *s;
{
    if (Preserve_empty) {
	while (*s) {
	    if (*s == '\n') {
		if (position)
		    empty_lines ++;
		else
		    pp_force_newline();
	    }
	    s ++;
	}
    }
}

void pp_fit_start()
{
    if (! Dev_indep)
	return;
    if (! position) {
	pp_indent();
	position ++;
    }
    if (X)
	fprintf(fpout, "<fit %d>", indent);
	
}

void pp_fit_end()
{
    if (! Dev_indep)
	return;
    if (X)
	fprintf(fpout, "</fit>");
}

void pp_start_module()
{
    if (Dev_indep)
	fprintf(fpout, "<module>");
}

void pp_end_module()
{
    if (Dev_indep)
	fprintf(fpout, "</module>");
}

/* Initialization of Pretty Print */
void init_pp(fp)
FILE *fp;
{
    fpout = fp;

    if (Numbers)
	Length -= 5;

    tagnr = Tag_start - 1;
}

struct option options[] = {
    { "-l", T_INT, 78, &Length},
    { "-idpre", T_CHARP, (int) NULL, &Identifier_pre},
    { "-idpost", T_CHARP, (int) NULL, &Identifier_post},
    { "-i", T_INT, 4, &Indent},
    { "-pc", T_INT, 1, &Place_comment},
    { "-nbas", T_BOOL, 1, &Blank_section},
    { "-bas", T_BOOL, 0, &Blank_section},
    { "-bac", T_BOOL, 0, &Blank_comma},
    { "-nbac", T_BOOL, 1, &Blank_comma},
    { "-nbc", T_BOOL, 1, &Blank_line_comma},
    { "-bc", T_BOOL, 0, &Blank_line_comma},
    { "-ci", T_INT, 4, &Cont_indent},
    { "-nbr", T_BOOL, 1, &Right_block},
    { "-br", T_BOOL, 0, &Right_block},
    { "-nbsr", T_BOOL, 1, &Start_right_block},
    { "-bsr", T_BOOL, 0, &Start_right_block},
    { "-nbei", T_BOOL, 1, &BeginEnd_indent},
    { "-bei", T_BOOL, 0, &BeginEnd_indent},
    { "-bals", T_BOOL, 0, &Arg_sep},
    { "-nbals", T_BOOL, 1, &Arg_sep},
    { "-df", T_BOOL, 0, &Def_format},
    { "-ndf", T_BOOL, 1, &Def_format},
    { "-npel", T_BOOL, 1, &Preserve_empty},
    { "-pel", T_BOOL, 0, &Preserve_empty},
    { "-ntcn", T_BOOL, 1, &Tag_continuous},
    { "-sf", T_BOOL, 0, &Set_format},
    { "-nsf", T_BOOL, 1, &Set_format},
    { "-tcn", T_BOOL, 0, &Tag_continuous},
    { "-ntg", T_BOOL, 1, &Tag_generate},
    { "-tg", T_BOOL, 0, &Tag_generate},
    { "-tl", T_INT, 2, &Tag_length},
    { "-ts", T_INT, 1, &Tag_start},
    { "-nkef", T_BOOL, 1, &Keep_exp_format},
    { "-kef", T_BOOL, 0, &Keep_exp_format},
    { "-n", T_BOOL, 0, &Numbers},
    { "-ai", T_INT, 12, &Arg_indent},
    { "-ti", T_INT, 0, &Tag_indent},
    { "-ei", T_INT, 8, &Equ_indent},
    { "-pei", T_INT, 3, &Exp_indent},
    { "-di", T_INT, 12, &Def_indent},
    { "-si", T_INT, 8, &Set_indent},
    { "-phi", T_INT, 24, &Placeh_indent},
    { "-kwpre", T_CHARP, (int) NULL, &Keyword_pre},
    { "-kwpost", T_CHARP, (int) NULL, &Keyword_post},
    { "-D", T_BOOL, 0, &Dev_indep},
    { NULL, 0, 0, NULL},
};

extern int yyparse PROTO_ARGS((void));

int pretty_print(argc, argv, fp)
int argc;
char *argv[];
FILE *fp;
{
    int i;
    psf_file pf;
    static suffix suffixes[] = {{PSFSUFFIX, PSFSUFFIX_DEFAULT}, {NULL, NULL}};

    i = get_ext_options(argc - 1, argv + 1, "PSFPRETTY", options);
    i ++;
    switch (get_ext_options_error()) {
    case 0:
	break;
    case 1:
	fprintf(stderr, "illegal option %s\n", argv[i]);
	exit(1);
    case 2:
	fprintf(stderr, "error in environment variable %s\n", "PSF_PRETTY");
	exit(1);
    }
    if (Def_format)
	Keep_exp_format = 0;

    init_pp(fp);

    if (Dev_indep)
	fprintf(fpout, "<psf>");
    if (i == argc) {
	yyin = stdin;
	if (yyparse())
	    return(EXIT_SYNTAX_ERR);
    } else
	for (; i < argc; i++) {
	    pf = psf_fopen(argv[i], suffixes);
	    yyin = pf.fp;
	    filename = pf.name;
	    if (yyparse())
		return(EXIT_SYNTAX_ERR);
	    fclose(yyin);
	}
    if (Dev_indep)
	fprintf(fpout, "</psf>");

    return(errorseen);
}
