/*
 * ppp back-end for groff
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "psf_prototype.h"
#include "psf_standards.h"
#include "version.h"

#define USAGE "usage: ppptoroff [-hv]\n"

#define C_PSF 1
#define C_ENDPSF 2
#define C_MODULE 3
#define C_ENDMODULE 4
#define C_SECTION 5
#define C_INDENT 6
#define C_TO 7
#define C_SETINDENT 8
#define C_KEYWORD 9
#define C_IDENTIFIER 10
#define C_FITS 11
#define C_FITE 12
#define C_FITOP 13
#define C_OP 14
#define C_FLUSH 15
#define C_LHOOK 16
#define C_RHOOK 17

#define CHAR_INCH 16
#define POINTS_INCH 72000

struct code {
    char *str;
    int nr;
};

static struct code comm[] = {
    { "psf", C_PSF },
    { "/psf", C_ENDPSF },
    { "module", C_MODULE },
    { "/module", C_ENDMODULE },
    { "section", C_SECTION },
    { "indent", C_INDENT },
    { "to", C_TO },
    { "I", C_SETINDENT },
    { "keyword", C_KEYWORD },
    { "identifier", C_IDENTIFIER },
    { "fit", C_FITS },
    { "/fit", C_FITE },
    { "fitop", C_FITOP },
    { "op", C_OP },
    { "flush", C_FLUSH },
    { "lhook", C_LHOOK },
    { "rhook", C_RHOOK },
    { NULL, 0 }
};

static int get_code(s)
char *s;
{
    int i;

    for (i = 0; comm[i].str != NULL; i ++) {
	if (!strcmp(comm[i].str, s))
	    return(comm[i].nr);
    }
    return(-1);
}

static int pos = 0; /* position on the output-line */
static int indent = 0;
static int NewLine = 1;

static void pr_char(c)
int c;
{
    int c2;

    if (c == '.' && pos == 0)
	printf("\\&");
    else if (c == '\\')
	putchar('\\');
    else if (c == '-') {
	if ((c2 = getchar()) == '>') {
	    printf("\\(->");
	    return;
	} else
	    ungetc(c2, stdin);
    } else if (c == ' ' && (pos == 0 || NewLine))
	return;
    if (! (c == '\n' && NewLine))
	putchar(c);
    if (c == '\n') {
	if (pos == 0 && NewLine == 1)
	    printf(".if \\n(.ku+\\w'\\*[psfev*line]'u=0 .sp\n");
	else
	    printf(".br\n");
	NewLine = 1;
	pos = 0;
    } else {
	pos ++;
	NewLine = 0;
    }
}

static void newline()
{
    if (! NewLine)
	printf("\n");
}

static char buf[256];
static int fitind;
static int lastindent;

static void command()
{
    int i;
    char c;

    scanf("%[^> ]s", buf);
    switch(get_code(buf)) {
    case C_PSF:
	/*
	 * Break the current line.
	 */
	printf(".br\n");
	printf(".ne 5\n");
	/*
	 * Header of the pretty printed code. The macro PSFB can be
	 * defined by the user.
	 */
	printf(".if !dPSFB \\{\\\n");
	printf(".de PSFB\n");
	printf("\\l'\\n(.lu-\\n(.iu'\n");
	printf(".sp 0.1i\n");
	printf(".nr psfev*in 0+0.5i\n");
	printf(".nr psfev*ll 0-0.5i\n");
	printf("..\n");
	printf(".\\}\n");
	printf(".PSFB\n");
	/*
	 * We switch to a new environment, but we need the linelength and
	 * indentation of the current environment.
	 */
	printf(".nr psfev*ll +\\n(.lu\n");
	printf(".nr psfev*in +\\n(.iu\n");
	printf(".ev psfev\n");
	printf(".nf\n");
	printf(".ll \\n[psfev*ll]u\n");
	/*
	 * Initialization of the strings that build up the line.
	 * They need to be defined, although they contain nothing at the
	 * moment.
	 */
	printf(".ds psfev*line \"\n");
	printf(".ds psfev*pexpr \"\n");
	    /* keywords and identifiers pre and post strings */
	printf(".if !dpsfev*kwpre .ds psfev*kwpre \\fB\n");
	printf(".if !dpsfev*kwpost .ds psfev*kwpost \\fP\n");
		/* identifiers in italic with skew corrections */
	printf(".if !dpsfev*idpre .ds psfev*idpre \\,\\fI\n");
	printf(".if !dpsfev*idpost .ds psfev*idpost \\fP\\/\n");
	    /* define the process operator-strings */
	printf(".if !dpsfev*op|| .ds psfev*op|| \\v'-0.1m'\\s-1\\fB\\|\\(br\\|\\(br\\|\\fP\\s+1\\v'+0.1m'\n");
	printf(".if !dpsfev*op. .ds psfev*op. \\v'-0.2m'\\fB.\\fP\\v'+0.2m'\n");
	printf(".if !dpsfev*op+ .ds psfev*op+ \\fB+\\fP\n");
	printf(".if !dpsfev*op* .ds psfev*op* \\v'-0.2m'\\fB*\\fP\\v'+0.2m'\n");
	printf(".if !dpsfev*op# .ds psfev*op# \\v'-0.2m'\\fB#\\fP\\v'+0.2m'\n");
	printf(".if !dpsfev*op| .ds psfev*op| \\v'-0.1m'\\s-1\\fB\\ \\(br\\ \\fP\\s+1\\v'+0.1m'\n");
	printf(".if !dpsfev*op\\\\ .ds psfev*op\\\\ \\v'+0.2m'\\s+2\\fB\\e\\fP\\s-2\\v'-0.2m'\n");
	break;
    case C_ENDPSF:
	printf(".ev\n"); /* Return from the psf environment. */
	/*
	 * Trailer of the pretty printed code. The macro PSFE can be
	 * defined by the user.
	 */
	printf(".if !dPSFE \\{\\\n");
	printf(".de PSFE\n");
	printf("\\l'\\n(.lu-\\n(.iu'\n");
	printf("..\n");
	printf(".\\}\n");
	printf(".PSFE\n");
	break;
    case C_MODULE:
	printf(".ne 5\n");
	break;
    case C_ENDMODULE:
	break;
    case C_SECTION:
	printf(".ne 3\n");
	break;
    case C_INDENT:
	/*
	 * If we have something in psfev*line, then print it.
	 * Do an indent.
	 * Start a new psfev*line.
	 */
	printf(".if \\w'\\*[psfev*line]'u>0 \\*[psfev*line]\n");
	printf(".in \\n[psfev*in]u+%du\n", (indent*POINTS_INCH) / CHAR_INCH);
	printf(".ds psfev*line \"\\&");
	NewLine = 0;
	lastindent = indent;
	break;
    case C_TO:
	scanf(" %d", &i);
	newline();
	/*
	 * If there fits a space between the point we are and the point we
	 * are going to, then append a horizontal move to this point.
	 * Else, append a horizontal move the width of a space.
	 * Start an append to psfev*line.
	 */
	printf(".ie %du-\\n(.ku-\\w'\\*[psfev*line] 'u>0 \\{\\\n",
	    ((i)*POINTS_INCH)/CHAR_INCH);
	printf(".nr psfev*k \\n(.ku+\\w'\\*[psfev*line]'u\n");
	printf(".as psfev*line \\h'%du-\\n[psfev*k]u'\n",
	    ((i)*POINTS_INCH)/CHAR_INCH);
	printf(".\\}\n");
	printf(".el .as psfev*line \\h'\\w' 'u'\n");
	printf(".as psfev*line \"");
	NewLine = 0;
	break;
    case C_SETINDENT:
	scanf(" %c%d", &c, &i);
	if (c == '-')
	    indent -= i;
	else
	    indent += i;
	break;
    case C_KEYWORD:
	scanf(" %[^>]s", buf);
	/*
	 * Keywords with pre and post strings.
	 */
	printf("\\*[psfev*kwpre]%s\\*[psfev*kwpost]", buf);
	pos += strlen(buf);
	NewLine = 0;
	break;
    case C_IDENTIFIER:
	scanf(" %[^>]s", buf);
	/*
	 * Identifiers with pre and post strings.
	 */
	printf("\\*[psfev*idpre]%s\\*[psfev*idpost]", buf);
	pos += strlen(buf);
	NewLine = 0;
	break;
    case C_FITS:
	scanf(" %d", &fitind);
	newline();
	/*
	 * Start a string for the thing we have to fit.
	 */
	printf(".ds psfev*pexpr \"\\&");
	NewLine = 0;
	break;
    case C_FITE:
	newline();
	/*
	 * If pexpr doesn't fit on this line, then
	 *    output the line
	 *    go to a new line
	 *    do an indent
	 *    define the start of a newline to be pexpr
	 * Else append pexpr to the line.
	 */
	printf(".ie \\w'\\*[psfev*line] \\*[psfev*pexpr]'-\\n(.lu+\\n(.ku+\\n(.iu>0 \\{\\\n");
	printf("\\*[psfev*line]\n");
	printf(".br\n");
	printf(".in \\n[psfev*in]u+%du\n", (indent*POINTS_INCH) / CHAR_INCH);
	printf(".ds psfev*line \"\\*[psfev*pexpr]\n");
	printf(".\\}\n");
	printf(".el \\{\\\n.as psfev*line \"\\*[psfev*pexpr]\n.\\}\n");
	NewLine = 1;
	break;
    case C_FITOP:
	newline();
	scanf(" %[^>]s", buf);
	/*
	 * The same as with C_FITE, but we have also to do spacing around
	 * the operator.
	 */
	printf(".ie \\w'\\*[psfev*line]\\ \\*[psfev*op%s]\\ '-\\n(.lu+\\n(.ku+\\n(.iu>0 \\{\\\n", buf);
	printf("\\*[psfev*line]\n");
	printf(".br\n");
	printf(".in \\n[psfev*in]u+%du\n", (indent*POINTS_INCH) / CHAR_INCH);
	printf(".ds psfev*line \"\\&\\*[psfev*op%s]\\ \n", buf);
	printf(".\\}\n");
	printf(".el \\{\\\n.as psfev*line \"\\ \\*[psfev*op%s]\\ \n.\\}\n", buf);
	NewLine = 1;
	pos ++;
	break;
    case C_OP:
	newline();
	scanf(" %[^>]s", buf);
	/*
	 * Simple add the string for the operator.
	 */
	printf(".as psfev*line \\*[psfev*op%s]\n", buf);
	NewLine = 1;
	pos ++;
	break;
    case C_FLUSH:
	newline();
	/*
	 * Output the line and make it empty.
	 */
	printf(".if \\w'\\*[psfev*line]'u>0 \\*[psfev*line]\n");
	printf(".ds psfev*line \"\\&\n");
	NewLine = 1;
	break;
    case C_LHOOK:
	pr_char('<');
	break;
    case C_RHOOK:
	pr_char('>');
	break;
    default:
	fprintf(stderr, "unknown command \"%s\"\n", buf);
	exit(1);
    }
}

int main(argc, argv)
int argc;
char *argv[];
{
    char *progname;
    int opt;
    int c;

    progname = psf_basename(argv[0]);
    while ((opt = getopt(argc, argv, "hv")) != -1) {
	switch (opt) {
	case 'h':
	case '?':
	    fprintf(stderr, USAGE);
	    exit(1);
	case 'v':
	    fprintf(stderr, "%s version %s, last compiled %s\n", progname,
		VERSION, __DATE__);
	    exit(0);
	}
    }

    while ((c = getchar()) != EOF) {
	if (c == '<') {
	    command();
	    getchar(); /* '>' */
	} else {
	    pr_char(c);
	}
    }
    exit(0);
}
