#include <stdio.h>
#include "psf_prototype.h"
#include "psf_malloc.h"
#include "tiltype.h"
#include "ptree.h"
#include "prtilparts.h"

/*
 * functions for Ptree.
 */

static ptree Null = { NULL, NULL, '\\', NULL, NULL, NULL, NULL };
ptree *NullPtree = &Null;

static ptree *ptreeFreeList = NULL;
static struct {
    int new;
    int old;
    int free;
} account = { 0, 0, 0 };

ptree *AllocPtree()
{
    ptree *pt;

    if (ptreeFreeList == NULL) {
	account.new ++;
	pt = PSF_MALLOC(ptree);
    } else {
	account.old ++;
	pt = ptreeFreeList;
	ptreeFreeList = (ptree *) ptreeFreeList->next;
    }
    pt->id = NULL;
    pt->pid = NULL;
    pt->pe = NULL;
    pt->edge = NULL;
    pt->comm = NULL;
    return(pt);
}

void FreePtree(pt)
ptree *pt;
{
    pt->next = ptreeFreeList;
    ptreeFreeList = pt;
    account.free ++;
}

void PrintAccountPtree()
{
    fprintf(stderr, "Ptree  new: %d  old: %d  free: %d\n", account.new, account.old,
	account.free);
}

static pedge *pedgeFreeList = NULL;
static struct {
    int new;
    int old;
    int free;
} accountPedge = { 0, 0, 0 };

pedge *AllocPedge()
{
    pedge *e;

    if (pedgeFreeList == NULL) {
	accountPedge.new ++;
	e = PSF_MALLOC(pedge);
    } else {
	accountPedge.old ++;
	e = pedgeFreeList;
	pedgeFreeList = pedgeFreeList->next;
    }
    e->node = NULL;
    e->atom = NULL;
    e->next = NULL;
    return(e);
}

void FreePedge(e)
pedge *e;
{
    e->next = pedgeFreeList;
    pedgeFreeList = e;
    accountPedge.free ++;
}

void PrintAccountPedge()
{
    fprintf(stderr, "Pedge  new: %d  old: %d  free: %d\n", accountPedge.new,
	accountPedge.old, accountPedge.free);
}

static atomlist *atomFreeList = NULL;
static struct {
    int new;
    int old;
    int free;
} accountAtom = { 0, 0, 0 };

atomlist *AllocAtom()
{
    atomlist *a;

    if (atomFreeList == NULL) {
	accountAtom.new ++;
	a = PSF_MALLOC(atomlist);
    } else {
	accountAtom.old ++;
	a = atomFreeList;
	atomFreeList = atomFreeList->next;
    }
    a->atom = NULL;
    a->props.sum = 0;
    a->props.encaps = 0;
    a->props.hide = 0;
    a->props.pid = 0;
    a->props.comm = 0;
    a->next = NULL;
    return(a);
}

void FreeAtom(a)
atomlist *a;
{
    a->next = atomFreeList;
    atomFreeList = a;
    accountAtom.free ++;
}

void PrintAccountAtom()
{
    fprintf(stderr, "Atom  new: %d  old: %d  free: %d\n", accountAtom.new,
	accountAtom.old, accountAtom.free);
}

/*
 * Ptree
 */

static void CreatePtree(pt)
ptree *pt;
{
    int i;
    char fun;
    pedge *e1, *e2;

    fun = pt->pe->fun;
    switch(fun) {
    case SKP:
    case AET:
    case DLK:
	pt->fun = fun;
	break;
    case ALT:
    case SEQ:
    case PAR:
    case INTR:
    case DISR:
    case STAR:
    case SHARP:
	pt->fun = fun;
	e1 = NULL;
	for (i = pt->pe->proc_expr.pe3.a - 1; i >= 0; i --) {
	    e2 = AllocPedge();
	    e2->node = AllocPtree();
	    e2->node->pe = &pt->pe->proc_expr.pe3.pe[i];
	    e2->next = e1;
	    e1 = e2;
	    CreatePtree(e2->node);
	}
	pt->edge = e1;
	break;
    case MRG:
	pt->fun = fun;
	e1 = AllocPedge();
	e1->node = AllocPtree();
	e1->node->pe = pt->pe->proc_expr.pe1.pe;
	CreatePtree(e1->node);
	e1->next = NULL;
	e2 = AllocPedge();
	e2->node = AllocPtree();
	e2->node->pe = pt->pe->proc_expr.pe1.pe;
	CreatePtree(e2->node);
	e2->next = e1;
	pt->edge = e2;
	break;
    case SUM:
    case ENC:
    case HID:
	pt->fun = fun;
	pt->edge = AllocPedge();
	pt->edge->node = AllocPtree();
	pt->edge->node->pe = pt->pe->proc_expr.pe1.pe;
	CreatePtree(pt->edge->node);
	break;
    case IF:
	pt->fun = fun;
	pt->edge = AllocPedge();
	pt->edge->node = AllocPtree();
	pt->edge->node->pe = pt->pe->proc_expr.pe4.pe;
	CreatePtree(pt->edge->node);
	break;
    case PRIO:
	pt->fun = fun;
	pt->edge = AllocPedge();
	pt->edge->node = AllocPtree();
	pt->edge->node->pe = pt->pe->proc_expr.pe5.pe;
	CreatePtree(pt->edge->node);
	break;
    default:
	break;
    }
}

void DestroyPtree(pt)
ptree *pt;
{
    int i;
    pedge *e, *e2;
    atomlist *a1, *a2;

    if(pt == NullPtree)
	return;
    if (pt->edge != NULL) {
	for (e = pt->edge; e != NULL; ) {
	    DestroyPtree(e->node);
	    for (a1 = e->atom; a1 != NULL; ) {
		a2 = a1;
		a1 = a1->next;
		FreeAtom(a2);
	    }
	    e2 = e;
	    e = e->next;
	    FreePedge(e2);
	}
    }
    FreePtree(pt);
}

ptree *BuildPtree(aet, pexpr)
ae_term *aet;
process_expr *pexpr;
{
    ptree *pt;

    pt = AllocPtree();
    pt->id = aet;
    pt->fun = AET;
    pt->edge = AllocPedge();
    pt->edge->node = AllocPtree();
    pt->edge->node->pe = pexpr;

    CreatePtree(pt->edge->node);

    return(pt);
}

ptree *CopyPtree(ptree *);

static pedge *CopyEdgeList(e)
pedge *e;
{
    pedge *ce;

    if (e == NULL)
	return(NULL);
    ce = PSF_MALLOC(pedge);
    *ce = *e;
    ce->node = CopyPtree(e->node);
    ce->next = CopyEdgeList(e->next);
    return(ce);
}

ptree *CopyPtree(pt)
ptree *pt;
{
    ptree *cg;
    pedge *ce;

    cg = PSF_MALLOC(ptree);
    *cg = *pt;
    cg->edge = CopyEdgeList(pt->edge);
    return(cg);
}

/******/

static int indent = 0;

static void xPrintProcessFunction(c)
char c;
{
    switch(c) {
    case ALT:
	printf("+"); break;
    case SEQ:
	printf("."); break;
    case PAR:
	printf("||"); break;
    case INTR:
	printf("interrupt"); break;
    case DISR:
	printf("disrupt"); break;
    case STAR:
	printf("*"); break;
    case SHARP:
	printf("#"); break;
    case SUM:
	printf("sum"); break;
    case MRG:
	printf("merge"); break;
    case ENC:
	printf("encaps"); break;
    case HID:
	printf("hide"); break;
    case IF:
	printf("if"); break;
    case PRIO:
	printf("prio"); break;
    case SKP:
	printf("skip"); break;
    default:
	printf("*%c", c); break;
    }
}

void PrintPtree(pt)
ptree *pt;
{
    int i;
    pedge *e;

    if (pt->id != NULL) {
	print_ae_term(pt->id);
	printf(" =\n");
	indent += 8;
    }
    if (pt->edge != NULL) {
	if (pt->fun != AET) {
	    for (i = 0; i < indent; i ++) {
		printf(" ");
	    }
	    xPrintProcessFunction(pt->fun);
	    printf("\n");
	}
    } else {
	for (i = 0; i < indent; i ++) {
	    printf(" ");
	}
	print_process_expr(pt->pe);
	printf("\n");
    }
    indent += 4;
    for (e = pt->edge; e != NULL; e = e->next) {
	PrintPtree(e->node);
    }
    indent -= 4;
    if (pt->id != NULL) {
	indent -= 8;
    }
}
