/**********************************************************************\

   Module for deleting cycles of inert transitions and sorting the
   non-bottom states of B0.

\**********************************************************************/

#include <stdio.h>
#include "psf_prototype.h"
#include "datatypes.h"
#include "dlist.h"
#include "global.h"
#include "print.h"
#include "readgraph.h"
#include "search.h"

#ifdef SEARCH
#include "print.h"
#endif

static ListP Stack;
static int count;
static blockP B;
static stateP Next;

void ReplaceState(search, replace)
/**********************************************************************\
   Replaces state search with state replace by replacing Inert(replace)
   with the union of Inert(search) and Inert(replace), discarding
   loops and removing state search.
\**********************************************************************/
    stateP search, replace;
{
    state_ptr tmp;
    transitionP T, Next;

#ifdef SEARCH
    printf("transitions are being transferred:\n");
#endif

    if (VERBOSE) {
	print_state(stderr, search);
	fprintf(stderr, " has been replaced by ");
	print_state(stderr, replace);
	fprintf(stderr, ".\n");
    }
    while (!DEmptyList(INERT(search))) {
	T = DFirst(INERT(search));
	DDelete(INERT(search), T);
	if (STATE(TARGET(T)) != STATE(replace)) {
	    DAddTail(INERT(replace), T);

#ifdef SEARCH
	    print_transition(stdout, T);
	    printf("has been moved from Inert(");
	    print_state(stdout, search);
	    printf(") to Inert(");
	    print_state(stdout, replace);
	    printf(")\n");
#endif
	} else {

#ifdef SEARCH
	    print_transition(stdout, T);
	    printf(" has been removed from Inert(");
	    print_state(stdout, search);
	    printf(")\n");
#endif

	    if (VERBOSE) {
		print_transition(stderr, T);
		fprintf(stderr, " has been discarded.\n");
	    }
	    PSF_FREE(T);
	    NumberOfTransitions--;
	}
    }				/* move inert transitions */
    PSF_FREE(INERT(search));

#ifdef SEARCH
    printf("looking for superfluous transitions:\n");
#endif

    for (T = DFirst(INERT(replace)); DValid(T); T = Next) {
	Next = DNext(T);
	tmp = STATE(TARGET(T));
	if (tmp == STATE(search) || tmp == STATE(replace)) {
	    DDelete(INERT(replace), T);

#ifdef SEARCH
	    print_transition(stdout, T);
	    printf(" has been removed from Inert(");
	    print_state(stdout, replace);
	    printf(")\n");
#endif

	    PSF_FREE(T);
	    NumberOfTransitions--;
	}
    }				/* remove superfluous transitions */

#ifdef SEARCH
    printf("state ");
    print_state(stdout, search);
    printf(" has been removed from B0\n");
#endif

    tmp = STATE(search);
    STATE(search) = STATE(replace);
    DDelete(BOTTOM(B), search);
    PSF_FREE(tmp);
    NumberOfStates--;

#ifdef SEARCH
    print_block(B);
#endif
}				/* ReplaceState */

void Search(s)
/**********************************************************************\
   Performs a depth-first traversal of the inert transitions of B0,
   collapsing cycles of inert transitions to a single state.
\**********************************************************************/
    stateP s;
{
    elementP top;
    stateP t, x;
    transitionP TP;
    bool Collapsing;

    DFNUMBER(s) = count;
    LOWLINK(s) = count++;

#ifdef SEARCH
    printf("\n\nDFNUMBER(");
    print_state(stdout, s);
    printf(")=%d, LOWLINK(", DFNUMBER(s));
    print_state(stdout, s);
    printf(")=%d\n", LOWLINK(s));
#endif

    FLAG(s) = ONSTACK;
    AddPointer(Stack, s);

#ifdef SEARCH
    print_state(stdout, s);
    printf(" is being put on the stack\n");
#endif

    for (TP = DFirst(INERT(s)); DValid(TP); TP = DNext(TP)) {
	t = TARGET(TP);

#ifdef SEARCH
	printf("\n");
	print_state(stdout, t);
	printf(" is the target of an inert transition from ");
	print_state(stdout, s);
	printf("\n");
#endif

	if (FLAG(t) == NON) {

#ifdef SEARCH
	    print_state(stdout, t);
	    printf(" is a non-bottom state in Inert(");
	    print_state(stdout, s);
	    printf("\n");
#endif

	    Search(t);
	    if (LOWLINK(t) < LOWLINK(s))
		LOWLINK(s) = LOWLINK(t);
	} else if (FLAG(t) == ONSTACK)
	    if ((DFNUMBER(t) < DFNUMBER(s)) && (DFNUMBER(t) < LOWLINK(s)))
		LOWLINK(s) = DFNUMBER(t);
    }				/* for all inert transitions starting in s */
    if (LOWLINK(s) == DFNUMBER(s)) {
	/* s is the root of a cycle of inert transitions */

#ifdef SEARCH
	printf("\n");
	print_state(stdout, s);
	printf(" is the root of a cycle of inert transitions!!!\n");
	printf("-----------------------------------------------\n");
#endif

	Collapsing = TRUE;
	while (Collapsing) {
	    top = DFirst(Stack);
	    x = DeletePointer(Stack, top);

#ifdef SEARCH
	    print_state(stdout, x);
	    printf(" is in the cykel and has been popped off the stack\n");
#endif

	    if (STATE(x) == STATE(s)) {
		Collapsing = FALSE;
		FLAG(s) = CLEAR;

#ifdef SEARCH
		printf("the flag of ");
		print_state(stdout, s);
		printf(" has been reset\n");
#endif

		Next = DNext(s);
		DDelete(BOTTOM(B), s);
		DAddTail(NONBOTTOM(B), s);

#ifdef SEARCH
		print_state(stdout, s);
		printf(" has been moved from BOTTOM to NONBOTTOM\n");
#endif
	    }			 /* move root of cycle to NonBottom(B0) */
	    else {

#ifdef SEARCH
		print_state(stdout, x);
		printf(" will be replaced by ");
		print_state(stdout, s);
		printf(" .... \n");
#endif

		ReplaceState(x, s);

#ifdef SEARCH
		printf("done\n");
#endif
	    }
	}			/* while collapsing cycle */
    }				/* if s is root of cycle */
}				/* Search */

void sort_states(B0)
/**********************************************************************\
   Splits the states into bottom states and non-bottom states and
   sorts the non-bottom states of B0 such that for all s,t in
   NonBottom(B0): if there exists an inert transition from s to t,
   then t is before s in NonBottom(B0). Cycles of inert transitions
   are collapsed to a single state; method used: Aho, Hopcroft &
   Ullman (1974).
\**********************************************************************/
    blockP B0;
{
    stateP S;

    B = B0;
    Stack = list_alloc();

#ifdef SEARCH
    printf("begin sortering all states\n");
    print_block(B);
#endif

    count = 1;
    DNewList(Stack);
    for (S = DFirst(BOTTOM(B)); DValid(S); S = Next) {

#ifdef SEARCH
	printf("**************************************");
	printf("*************************************\n");
	printf("*                             sort_state: ");
	print_state(stdout, S);
	printf("                              *\n");
	printf("**************************************");
	printf("*************************************\n");
#endif

	Next = DNext(S);
	if (FLAG(S) == NON) {

#ifdef SEARCH
	    print_state(stdout, S);
	    printf(" is a non-bottom state in BOTTOM\n");
#endif

	    Search(S);
	}			/* if S non-bottom state */
	FLAG(S) = CLEAR;

#ifdef SEARCH
	printf("the flag of ");
	print_state(stdout, S);
	printf(" has been reset\n");
#endif
    }				/* for all states */

#ifdef SEARCH
    printf("the states have been sorted\n");
#endif
}				/* sort_states */
