/**********************************************************************\

   Module for converting input to internal representation.

\**********************************************************************/

#include <stdio.h>
#include "psf_prototype.h"
#include "datatypes.h"
#include "dlist.h"
#include "global.h"
#include "interface.h"
#include "print.h"
#include "readgraph.h"

/**********************************************************************\
   Add a transition to a list, sorted by label
\**********************************************************************/
void AddSorted(LP, TP)
    ListP LP;
    transitionP TP;
{
    transitionP tmp = DFirst(LP), prev = tmp;
    bool found = FALSE;

    while (DValid(tmp)) {
	if (LABEL(tmp) < LABEL(TP)) {
	    prev = tmp;
	    tmp = DNext(tmp);
	} else {
	    found = TRUE;
	    break;
	}
    }
    if (found) {
	if (tmp == prev)
	    DAddHead(LP, TP);	/* LP is empty or TP has the lowest label */
	else
	    DInsert(LP, prev, TP);	/* found the right spot */
    } else
	DAddTail(LP, TP);	/* TP has the highest label */
}				/* AddSorted */

/**********************************************************************\
   Add a state with the unique number name to the list of non-bottom
   states of B if not already added.
   Returns a pointer to the state.
\**********************************************************************/
stateP AddState(B, name)
    blockP B;
    int name;
{
    stateP SP;

    if ((SP = match_state(BOTTOM(B), name)) == NULL) {
	SP = new_state();
	DAddHead(BOTTOM(B), SP);
	BLOCKREF(SP) = B;	/* set block-reference to B */
	NAME(SP) = name;
	NumberOfStates++;
    }
    return (SP);
}				/* extract_state */

/**********************************************************************\
   Add a transition to block B by extracting a source state, a label
   and a target state from the input via the interface and adding them
   to block B.
   Returns a pointer to the transition.
\**********************************************************************/
transitionP AddTransition(B)
    blockP B;
{
    transitionP TP;
    stateP SP;
    int label;

    TP = new_transition();
    label = ExtractSource();
    SP = AddState(B, label);
    SOURCE(TP) = SP;
    if (ACTUALNAME(SP) == NULL)
	ACTUALNAME(SP) = NameToString(label);

    label = ExtractLabel();
    LABEL(TP) = label;
    ACTUALLABEL(TP) = LabelToString(label);

    label = ExtractTarget();
    SP = AddState(B, label);
    TARGET(TP) = SP;
    if (ACTUALNAME(SP) == NULL)
	ACTUALNAME(SP) = NameToString(label);
    NumberOfTransitions++;
    return (TP);
}				/* extract_trans */

/**********************************************************************\
   Extract all transitions from the input via the interface,
   discarding loops of inert transitions (i.e. S --TAU--> S).
   Returns a pointer to a block that contains all of the states
   and all of the transitions.
\**********************************************************************/
blockP extract_graph()
{
    transitionP TP;		/* transition */
    stateP s, t;		/* start & target state */
    blockP B0;			/* initial block */

    if (VERBOSE)
	fprintf(stderr, "\nextracting transition system from input ...\n");
    B0 = new_block();
    BLOCKNUM(B0) = NumberOfBlocks++;
    while (MoreTransitions()) {
	TP = AddTransition(B0);
	s = SOURCE(TP);
	t = TARGET(TP);
	if (SHOWINPUT) {
	    fprintf(fd_out, "transition: ");
	    print_transition(fd_out, TP);
	    fprintf(fd_out, "\n");
	}
	if (LABEL(TP) == TAU ) {
	    /* s is a non-bottom state in B0 */
	    if (s == t) {
		PSF_FREE(TP);	/* discard loops */
		NumberOfTransitions--;
		if (FLAG(s) == CLEAR)
		    FLAG(s) = BOT;
	    } else {
		if (FLAG(s) == CLEAR)
		    FLAG(s) = BOT /* was NON Mark, 8/9/93 */;
		else if (FLAG(s) == BOT)
		    FLAG(s) = NON;
		if (FLAG(t) == CLEAR)
		    FLAG(t) = BOT;
		/* TP is an inert transition starting in s */
		DAddHead(INERT(s), TP);
	    }
	} else {
	    /*
	     * label is not TAU, so TP is a non-inert transition, s and t
	     * become bottom states temporarily if they were not already in B0
	     */
	    if (FLAG(s) == CLEAR)
		FLAG(s) = BOT;
	    if (FLAG(t) == CLEAR)
		FLAG(t) = BOT;
	    /* TP is not an inert transition */
	    AddSorted(NONINERT(B0), TP);
	}
    }				/* for all transitions */
    if (VERBOSE)
	fprintf(stderr, "done\n");
    return (B0);
}				/* extract_graph */
