#include <stdio.h>
#include "psf_prototype.h"
#include "datatypes.h"
#include "dlist.h"
#include "global.h"
#include "print.h"

#ifdef SEARCH
#include "readgraph.h"
#endif

/**********************************************************************\
   Print a single state. If the state is marked , then an asterisk
   appears before it.
\**********************************************************************/
void print_state(fp, S)
    FILE *fp;
    stateP S;
{
    fprintf(fp, "%c%s", FLAG(S) == MARK ? '*' : ' ', ACTUALNAME(S));
}				/* print_state */

/**********************************************************************\
   Print a list of states. If SEARCH is defined, then it prints the
   value of the flag of the state.
\**********************************************************************/
void print_states(L)
    ListP L;
{
    stateP S;
    int i = 0;

    fprintf(fd_out, "{ ");
    for (S = DFirst(L); DValid(S); S = DNext(S)) {
	if (i++)
	    fprintf(fd_out, ",\n  ");

#ifdef SEARCH
	switch FLAG(S) {
	    case CLEAR: printf("CLEAR "); break;
	    case MARK: printf("MARK "); break;
	    case NON: printf("NON "); break;
	    case BOT: printf("BOT "); break;
	    case ONSTACK: printf("ONSTACK "); break;
	    default: break;
	}
#endif

	print_state(fd_out, S);
	print_transitions(INERT(S));
    }
    fprintf(fd_out, "\n}\n");
}				/* print_states */

/**********************************************************************\
   Print a single transition.
\**********************************************************************/
void print_transition(fp, T)
    FILE *fp;
    transitionP T;
{
    print_state(fp, SOURCE(T));
    fprintf(fp, " --%s-->", ACTUALLABEL(T));
    print_state(fp, TARGET(T));
}				/* print_transition */

/**********************************************************************\
   Print a list of transitions.
\**********************************************************************/
void print_transitions(L)
    ListP L;
{
    transitionP T;
    int i = 0;

    fprintf(fd_out, "{ ");
    for (T = DFirst(L); DValid(T); T = DNext(T)) {
	if (i++)
	    fprintf(fd_out, ",\n  ");
	print_transition(fd_out, T);
    }
    fprintf(fd_out, "}");
}				/* print_transitions */

/**********************************************************************\
   Print a single block in an informative way.
   The bottom states and non-bottom states are listed separately,
   and for each non-bottom state the list of outgoing inert
   transitions is shown. Also the list of non-inert transitions
   is shown.
\**********************************************************************/
void print_block(B)
    blockP B;
{
    fprintf(fd_out, "BLOCK B%d:\n", BLOCKNUM(B));
    fprintf(fd_out, "bottom states:\n");
    print_states(BOTTOM(B));
    fprintf(fd_out, "non-bottom states:\n");
    print_states(NONBOTTOM(B));
    fprintf(fd_out, "non-inert transitions:\n");
    print_transitions(NONINERT(B));
    fprintf(fd_out, "\nEND BLOCK B%d\n\n\n", BLOCKNUM(B));
}				/* print_block */

/**********************************************************************\
   Print the partition which L points to in an informative way.
\**********************************************************************/
void print_blocks(L)
    ListP L;
{
    blockP B;
    int i;

    for (B = DFirst(L), i = 0; DValid(B); B = DNext(B), i++)
	print_block(B);
    if (VERBOSE)
	fprintf(stderr, "%d equivalence classes\n", i);
}				/* print_blocks */

/**********************************************************************\
   Print a block as a set of states.
\**********************************************************************/
void list_states(fp, B)
    FILE *fp;
    blockP B;
{
    stateP S;
    int i = 0;

    fprintf(fp, "{");
    for (S = DFirst(NONBOTTOM(B)); DValid(S); S = DNext(S)) {
	if (i++)
	    fprintf(fp, ",\n");
	print_state(fp, S);
    }				/* print non-bottom states */
    for (S = DFirst(BOTTOM(B)); DValid(S); S = DNext(S)) {
	if (i++)
	    fprintf(fp, ",\n");
	print_state(fp, S);
    }				/* print bottom states */
    fprintf(fp, " }");
}				/* list_states */

/**********************************************************************\
   Print the partition which P points to as a collection of sets of
   states, that is, only the states in each block are shown.
   ex: { B1 = { s1, s2 }, B2 = { s3, s4, s5 } }
\**********************************************************************/
void print_partition(P)
    ListP P;
{
    blockP B;
    int i = 0;

    fprintf(fd_out, "{ ");
    for (B = DFirst(P); DValid(B); B = DNext(B)) {
	if (i++)
	    fprintf(fd_out, ",\n");
	fprintf(fd_out, "B%d = ", BLOCKNUM(B));
	list_states(fd_out, B);
    }				/* for each block */
    fprintf(fd_out, " }\n");
    if (VERBOSE)
	fprintf(stderr, "%d equivalence classes\n", i);
}				/* print_partition */

/**********************************************************************\
   Print the partition which P points to as a list of equivalence
   classes. For example:
   B1: s1 s2
   B2: s3 s4 s5
\**********************************************************************/
void print_equivs(P)
    ListP P;
{
    blockP B;
    stateP S;
    int i;

    for (B = DFirst(P), i = 0; DValid(B); B = DNext(B), i++) {
	fprintf(fd_out, "B%d:", BLOCKNUM(B));
	for (S = DFirst(NONBOTTOM(B)); DValid(S); S = DNext(S))
	    print_state(fd_out, S);
	for (S = DFirst(BOTTOM(B)); DValid(S); S = DNext(S))
	    print_state(fd_out, S);
	fprintf(fd_out, "\n");
    }				/* for each block */
    if (VERBOSE)
	fprintf(stderr, "%d equivalence classes\n", i);
}				/* print_equivs */
