#include <stdio.h>
#include <string.h>
#include "psf_prototype.h"
#include "psf_standards.h"
#include "datatypes.h"
#include "bbdf.h"
#include "bisimilar.h"
#include "dlist.h"
#include "formula.h"
#include "global.h"
#include "interface.h"
#include "main.h"
#include "print.h"
#include "readgraph.h"
#include "reduce.h"
#include "tree.h"
#include "treeprint.h"
#include "version.h"

#define LENGTH_OF_LONGEST_STATE_NAME_EVER	32

static bool prompting;
static char **states;
static int nstates;		/* number of occupied slots in states[] */

static void version()
{
    fprintf(stderr, "equiv version %s, last compiled %s\n", VERSION, __DATE__);
}				/* version */

static void usage(exit_status)
    int exit_status;
{
    fprintf(stderr, "usage:\t%s [-bcdhirvxBFITY] [-o outputfile] ", progname);
    fprintf(stderr, "[-t s1,s2]... [[-f] inputfile]\n\n");
    fprintf(stderr, "options:\n");
    fprintf(stderr, "\t-b\t print block numbers of states being compared\n");
    fprintf(stderr, "\t-c\t show equivalence classes\n");
    fprintf(stderr, "\t-d\t generate distinguishing formulae\n");
    fprintf(stderr, "\t-f file\t use 'file' as input file\n");
    fprintf(stderr, "\t-h\t print this help and quit immediately\n");
    fprintf(stderr, "\t-i\t interactive equivalence testing\n");
    fprintf(stderr, "\t-o file\t use 'file' as output file\n");
    fprintf(stderr, "\t-r\t print reduced automaton\n");
    fprintf(stderr, "\t-t s1,s2 test equivalence between s1 and s2. ");
    			fprintf(stderr, "(may be repeated)\n");
    fprintf(stderr, "\t-v\t verbose mode\n");
    fprintf(stderr, "\t-x\t print blocks informatively\n");
    fprintf(stderr, "\t-B\t print block tree\n");
    fprintf(stderr, "\t-F\t print final partition\n");
    fprintf(stderr, "\t-I\t print initial partition\n");
    fprintf(stderr, "\t-T\t show transitions when converting input\n");
    fprintf(stderr, "\t-Y\t symmetrical distinguishing formulae\n");
    exit(exit_status);
}				/* usage */

static void parse_args(argc, argv)
    int argc;
    char **argv;
{   
    int opt;
    int max = 0;		/* current length of states[] */
    char *s1, *s2;

    while ((opt = getopt(argc, argv, "bcdf:hio:rt:vxBFINTY")) != EOF)
	switch (opt) {
	case 'b':
	    BLOCKNUMBERS = TRUE;
	    break;
	case 'c':
	    EQUIVCLASSES = TRUE;
	    break;
	case 'd':
	    DISTFORM = TRUE;
	    break;
	case 'f':
	    input_file = optarg;
	    break;
	case 'h':
	    usage(EXIT_SUCCESS);	/* doesn't return */
	case 'i':
	    INTERACTIVE = TRUE;
	    break;
	case 'o':
	    if ((fd_out = fopen(optarg, "w")) == NULL) {
		(void) fprintf(stderr, "%s: can't create file ", progname);
		perror(optarg);
		exit(EXIT_IO_ERR);
	    }
	    break;
	case 'r':
	    REDUCED = TRUE;
	    break;
	case 't':
	    if (states == NULL)
		states = PSF_NMALLOC(char *, max);
	    s1 = psf_strdup(optarg);
	    s2 = strchr(s1, ',');
	    if (!s2) {
		(void) fprintf(stderr,
			"%s: -t option requires two states, ", progname);
		(void) fprintf(stderr,
			"separated by a comma\n");
		exit(EXIT_CMD_LINE_ERR);
	    }
	    if (!s2[1]) {
		(void) fprintf(stderr,
			"%s: inserting a space after the comma in the ",
			progname);
		(void) fprintf(stderr, "argument for -t is not allowed\n");
		exit(EXIT_CMD_LINE_ERR);
	    }
	    *s2++ = '\0';

#ifdef DEBUG
	    fprintf(stderr, "state %i: \"%s\"\n", nstates, s1);
	    fprintf(stderr, "state %i: \"%s\"\n", nstates + 1, s2);
#endif

	    if (nstates + 2 > max) {
		max *= 2;
		states = PSF_REALLOC(states, char *, max);
	    }
	    states[nstates++] = s1;
	    states[nstates++] = s2;
	    break;
	case 'v':
	    VERBOSE = TRUE;
	    version();
	    break;
	case 'x':
	    INFORMATIVE = TRUE;
	    break;
	case 'B':
	    BLOCKTREE = TRUE;
	    break;
	case 'F':
	    FINAL = TRUE;
	    break;
	case 'I':
	    INITIAL = TRUE;
	    break;
	case 'T':
	    SHOWINPUT = TRUE;
	    break;
	case 'Y':
	    SYMMETRICAL = TRUE;
	    break;
	default:
	    usage(EXIT_CMD_LINE_ERR);	/* doesn't return */
	}

    switch (argc - optind) {
    case 1:
	input_file = argv[optind];
	break;
    case 0:
	input_file = NULL;
	break;
    default:
	usage(EXIT_CMD_LINE_ERR);	/* doesn't return */
    }

    if (INTERACTIVE && !input_file) {
	(void)fprintf(stderr,
		"%s: can't do interactive testing when reading from stdin\n",
		progname);
	exit(EXIT_CMD_LINE_ERR);
    }
    if (!FINAL && !REDUCED && !states && input_file)
	INTERACTIVE = TRUE;
}

static bool get_state_name(prompt, buf)
    char *prompt;
    char buf[LENGTH_OF_LONGEST_STATE_NAME_EVER];
{
    int l;

    if (prompting) {
	printf(prompt);
	fflush(stdout);
    }
    (void) fgets(buf, LENGTH_OF_LONGEST_STATE_NAME_EVER, stdin);
    if (*buf == '\n')
	return FALSE;

    l = strlen(buf) - 1;
    if (buf[l] == '\n') {
	buf[l] = '\0';
	return TRUE;
    }

    (void)fprintf(stderr,
	"%s: sorry, can't handle state names longer than %d characters\n",
	progname, LENGTH_OF_LONGEST_STATE_NAME_EVER - 2);
    (void)fprintf(stderr,
	"%s: if this causes problems, write to psfbugs@win.tue.nl\n", progname);
    exit(EXIT_MEM_ERR);

    return FALSE;		/* to shut up lint */
}

int main(argc, argv)
    int argc;
    char *argv[];
{
    blockP transition_graph;
    ListP Partition;
    int i;
    char state1[LENGTH_OF_LONGEST_STATE_NAME_EVER];
    char state2[LENGTH_OF_LONGEST_STATE_NAME_EVER];

    init_globals(argv);
    parse_args(argc, argv);
    (void) InitInterface();	/* read input file */
    transition_graph = extract_graph();	/* convert input to our format */

    if (VERBOSE) {
	fprintf(stderr, "%d states, %d transitions\n",
		NumberOfStates, NumberOfTransitions);
	fprintf(stderr, "S = ");
	list_states(stderr, transition_graph);
	fprintf(stderr, "\n");
    }				/* print the set of states */

    /* partition the graph into sets of branching bimilar states */
    Partition = bbdf(transition_graph);

    if (FINAL) {
	fprintf(fd_out, "\nThe final partition is:\n=======================\n");
	if (INFORMATIVE)
	    print_blocks(Partition);
	else
	    print_partition(Partition);
    }				/* print the final partition */

    if (EQUIVCLASSES) {
	fprintf(fd_out, "\nEquivalence classes:\n====================\n");
	print_equivs(Partition);
    }				/* print equivalence classes */

    if (REDUCED) {
	fprintf(fd_out, "\nReduced automaton:\n==================\n");
	print_block(ReducedAutomaton(Partition));
    }				/* Reduced Automaton desired */

    if (DISTFORM && BLOCKTREE) {
	fprintf(fd_out, "\nDepth-first traversal of the block-tree:\n");
	fprintf(fd_out, "========================================");
	df_treeprint(root);
    }				/* print the block-tree */

    for (i = 0; i < nstates - 1; i+=2)
	TestEquivalence(Partition, states[i], states[i+1]);

    if (INTERACTIVE) {
	prompting = isatty(fileno(stdin));
	for (;;) {
	    if(!get_state_name("\nEnter state #1 (none to exit): ", state1))
		break;
	    if(!get_state_name("Enter state #2 (none to exit): ", state2))
		break;
	    TestEquivalence(Partition, state1, state2);
	}			/* forever */
    }				/* if INTERACTIVE */

    return 0;
}				/* main */
