/**********************************************************************\

   Module for converting input to desired format.

   These functions need to be changed for a different input format.

\**********************************************************************/

#include <stdio.h>
#include <string.h>
#include "env_variables.h"
#include "psf_prototype.h"
#include "psf_fopen.h"
#include "psf_malloc.h"
#include "psf_standards.h"

#include "datatypes.h"
#include "dlist.h"
#include "tiltype.h"
#include "fieldex.h"
#include "global.h"
#include "interface.h"
#include "readtil.h"

static int current, max_transitions;
static struct module *mod;
static FILE *fp;

/**********************************************************************\
   Convert the unique number representing label a to its actual name,
   a string. The interface must be initialized.
\**********************************************************************/
char *LabelToString(a)
    int a;
{
    char *str;

    if (a == TAU)
	str = STR_SKIP;
    else
	str = field_extract("n", mod->atm[a].ff, ATM, a);
    return (psf_strdup(str));
}				/* LabelToString */

/**********************************************************************\
   Convert the unique number representing state s to its actual name,
   a string. The interface must be initialized.
\**********************************************************************/
char *NameToString(s)
    int s;
{
    if (mod->entries_table[PRO] < s) {
	fprintf(stderr, "%s: no such process", progname);
	exit(EXIT_SYNTAX_ERR);
    }
    return (psf_strdup(field_extract("n", mod->pro[s].ff, PRO, s)));
}				/* NameToString */

/**********************************************************************\
   Extract the unique number representing the source state from the
   current transition. The interface must be initialized.
\**********************************************************************/
int ExtractSource()
{
    return (mod->def[current].ae_t.ind.key);
}				/* ExtractSource */

/**********************************************************************\
   Extract the unique number representing the target state from the
   current transition. The interface must be initialized.
\**********************************************************************/
int ExtractTarget()
{
    return (mod->def[current].p_expr.proc_expr.pe3.pe[1].proc_expr.pe2.ind.key);
}				/* ExtractTarget */

/**********************************************************************\
   Extract the unique number representing the label from the
   current transition. The interface must be initialized.
\**********************************************************************/
int ExtractLabel()
{
    struct process_expr *pe;

    pe = &mod->def[current].p_expr.proc_expr.pe3.pe[0];
    if (pe->fun == SKP)
	return (TAU);
    else if (pe->fun == AET)
	return (pe->proc_expr.pe2.ind.key);
    else {
	fprintf(stderr,
		"%s: file %s does not contain a Labelled Transition System\n",
		progname, filename);
	fprintf(stderr, "%s: try using the trans tool first\n", progname);
	exit(EXIT_SYNTAX_ERR);
    }

    return 0;			/* to shut up lint */
}				/* ExtractLabel */

/**********************************************************************\
   Return TRUE if there are more transitions to be converted from the input.
   Exit with error message if the next process definition does not define
   a transition.
\**********************************************************************/
bool MoreTransitions()
{
    struct process_expr p;

    if (current++ < max_transitions) {
	p = mod->def[current].p_expr;
	if (!(
	      p.fun == SEQ &&
	      p.proc_expr.pe3.a == 2 &&
	      p.proc_expr.pe3.pe[1].fun == AET &&
	      p.proc_expr.pe3.pe[1].proc_expr.pe2.ind.table == PRO &&
	      (
	       (p.proc_expr.pe3.pe[0].fun == AET &&
		p.proc_expr.pe3.pe[0].proc_expr.pe2.ind.table == ATM
		) ||
	       p.proc_expr.pe3.pe[0].fun == SKP ||
	       p.proc_expr.pe3.pe[0].fun == DLK
	       )
	      )) {
	      fprintf(stderr,
		  "%s: file %s does not contain a Labelled Transition System\n",
		  progname, filename);
	      fprintf(stderr,
		  "%s: process definition number %d ", progname, current);
	      fprintf(stderr, "is not a simple transtion\n");
	      fprintf(stderr, "%s: try using the trans tool first\n", progname);
	      exit(EXIT_SYNTAX_ERR);
	}
	if (p.proc_expr.pe3.pe[0].fun == DLK) {
	    fprintf(stderr, "%s: delta not yet supported\n", progname);
	    fprintf(stderr, "%s: try using the trans tool first\n", progname);
	    exit(EXIT_SYNTAX_ERR);
	}
	return TRUE;
    } else
	return FALSE;
}				/* MoreTransitions */

/**********************************************************************\
   Read TIL module from input-file and initialize the interface.
\**********************************************************************/
void InitInterface()
{
    static suffix suffixes[] = {
	{LTSSUFFIX, LTSSUFFIX_DEFAULT},
	{TILSUFFIX, TILSUFFIX_DEFAULT},
	{NULL, NULL}
    };
    psf_file pf;

    if (input_file == NULL) {
	fp = stdin;
	filename = "stdin";
    } else {
	pf = psf_fopen(input_file, suffixes);
	fp = pf.fp;
	filename = pf.name;
    }
    mod = PSF_MALLOC(struct module);
    if (VERBOSE)
	printf("reading TIL module from file %s...\n",filename);
    if (read_module(mod, fp)) {
	fprintf(stderr, "%s: error while reading TIL module.\n", progname);
	exit(EXIT_SYNTAX_ERR);
    }
    fclose(fp);
    if (VERBOSE)
	fprintf(stderr, "done\n");
    if (mod->entries_table[PRO] == 0) {
	fprintf(stderr, "%s: no processes in file \"%s\"\n",
		       progname, filename);
	exit(EXIT_SUCCESS);
    }
    max_transitions = mod->entries_table[DEF];
    current = 0;
}				/* InitInterface */
