#include <stdio.h>
#include "psf_prototype.h"
#include "datatypes.h"
#include "dlist.h"
#include "formula.h"
#include "global.h"
#include "print.h"
#include "tree.h"

/**********************************************************************\
   Returns the Lowest Common Ancestor of nodes N1 and N2. First trace
   back to the parent of the lower of the two nodes until the nodes
   are at the same height. Then step up until the parents are equal.
   If BLOCK(N2) is a subset of the right child of the LCA then
   not_a_subset is set to FALSE, otherwise to TRUE.
\**********************************************************************/
treeP LCA(N1, N2, not_a_subset)
    treeP N1, N2;
    bool *not_a_subset;
{
    treeP P1, P2, save;
    int L1, L2;

    save = N2;
    *not_a_subset = TRUE;
    P1 = PARENT(N1);
    P2 = PARENT(N2);
    L1 = LEVEL(P1);
    L2 = LEVEL(P2);
    while (L1 < LEVEL(P2)) {
	save = P2;
	P2 = PARENT(P2);
    }
    while (L2 < LEVEL(P1))
	P1 = PARENT(P1);
    while (P1 != P2) {
	P1 = PARENT(P1);
	save = P2;
	P2 = PARENT(P2);
    }
    if (RIGHT(P2) == save)
	*not_a_subset = FALSE;
    return (P1);
}				/* LCA */

/**********************************************************************\
   Print a part of the Distinguishing Formula.
\**********************************************************************/
static void PrintPhi(P, first_half)
    treeP P;
    bool first_half;
{
    int i = 0;
    elementP C;
    ListP L = first_half ? ReachableTau(P) : Reachable(P);

    if (DEmptyList(L))
	fprintf(fd_out, "t");
    else {
	fprintf(fd_out, "(");
	for (C = DFirst(L); DValid(C); C = DNext(C)) {
	    if (i++)
		fprintf(fd_out, " and ");
	    PrintFormula(first_half ? P : NODE(SPLITTER(P)), (treeP) DATA(C));
	}
	fprintf(fd_out, ")");
    }
}				/* PrintPhi */

/**********************************************************************\
   Prints the Distinguishing Formula that distinguishes the block at
   node N1 from the block at node N2. Method used: see H. Korver,
   "Computing Distinguishing Formulas for Branching Bisimulation",
   PRG Report P9103.
\**********************************************************************/
void PrintFormula(N1, N2)
    treeP N1, N2;
{
    treeP P;
    bool not_a_subset;

    P = LCA(N1, N2, &not_a_subset);
    if (not_a_subset)
	fprintf(fd_out, "not(");
    PrintPhi(P, TRUE);
    fprintf(fd_out, "<%s>", SPLITLABEL(P));
    if (identical(STR_SKIP, SPLITLABEL(P))) {
	fprintf(fd_out, "(");
	PrintPhi(P, FALSE);
	fprintf(fd_out, " and ");
	PrintFormula(NODE(SPLITTER(P)), RIGHT(P));
	fprintf(fd_out, ")");
    } else
	PrintPhi(P, FALSE);
    if (not_a_subset)
	fprintf(fd_out, ")");
}				/* PrintFormula */
