#include <string.h>
#include "psf_prototype.h"
#include "datatypes.h"
#include "dlist.h"
#include "global.h"
#include "psf_malloc.h"

static blockP TheNewBlock;

/* memory-allocation for a state */
#define state_alloc() (stateP) PSF_MALLOC(struct state)
#define st_data_alloc() (state_ptr) PSF_MALLOC(struct state_data)

/* memory-allocation for a transition */
#define trans_alloc() (transitionP)PSF_MALLOC(struct transition)

/* memory-allocation for a block */
#define block_alloc() (blockP) PSF_MALLOC(struct block)

/**********************************************************************\
   create a new state and initialize it.
\**********************************************************************/
stateP new_state()
{
    stateP ptr;

    ptr = state_alloc();
    ptr->Next = NULL;
    ptr->Prev = NULL;
    ptr->state = st_data_alloc();
    ptr->state->label = UNDEFINED;	/* initialize label */
    ptr->state->act_label = NULL;	/* initialize pointers */
    ptr->state->block = NULL;
    ptr->state->inert = list_alloc();
    DNewList(ptr->state->inert);
    ptr->state->flag = CLEAR;	/* reset flag */
    ptr->state->dfnumber = 0;
    ptr->state->lowlink = 0;
    return ptr;
}				/* new_state */

/**********************************************************************\
   create a new transition and initialize it.
\**********************************************************************/
transitionP new_transition()
{
    transitionP ptr;

    ptr = trans_alloc();
    ptr->label = UNDEFINED;	/* initialize label */
    ptr->act_label = NULL;	/* initialize pointers */
    ptr->source = NULL;
    ptr->target = NULL;
    ptr->Next = NULL;
    ptr->Prev = NULL;
    return ptr;
}				/* new_transition */

/**********************************************************************\
   create a new block and initialize it.
\**********************************************************************/
blockP new_block()
{
    blockP ptr;

    ptr = block_alloc();
    ptr->bottom = list_alloc();
    DNewList(ptr->bottom);	/* initialize pointers */
    ptr->non_bottom = list_alloc();
    DNewList(ptr->non_bottom);
    ptr->non_inert = list_alloc();
    DNewList(ptr->non_inert);
    ptr->Next = NULL;
    ptr->Prev = NULL;
    ptr->flag = CLEAR;		/* reset flag */
    ptr->node = NULL;
    ptr->label = UNDEFINED;
    return ptr;
}				/* new_block */

/**********************************************************************\
   Add a pointer to element D to a list of pointers to element BL.
\**********************************************************************/
void AddPointer(BL, D)
    ListP BL;
    void *D;
{
    elementP ptr;

    ptr = (elementP) PSF_MALLOC(struct element);
    ptr->data = (void *) D;
    DAddHead(BL, ptr);
}				/* AddPointer */

/**********************************************************************\
   Delete a pointer to element D from a list of pointers to element
   BL, element D is returned.
\**********************************************************************/
void *DeletePointer(BL, ptr)
    ListP BL;
    elementP ptr;
{
    void *D;

    DDelete(BL, ptr);
    D = (void *) ptr->data;
    PSF_FREE(ptr);
    return D;
}				/* DeletePointer */

/**********************************************************************\
   Search list LP for a state with a certain label. Returns a pointer
   to it if it was found, NULL otherwise.
\**********************************************************************/
stateP match_state(LP, a)
    ListP LP;
    int a;
{
    stateP P;

    for (P = DFirst(LP); DValid(P); P = DNext(P))
	if (a == P->state->label)
	    return P;
    return NULL;
}				/* match_state */

/**********************************************************************\
   Search list LP for a state with a certain actual label. Returns a
   pointer to it if it was found, NULL otherwise.
\**********************************************************************/
stateP match_by_name(LP, a)
    ListP LP;
    char *a;
{
    stateP P;

    for (P = DFirst(LP); DValid(P); P = DNext(P))
	if (identical(a, P->state->act_label))
	    return P;
    return NULL;
}				/* match_by_name */

/**********************************************************************\
   Return a pointer to a newly allocated copy of block p.
\**********************************************************************/
blockP fresh_block_copy(p)
    blockP p;
{
    blockP BP = new_block();

    TheNewBlock = BP;
    if (!SYMMETRICAL) {
	BP->bottom = copy_states(p->bottom);
	BP->non_bottom = copy_states(p->non_bottom);
    }				/* if not symmetrical, copy states */
    BP->node = p->node;
    BP->label = p->label;
    return BP;
}				/* fresh_block_copy */

/**********************************************************************\
   Return a pointer to a newly allocated copy of state p, which
   contains a pointer to the contents of state p.
\**********************************************************************/
stateP fresh_state_copy(p)
    stateP p;
{
    stateP SP = new_state();

    SP->state = p->state;
    return SP;
}				/* fresh_state_copy */

/**********************************************************************\
   Copy a list of states, generating newly allocated states.
\**********************************************************************/
ListP copy_states(source)
    ListP source;
{
    ListP dest;
    stateP SP;

    dest = list_alloc();
    DNewList(dest);
    for (SP = DFirst(source); DValid(SP); SP = DNext(SP))
	DAddTail(dest, fresh_state_copy(SP));
    return dest;
}				/* copy_states */

/**********************************************************************\
   Returns TRUE if name1 is identical to name2, FALSE otherwise.
\**********************************************************************/
bool identical(name1, name2)
    char *name1, *name2;
{
    return strcmp(name1, name2) == 0;
}				/* identical */

/**********************************************************************\
   Convert a positive integer to a character string.
\**********************************************************************/
char *itoa(n)
    int n;
{
#define BUFSIZE	32		/* 32 characters to represent int */

    char s[BUFSIZE];
    int i = BUFSIZE;

    s[--i] = '\0';
    do {
	s[--i] = n % 10 + '0';
    } while ((n /= 10) > 0);
    return psf_strdup(s+i);
}				/* itoa */
