/**********************************************************************\

   Algorithm for Branching Bisimulation + Distinguishing Formulas

   Complexity of the algorithm is O(n*(n+m)) where n is the number of
   states and m is the number of transitions.

   Set DISTFORM to TRUE if you want distinguishing formulas to be
   generated. In this case the space comlexity increases to O(n*n).

\**********************************************************************/

#include <stdio.h>
#include "psf_prototype.h"
#include "psf_redefs.h"

#include "datatypes.h"
#include "bbdf.h"
#include "dlist.h"
#include "global.h"
#include "print.h"
#include "readgraph.h"
#include "search.h"
#include "tree.h"

#define REACH		2
#define NOTREACH	253

static ListP tobeprocessed, stable, BL;

/**********************************************************************\
   Return an initial partition of the set of states S, which is
   simply the set containing S itself, with the non-bottom states
   sorted and cycles of inert transitions deleted.
\**********************************************************************/
ListP initial_partition(BP)
    blockP BP;
{
    ListP LP = list_alloc();

    if (VERBOSE)
	fprintf(stderr,
	    "\nsorting states and deleting cycles of inert transitions\n");
    if (!DEmptyList(BOTTOM(BP)))
	sort_states(BP);
    if (VERBOSE)
	fprintf(stderr, "... done\n%d states, %d transitions\n",
			NumberOfStates, NumberOfTransitions);
    DNewList(LP);
    DAddHead(LP, BP);
    return (LP);
}				/* initial_partition */

/**********************************************************************\
   Return a partition Pf, such that the equivalence ~p on S induced
   by Pf is a branching bisimulation equivalence. If DISTFORM is set,
   it also generates the block tree that is needed for generating
   Distinguishing Formulas.
\**********************************************************************/
ListP bbdf(BP)
    blockP BP;
{
    blockP B, C, B1, B2, Splitter;
    transitionP T;
    elementP Bref;
    int a;
    int NotSameBottom, FoundSplitter;
    ListP R1, R2;

    tobeprocessed = initial_partition(BP);
    if (VERBOSE) {
	fprintf(stderr, "\nstarting partition refinement ... ");
    }
    if (INITIAL) {
	fprintf(fd_out,
		"\nThe initial partition is:\n=========================\n");
	if (INFORMATIVE)
	    print_block(BP);
	else
	    print_partition(tobeprocessed);
    }				/* if INITIAL */
    if (DISTFORM) {
	root = new_node();
	NODE(BP) = root;
	BLOCK(root) = BP;
    }				/* if DISTFORM */
    BL = list_alloc();
    stable = list_alloc();
    DNewList(stable);
    while (!DEmptyList(tobeprocessed)) {

#ifdef DEBUG
	printf("the list \"tobeprocessed\" is not empty\n");
	printf("**********************************");
	printf("*********************************\n");
	printf("*                       tobeprocessed:");
	printf("                            *\n");
	printf("**********************************");
	printf("*********************************\n");
	print_blocks(tobeprocessed);
	printf("**********************************");
	printf("*********************************\n");
	printf("*                           stable:");
	printf("                               *\n");
	printf("**********************************");
	printf("*********************************\n");
	print_blocks(stable);
#endif

	C = DFirst(tobeprocessed);

#ifdef DEBUG
	printf("--------------------------------------");
	printf("--------------------------------------\n");
	printf("C = B%d is next in \"tobeprocessed\"\n", BLOCKNUM(C));
	print_block(C);
	printf("--------------------------------------");
	printf("--------------------------------------\n");
#endif

	FoundSplitter = FALSE;
	T = DFirst(NONINERT(C));
	while (!FoundSplitter && DValid(T)) {
	    a = LABEL(T);

#ifdef DEBUG
	    printf("\n******************** begin loop with label %d ");
	    printf("*******************\n", a);
#endif

	    BL = Consider(T, a);

#ifdef DEBUG
	    printf("BL has been constructed\n");
#endif

	    while (!DEmptyList(BL)) {

#ifdef DEBUG
	printf("--------------------------------------");
	printf("--------------------------------------\n");
		printf("BL is not empty\n");
#endif

		NotSameBottom = FALSE;
		Bref = DFirst(BL);
		B = DeletePointer(BL, Bref);

#ifdef DEBUG
		printf("B = B%d is next in BL\n", BLOCKNUM(B));
		print_block(B);
#endif

		IN_BL(B) = FALSE;

#ifdef DEBUG
		printf("the pointer to B = B%d has been removed from BL\n",
			BLOCKNUM(B));
#endif

		if (!FoundSplitter) {
		    if (SplitterOf(B)) {

#ifdef DEBUG
			printf("\007C = B%d is a splitter of B = B%d w.r.t. ",
				BLOCKNUM(C), BLOCKNUM(B));
			printf("label %d!\n", a);
#endif

			FoundSplitter = TRUE;

#ifdef DEBUG
			printf("FoundSplitter is now TRUE\n");
			printf("ExtendMarking:\n");
#endif

			ExtendMarking(B);

#ifdef DEBUG
			printf("The marking has been extended\n");
#endif

			B1 = B;
			if (DISTFORM) {
			    R1 = reachable(B1, TAU);
			    if (a == TAU)
				R2 = R1;
			    else
				R2 = reachable(B1, a);
			    B = fresh_block_copy(B1);
			    BLOCKNUM(B1) = NumberOfBlocks++;
			    BLOCK(NODE(B)) = B;
			    Splitter = BLOCK(NODE(C));
			}	/* if DISTFORM */
			delete(B1);

#ifdef DEBUG
			printf("B = B%d has been removed from the partition\n",
				BLOCKNUM(B));
#endif

			B2 = new_block();
			BLOCKNUM(B2) = NumberOfBlocks++;

#ifdef DEBUG
			printf("B2 = B%d is a new block\n", BLOCKNUM(B2));
#endif

			SplitBottom(B1, B2);

#ifdef DEBUG
			printf("the bottom states have been split\n");
#endif

			SplitNonBottom(B1, B2);

#ifdef DEBUG
			printf("the non-bottom states have been split\n");
#endif

			SplitTransitions(B1, B2, &NotSameBottom);
			if (DISTFORM) {
			    insert_left(B, B1);
			    insert_right(B, B2);
			    LabelLeftArc(B, ACTUALLABEL(T), Splitter);
			    LabelRightArc(B, R1, R2);
			}	/* if DISTFORM */

#ifdef DEBUG
			printf("the transitions have been split\n");
#endif

			DAddHead(tobeprocessed, B1);

#ifdef DEBUG
			printf("B1 = B%d has been added to \"tobeprocessed\"\n",
				BLOCKNUM(B1));
#endif

			DAddHead(tobeprocessed, B2);

#ifdef DEBUG
			printf("B2 = B%d has been added to \"tobeprocessed\"\n",
				BLOCKNUM(B2));
#endif
		    }		/* C splitter of B */
		    else
			ResetFlags(B);
		}		 /* not FoundSplitter */
		else
		    ResetFlags(B);
	    }			/* while BL not empty */

#ifdef DEBUG
	    printf("BL is now empty, or some block has been split\n");
#endif

	    if (NotSameBottom) {
		/*
		 * in general stability is not inherited under refinement if
		 * the refinement has different bottom states
		 */

#ifdef DEBUG
		printf("***********************************");
		printf("**********************************\n");
		printf("*                  there is a new bottom state!");
		printf("                   *\n");
		printf("***********************************");
		printf("**********************************\n");
#endif

		DAppend(tobeprocessed, stable);
		DNewList(stable);

#ifdef DEBUG
		printf("so \"stable\" will be linked to \"tobeprocessed\",\n");
		printf("and then \"stable\" will be emptied\n");
#endif
	    }			/* if NotSameBottom */
	    while (!FoundSplitter && DValid(T) && LABEL(T) == a)
		T = DNext(T);	/* search next label */

#ifdef DEBUG
	    printf("******************* end of loop with label %d ", a);
	    printf("*******************\n");
#endif
	}			/* for all labels of non-inert transitions */
				/* ending in C */

#ifdef DEBUG
	printf("all labels have been dealt with\n");
#endif

	if (!FoundSplitter) {
	    /* the current partition is stable w.r.t. block C */

#ifdef DEBUG
	    printf("the current partition is stable w.r.t. block C = B%d\n",
			BLOCKNUM(C));
#endif

	    DDelete(tobeprocessed, C);
	    DAddTail(stable, C);
	}			/* move C to list stable */
    }				/* while tobeprocessed */

#ifdef DEBUG
    printf("the partition is stable\n");
#endif

    if (VERBOSE)
	fprintf(stderr, "\ndone\n");

    return (stable);
}				/* bbdf */

ListP Consider(NonInert, a)
/**********************************************************************\
   Returns a list of pointers to a block BL such that each block in BL
   contains an a-transition to the current splitter. Starts looking
   for transitions in NonInert, while the label is a. Source states of
   these transitions are marked MARK.
\**********************************************************************/
    transitionP NonInert;
    int a;
{
    transitionP T;
    stateP S;
    blockP R;

    DNewList(BL);

#ifdef DEBUG
    printf("\nConsider: BL has been created\n");
#endif

    for (T = NonInert; DValid(T) && LABEL(T) == a; T = DNext(T)) {
	S = SOURCE(T);

#ifdef DEBUG
	printf("\tnon-inert transition found: ");
	print_transition(stdout, T);
	printf("\n");
#endif

	R = BLOCKREF(S);

#ifdef DEBUG
	printf("\tthe block reference of the source is R = B%d\n", BLOCKNUM(R));
#endif

	if (!IN_BL(R)) {

#ifdef DEBUG
	    printf("\tR = B%d is not yet in BL ... ", BLOCKNUM(R));
#endif

	    IN_BL(R) = TRUE;
	    AddPointer(BL, R);

#ifdef DEBUG
	    printf("added.\n", R);
#endif
	}
	FLAG(S) = MARK;

#ifdef DEBUG
	printf("\t");
	print_state(stdout, S);
	printf(" has been marked\n");
#endif
    }
    return (BL);
}				/* Consider */

/**********************************************************************\
   Return TRUE if not all bottom states of B are marked MARK (i.e. the
   current splitter is a splitter of block B), FALSE otherwise.
\**********************************************************************/
int SplitterOf(B)
    blockP B;
{
    stateP S;

    for (S = DFirst(BOTTOM(B)); DValid(S); S = DNext(S))
	if(FLAG(S) == CLEAR)
	    return TRUE;
    return FALSE;
}				/* SplitterOf */

/**********************************************************************\
   Extend the marking to the non-bottom states, i.e. a non-bottom
   state is marked MARK if the target state of an outgoing inert
   transition is marked MARK.
\**********************************************************************/
void ExtendMarking(B)
    blockP B;
{
    stateP S;
    transitionP T;
    bool NotExtended;

    for (S = DFirst(NONBOTTOM(B)); DValid(S); S = DNext(S)) {
	NotExtended = FLAG(S) == CLEAR;
	if (NotExtended)
	    for (T = DFirst(INERT(S)); DValid(T); T = DNext(T))
		if (FLAG(TARGET(T))) {
		    NotExtended = FALSE;
		    FLAG(S) = MARK;	/* Mark S if a target of its outgoing */
		    			/* inert transitions is marked */
#ifdef DEBUG
		    printf("\t");
		    print_state(stdout, S);
		    printf(" has been marked\n");
#endif
		    break;
		}	
    }				/* for all non-bottom states S */
}				/* ExtendMarking */

/**********************************************************************\
   Split the bottom states of B1. Bottom states marked MARK remain in
   B1, the other bottom states become bottom states of B2.
\**********************************************************************/
void SplitBottom(B1, B2)
    blockP B1, B2;
{
    stateP S, next;
    ListP Bottom_B1 = BOTTOM(B1);

    for (S = DFirst(Bottom_B1); DValid(S); S = next) {
	next = DNext(S);
	if (FLAG(S) == CLEAR) {
	    DDelete(Bottom_B1, S);
	    BLOCKREF(S) = B2;
	    DAddTail(BOTTOM(B2), S);

#ifdef DEBUG
	    printf("\t");
	    print_state(stdout, S);
	    printf(" has been moved from Bottom(B1 = B%d)", BLOCKNUM(B1));
	    printf(" to Bottom(B2 = B%d)\n", BLOCKNUM(B2));
#endif
	} else {
	    FLAG(S) = CLEAR;

#ifdef DEBUG
	    printf("\tflag of ");
	    print_state(stdout, S);
	    printf(" has been reset\n");
#endif
	}
    }
}				/* SplitBottom */

/**********************************************************************\
   Split the non-bottom states of B1. If a non-bottom state is not
   marked, it becomes a bottom state of B2.
\**********************************************************************/
void SplitNonBottom(B1, B2)
    blockP B1, B2;
{
    stateP S, next;
    ListP NonBottom_B1 = NONBOTTOM(B1);

    for (S = DFirst(NonBottom_B1); DValid(S); S = next) {
	next = DNext(S);
	if (FLAG(S) == CLEAR) {
	    DDelete(NonBottom_B1, S);
	    BLOCKREF(S) = B2;
	    DAddTail(NONBOTTOM(B2), S);

#ifdef DEBUG
	    printf("\t");
	    print_state(stdout, S);
	    printf(" has been moved from NonBottom(B1 = B%d)", BLOCKNUM(B1));
	    printf(" to NonBottom(B2 = B%d)\n", BLOCKNUM(B2));
#endif
	} else {
	    FLAG(S) = CLEAR;

#ifdef DEBUG
	    printf("\tflag of ");
	    print_state(stdout, S);
	    printf(" has been reset\n");
#endif
	}
    }
}				/* SplitNonBottom */

/**********************************************************************\
   Distribute the transitions of B1 over B1 and B2, not disturbing the
   ordering of the transitions. if a non-bottom state becomes a bottom
   state because there are no inert transitions left, then
   NotSameBottom is TRUE, otherwise FALSE.
\**********************************************************************/
void SplitTransitions(B1, B2, NotSameBottom)
    blockP B1, B2;
    bool *NotSameBottom;
{
    transitionP T, nextT;
    stateP S, nextS;
    ListP NonInert_B1 = NONINERT(B1);
    ListP NonInert_B2 = NONINERT(B2);
    ListP NonBottom_B1 = NONBOTTOM(B1);
    ListP Inert_S;

    for (T = DFirst(NonInert_B1); DValid(T); T = nextT) {
	nextT = DNext(T);
	if (BLOCKREF(TARGET(T)) == B2) {
	    DDelete(NonInert_B1, T);
	    DAddTail(NonInert_B2, T);

#ifdef DEBUG
	    printf("\t");
	    print_transition(stdout, T);
	    printf(" has been moved from NonInert(B1 = B%d)", BLOCKNUM(B1));
	    printf(" to NonInert(B2 = B%d)\n", BLOCKNUM(B2));
#endif
	}
    }
    for (S = DFirst(NonBottom_B1); DValid(S); S = nextS) {
	nextS = DNext(S);
	Inert_S = INERT(S);
	for (T = DFirst(Inert_S); DValid(T); T = nextT) {
	    nextT = DNext(T);
	    if (BLOCKREF(TARGET(T)) == B2) {
		DDelete(Inert_S, T);
		DAddHead(NonInert_B2, T);

#ifdef DEBUG
		printf("\t");
		print_transition(stdout, T);
		printf(" has been moved from Inert(");
		print_state(stdout, S);
		printf(") to NonInert(B2 = B%d)\n", BLOCKNUM(B2));
#endif
	    }
	}
	if (DEmptyList(Inert_S)) {
	    *NotSameBottom = TRUE;

#ifdef DEBUG
	    printf("\007NotSameBottom is now TRUE!\n");
#endif

	    DDelete(NonBottom_B1, S);
	    DAddTail(BOTTOM(B1), S);

#ifdef DEBUG
	    printf("\t");
	    print_state(stdout, S);
	    printf(" has been moved from NonBottom(B1 = B%d)", BLOCKNUM(B1));
	    printf(" to Bottom(B1 = B%d)\n", BLOCKNUM(B1));
#endif
	}
    }
}

/**********************************************************************\
   Reset the flags of the states of B.
\**********************************************************************/
void ResetFlags(B)
    blockP B;
{
    stateP S;

#ifdef DEBUG
    printf("the flags of B = B%d are going to be reset ... ", BLOCKNUM(B));
#endif

    for (S = DFirst(BOTTOM(B)); DValid(S); S = DNext(S))
	FLAG(S) = CLEAR;
    for (S = DFirst(NONBOTTOM(B)); DValid(S); S = DNext(S))
	FLAG(S) = CLEAR;

#ifdef DEBUG
    printf("done\n");
#endif
}

/**********************************************************************\
   Return a list of pointers to node. Each node in the list points
   to a block which is reachable from a state in B2 via a non-inert
   a-transition.
\**********************************************************************/
ListP reachable(B2, a)
    blockP B2;
    int a;
{
    ListP R2 = list_alloc();
    blockP B;
    elementP Ref;

    DNewList(R2);
    for (B = DFirst(tobeprocessed); DValid(B); B = DNext(B))
	decide_reachable(B, B2, a, R2);
    for (B = DFirst(stable); DValid(B); B = DNext(B))
	decide_reachable(B, B2, a, R2);
    for (Ref = DFirst(R2); DValid(Ref); Ref = DNext(Ref)) {
	B = BLOCK((treeP) DATA(Ref));
	IN_BL(B) &= NOTREACH;
    }				/* reset flags */
    return (R2);
}				/* reachable */

/**********************************************************************\
   Decide if block TO is reachable in the parent partition of FROM,
   from a state in FROM via a non-inert a-transition. If it is, a
   pointer to TO is added to R2.
\**********************************************************************/
void decide_reachable(TO, FROM, a, R2)
    blockP TO, FROM;
    int a;
    ListP R2;
{
    transitionP T;
    blockP C;
    int NotAdded = TRUE;

    for (T = DFirst(NONINERT(TO)); DValid(T) && LABEL(T) < a; T = DNext(T));
    while (NotAdded && DValid(T) && LABEL(T) == a) {
	if (FLAG(SOURCE(T)) == CLEAR) {
	    C = BLOCKREF(SOURCE(T));
	    if (C == FROM && !(IN_BL(TO) & REACH)) {
		if (C != TO || FLAG(TARGET(T)) != CLEAR) {
		    IN_BL(TO) += REACH;
		    AddPointer(R2, NODE(TO));
		    NotAdded = FALSE;
		}		/* if not looking at block TO */
	    }			/* if not already added */
	}			/* disregard if source state is in B1 (the left
				 * child) */
	T = DNext(T);
    }				/* while */
}				/* decide_reachable */
