#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#include "psf_prototype.h"
#include "psf_standards.h"
#include "env_variables.h"
#include "psf_malloc.h"
#include "psf_fopen.h"

#include "typedef.h"
#include "tiltype.h"
#include "readtil.h"
#include "main.h"
#include "check_lts.h"
#include "version.h"
#include "ident.h"
#include "initial_block.h"
#include "make_transitions.h"
#include "partition.h"
#include "results.h"

#define TU (float) 1000		/* time unit factor millisec -> sec */
#define PC (float) 100		/* decimal to percentage */
ident skip;
ident tick;
state *delta;
state *final;
state_set ss;

transition *trans;
int nr_trans = 0;
char *filename;			/* name of the current file */
char *progname;
bool LONG_NAMES;
bool VERBOSE = FALSE;
bool VERBOSE_CHECKS = FALSE;
bool VERBOSE_MAKE = FALSE;
bool VERBOSE_CYCLES = FALSE;
bool VERBOSE_STATES = FALSE;
bool VERBOSE_PARTITION = FALSE;
bool UNIQUE = FALSE;
bool OUTPUT = FALSE;
int OUTPUT_LANG = 0;
char *VERBOSE_HMLU = (char *) NULL;
bool timing = FALSE;
bool result = TRUE;
char **file_list;

extern long clock();
static void version()
{
    (void) fprintf(stderr, "bb version %s, last compiled %s\n",
		   VERSION, __DATE__);
}				/* version */

static void usage(exit_status)
    int exit_status;
{
    (void) fprintf(stderr, "usage:\t%s [-cdfhmrostuvV][-b state,state...][til-file ...]\n", progname);
    (void) fprintf(stderr, "%s%s%s%s%s%s%s%s%s%s%s%s",
		   "This program only notices processes which are\n",
		   "in the definitions section of the til module.\n",
		   "The definitions should be of the following form:\n",
		   "\tp = a . q \ta an atom or skip\n",
		   "\tp = q     \tq a process or delta\n",
		   "\tp = a\n",
		   "Transitions of the form p = a, are interprated as\n",
		   "\tp = a . FINAL_STATE.\n\n",
		   "It uses the following internal hidden transition:\n",
		   "\tFINAL_STATE = tick . delta\n",
		   "to distinguish between FINAL_STATE and delta\n",
	       "User supplied tick's are different from the internal one.\n\n"
	);

    (void) fprintf(stderr, "%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s",
		   "Switches:\n",
	"\t-b\t if states are not bisimilar compute distinguising formula,\n",
	"\t\t or in combination with -d or -o, print original input graphs\n",
	       "\t\t containing those states in the corresponding language\n",
		   "\t-c\t print process expressions while checking them.\n",
		   "\t-d\t print final partition as dot\n",
		   "\t-f\t print final partition\n",
		   "\t-h\t print this help and quit immediately\n",
	 "\t-l\t Use set of states in block as block name instead of block\n",
		   "\t\t number.\n",
		   "\t-m\t print transitions found\n",
		   "\t-o\t print final partition as til\n",
		   "\t-r\t no interactive questioning\n",
		   "\t-s\t print equivalence classes and renamings\n",
		   "\t-t\t time partitioning\n",
		   "\t-u\t postfix state names with filename\n",
		   "\t-v\t verbose mode\n",
		   "\t-V\t very verbose mode\n");
    exit(exit_status);
}				/* usage */

static void parse_args(argc, argv)
    int argc;
    char **argv;
{
    int opt;
    extern char *optarg;

    while ((opt = getopt(argc, argv, "b:cdfhlmorstuvV")) != EOF) {
	switch (opt) {
	case 'c':
	    VERBOSE_CHECKS = TRUE;
	    break;
	case 'h':
	    usage(EXIT_SUCCESS);/* doesn't return */
	case 'l':
	    LONG_NAMES = TRUE;
	    break;
	case 'm':
	    VERBOSE_MAKE = TRUE;
	    break;
	case 'r':
	    result = FALSE;
	    break;
	case 's':
	    VERBOSE_STATES = TRUE;
	    break;
	case 'f':
	    VERBOSE_PARTITION = TRUE;
	    break;
	case 'o':
	    OUTPUT = TRUE;
	    result = FALSE;
	    OUTPUT_LANG = TIL;
	    break;
	case 'd':
	    OUTPUT = TRUE;
	    result = FALSE;
	    OUTPUT_LANG = DOT;
	    break;
	case 'b':
	    VERBOSE_HMLU = optarg;
	    break;
	case 't':
	    timing = TRUE;
	    break;
	case 'u':
	    UNIQUE = TRUE;
	    break;
	case 'v':
	    VERBOSE = TRUE;
	    version();
	    break;
	case 'V':
	    VERBOSE = TRUE;
	    VERBOSE_CHECKS = TRUE;
	    VERBOSE_MAKE = TRUE;
	    VERBOSE_CYCLES = TRUE;
	    timing = TRUE;
	    version();
	    break;
	default:
	    usage(EXIT_CMD_LINE_ERR);	/* doesn't return */
	}
    }
    if (argc - optind > 0)
	file_list = &argv[optind];
    else if (argc == optind)
	file_list = (char **) NULL;
    else
	usage(EXIT_CMD_LINE_ERR);	/* doesn't return */
}

void parse_file(pf)
    psf_file *pf;
{
    module *mod;

#ifdef FREE_MODULE
    freeformat f;
    int i;

#endif

    mod = PSF_MALLOC(struct module);
    if (VERBOSE)
	(void) fprintf(stdout, "%s: reading from %s\n",
		       progname, filename);
    (void) read_module(mod, pf->fp);
    (void) fclose(pf->fp);
    mod->name = psf_basename(pf->name);

    if (VERBOSE)
	(void) fprintf(stdout,
		       "%s: checking process expressions of %s\n",
		       progname, mod->name);

    check_lts(mod);

#ifdef FREE_MODULE
    for (i = 0; i < mod->entries_table[ADM]; i++) {
	f = mod->adm[i].ff;
	while (f) {
	    if (f->tag)
		PSF_FREE(f->tag);
	    if (f->info)
		PSF_FREE(f->info);
	    f = f->next;
	}
    }
    PSF_FREE(mod->adm);
    for (i = 0; i < mod->entries_table[SOR]; i++) {
	f = mod->sor[i].ff;
	while (f) {
	    if (f->tag)
		PSF_FREE(f->tag);
	    if (f->info)
		PSF_FREE(f->info);
	    f = f->next;

	}
    }

    PSF_FREE(mod->sor);
    for (i = 0; i < mod->entries_table[FUN]; i++) {
	f = mod->fun[i].ff;
	while (f) {
	    if (f->tag)
		PSF_FREE(f->tag);
	    if (f->info)
		PSF_FREE(f->info);
	    f = f->next;
	}
    }
    PSF_FREE(mod->fun);
    for (i = 0; i < mod->entries_table[ATM]; i++) {
	f = mod->atm[i].ff;
	while (f) {
	    if (f->tag)
		PSF_FREE(f->tag);
	    if (f->info)
		PSF_FREE(f->info);
	    f = f->next;
	}
    }
    PSF_FREE(mod->atm);
    for (i = 0; i < mod->entries_table[PRO]; i++) {
	f = mod->pro[i].ff;
	while (f) {
	    if (f->tag)
		PSF_FREE(f->tag);
	    if (f->info)
		PSF_FREE(f->info);
	    f = f->next;
	}
    }
    PSF_FREE(mod->pro);
    for (i = 0; i < mod->entries_table[SET]; i++) {
	f = mod->set[i].ff;
	while (f) {
	    if (f->tag)
		PSF_FREE(f->tag);
	    if (f->info)
		PSF_FREE(f->info);
	    f = f->next;
	}
    }
    PSF_FREE(mod->set);
    for (i = 0; i < mod->entries_table[COM]; i++) {
	f = mod->com[i].ff;
	while (f) {
	    if (f->tag)
		PSF_FREE(f->tag);
	    if (f->info)
		PSF_FREE(f->info);
	    f = f->next;
	}
    }
    PSF_FREE(mod->com);
    for (i = 0; i < mod->entries_table[VAR]; i++) {
	f = mod->var[i].ff;
	while (f) {
	    if (f->tag)
		PSF_FREE(f->tag);
	    if (f->info)
		PSF_FREE(f->info);
	    f = f->next;
	}
    }
    PSF_FREE(mod->var);
    for (i = 0; i < mod->entries_table[EQU]; i++) {
	f = mod->equ[i].ff;
	while (f) {
	    if (f->tag)
		PSF_FREE(f->tag);
	    if (f->info)
		PSF_FREE(f->info);
	    f = f->next;
	}
    }
    PSF_FREE(mod->equ);
    for (i = 0; i < mod->entries_table[DEF]; i++) {
	f = mod->def[i].ff;
	while (f) {
	    if (f->tag)
		PSF_FREE(f->tag);
	    if (f->info)
		PSF_FREE(f->info);
	    f = f->next;
	}
    }
    PSF_FREE(mod->def);
    PSF_FREE(mod);
#endif

}
int main(argc, argv)
    int argc;
    char **argv;
{
    psf_file pf;
    static suffix suffixes[] = {{LTSSUFFIX, LTSSUFFIX_DEFAULT},
    {TILSUFFIX, TILSUFFIX_DEFAULT},
    {NULL, NULL}
    };
    block *b, *p;
    long t, t0, t_reading, t_creating, t_initial, t_partition;
    float ratio;
    int i, nr_t = 0;
    transition *tr;
    time_t reals;
    time_t realb;

    extern int nrblocks;	/* needed for efficient block allocation */
    extern int lastblock;	/* see typedef.c, new_block() */
    extern int nrit;		/* needed for efficient splitting of blocks */
    extern int lastit;
    extern int ba_m;		/* needed for efficient keeping of hmlu info */
    extern int ba_n;

    t = t_reading = t_creating = t_initial = t_partition = 0;
    t0 = clock();
    realb = time((time_t *) NULL);
    progname = psf_basename(argv[0]);
    skip = make_ident("skip");
    tick = fresh_ident("tick");
    delta = new_state();
    delta->name = make_ident("delta");
    final = new_state();
    final->name = make_ident("FINAL_STATE");
    parse_args(argc, argv);
    /* reading the input */
    if (timing)
	t = clock();
    ss.n = 0;
    ss.m = 1;
    ss.s = PSF_MALLOC(state *);
    trans = PSF_MALLOC(transition);
    if (file_list) {
	while (*file_list) {
	    pf = psf_fopen(*file_list, suffixes);
	    filename = pf.name;
	    parse_file(&pf);
	    file_list++;
	}
    } else {
	filename = "stdin";
	pf.name = filename;
	pf.fp = stdin;
	parse_file(&pf);
    }
    if (timing)
	t_reading = (clock() - t) / 1000;

    /* done reading */
    /* creating transitions */
    if (VERBOSE)
	(void) fprintf(stdout, "%s: creating transitions\n",
		       progname);

    if (timing)
	t = clock();
    make_transitions();
    if (timing)
	t_creating = (clock() - t) / 1000;

    /* done creating transitions */
    lastblock = nrblocks = 1 + (ss.n * 2);	/* numbers for block allocation
						 * higher is  faster  to high
						 * costs to much memory */
    ba_n = ba_m = 10 * ss.m;	/* same for block_sets */


    if (!(VERBOSE_HMLU && OUTPUT)) {	/* if these are set certain input
					 * graphs are printed. */
	/* creating initial block */
	if (VERBOSE)
	    (void) fprintf(stdout, "%s: creating initial block\n",
			   progname);
	if (timing)
	    t = clock();
	b = initial_block();
	if (timing)
	    t_initial = (clock() - t) / 1000;
	lastit = nrit = ss.n * b->in;	/* needed for efficient allocating of
					 * arrays of transitions */
	/* done creating initial block */
	resize_idents();
	/* No new idents, save some memory */

	/* partitioning initial block */
	if (VERBOSE)
	    (void) fprintf(stdout, "%s: partitioning initial block\n",
			   progname);
	if (timing) {

	    for (i = 0; i < b->on; i++) {
		tr = b->ot[i];
		while (tr) {
		    nr_t++;
		    tr = tr->next;
		}
	    }
	    t = clock();
	}
	p = partition(b);	/* Now p is the pointer to the first of the
				 * list of stable blocks b is pointer to split
				 * tree */

	if (timing)
	    t_partition = (clock() - t) / 1000;

	/* done partitioning initial block */
	/* give time results */
	if (timing) {
	    t = (clock() - t0) / 1000;
	    (void) fprintf(stdout,
		     "\nCPU time in seconds, resolution: 0.016667 seconds\n");
	    (void) fprintf(stdout, "transitions: %5i, ", nr_t);
	    (void) fprintf(stdout, "states: %5i\n", ss.n);
	    t0 = t - t_reading - t_creating - t_initial - t_partition;
	    ratio = (float) t_partition / (float) (nr_t * ss.n);
	    (void) fprintf(stdout, "total\t\t\t%7.3f\t%6.2f%%\n",
			   (float) t / TU, PC * t / (float) t);
	    (void) fprintf(stdout, "reading input\t\t%7.3f\t%6.2f%%\n",
			   (float) t_reading / TU, PC * t_reading / (float) t);
	    (void) fprintf(stdout, "creating transitions\t%7.3f\t%6.2f%%\n",
			(float) t_creating / TU, PC * t_creating / (float) t);
	    (void) fprintf(stdout, "creating initial block\t%7.3f\t%6.2f%%\n",
			   (float) t_initial / TU, PC * t_initial / (float) t);
	    (void) fprintf(stdout, "partitioning\t\t%7.3f\t%6.2f%%, ",
		      (float) t_partition / TU, PC * t_partition / (float) t);
	    (void) fprintf(stdout, "time*1000/trans*states: %8f\n", ratio);
	    (void) fprintf(stdout, "other bits\t\t%7.3f\t%6.2f%%\n",
			   (float) t0 / TU, PC * t0 / (float) t);
	    reals = time((time_t *) NULL);
	    (void) fprintf(stdout, "real seconds: %li\n", reals - realb);
	}
    } else
	p = (block *) NULL;
    questions(p, VERBOSE_STATES, VERBOSE_PARTITION, VERBOSE_HMLU);
    if (result)
	interactive_questions(p);
    return 0;
}
