#include <assert.h>
#include <stdio.h>
#include <string.h>

#include "psf_prototype.h"
#include "psf_standards.h"
#ifdef THIS_IS_ITIL_TIL
#include "realkey.h"
#include "xtiltype.h"
#else
#include "tiltype.h"
#endif

#include "printed_tuples.h"
#include "write_subs_til.h"

static FILE *output = NULL;
static struct module *my_mod;

static void set_output_file(fp)
    FILE *fp;
{
    output = fp;
}

static char table_char(t)
    tabletype t;
{
    return ('0' + t);
}

void write_index(ind)
    struct indextype *ind;
{
    if (ind->table == SOR && ind->key == 0)
	fprintf(output, "[%c.0] ", table_char(ind->table));
    else
        fprintf(output, "[%c.%d] ", table_char(ind->table),

#ifdef THIS_IS_ITIL_TIL
		   get_real_key(ind)
#else
		   ind->key
#endif

	);
}

void write_indexlist(ind_l)
    struct indexlist *ind_l;
{
    int anum;

    (void) fprintf(output, "%u  ", ind_l->a);
    for (anum = 0; anum < ind_l->a; anum++)
	write_index(&ind_l->indlist[anum]);
}

void write_ae_term(ae_t)
    struct ae_term *ae_t;
{
    int anum;

    if (ae_t->t == TUPLE) {
	(void) fprintf(output, "< %u , ", ae_t->a);
	for (anum = 0; anum < ae_t->a; anum++)
	    write_ae_term(&ae_t->ae_list[anum]);
	(void) fprintf(output, ">");
    } else {
	write_index(&ae_t->ind);
	if (ae_t->a > 0) {
	    (void) fprintf(output, "(");
	    for (anum = 0; anum < ae_t->a; anum++)
		write_ae_term(&ae_t->ae_list[anum]);
	    (void) fprintf(output, ")");
	}
    }
}

void write_s_term(s_t)
    struct s_term *s_t;
{
    int anum;

    (void) fprintf(output, "<%c,%u>(", s_t->fun, s_t->a);
    for (anum = 0; anum < s_t->a; anum++) {
	if (s_t->u_tag[anum] == 0)
	    write_ae_term(s_t->arr[anum].ae_t);
	else
	    write_s_term(s_t->arr[anum].s_t);
    }
    (void) fprintf(output, ")");
}

static char *name_table[] = {
    "",
    "skip",
    "alt",
    "seq",
    "par",
    "sum",
    "merge",
    "encaps",
    "hide",
    "delta",
    "if",
    "interrupt",
    "disrupt",
    "prio",
    "star",
    "sharp"
};

static char *exp_name(expr)
    char expr;
{
    switch (expr) {
    case AET:
	return (name_table[0]);
    case SKP:
	return (name_table[1]);
    case ALT:
	return (name_table[2]);
    case SEQ:
	return (name_table[3]);
    case PAR:
	return (name_table[4]);
    case SUM:
	return (name_table[5]);
    case MRG:
	return (name_table[6]);
    case ENC:
	return (name_table[7]);
    case HID:
	return (name_table[8]);
    case DLK:
	return (name_table[9]);
    case IF:
	return (name_table[10]);
    case INTR:
	return (name_table[11]);
    case DISR:
	return (name_table[12]);
    case PRIO:
	return (name_table[13]);
    case STAR:
	return (name_table[14]);
    case SHARP:
	return (name_table[15]);
    default:
	assert(0);
    }

    return 0;			/* to shut up lint */
}
void write_pr_expr(p_expr)
    struct process_expr *p_expr;
{
    int anum;

    if (p_expr->fun == AET) {
	write_ae_term(&p_expr->proc_expr.pe2);
    } else {
	switch (p_expr->fun) {
	case SKP:
	    (void) fprintf(output, "<%s> ", exp_name(SKP));
	    break;
	case SUM:
	case MRG:
	case ENC:
	case HID:
	    (void) fprintf(output, "<%s>(",
			   exp_name(p_expr->fun));
	    write_index(&p_expr->proc_expr.pe1.ind);
	    write_pr_expr(p_expr->proc_expr.pe1.pe);
	    (void) fprintf(output, ") ");
	    break;
	case ALT:
	case SEQ:
	case PAR:
	case INTR:
	case DISR:
	case STAR:
	case SHARP:
	    (void) fprintf(output, "<%s,%u>(",
			   exp_name(p_expr->fun),
			   p_expr->proc_expr.pe3.a);
	    for (anum = 0; anum < p_expr->
		    proc_expr.pe3.a; anum++)
		write_pr_expr(&p_expr->
			      proc_expr.pe3.pe[anum]);
	    (void) fprintf(output, ")");
	    break;
	case DLK:
	    (void) fprintf(output, "<%s> ", exp_name(DLK));
	    break;
	case IF:
	    (void) fprintf(output, "<%s>(", exp_name(IF));
	    write_ae_term(&p_expr->proc_expr.pe4.aex);
	    write_ae_term(&p_expr->proc_expr.pe4.aey);
	    write_pr_expr(p_expr->proc_expr.pe4.pe);
	    (void) fprintf(output, ")");
	    break;
	case PRIO:
	    (void) fprintf(output, "<%s>(", exp_name(PRIO));
	    write_indexlist(&p_expr->proc_expr.pe5.sets);
	    write_pr_expr(p_expr->proc_expr.pe5.pe);
	    (void) fprintf(output, ")");
	    break;
	}
    }
}

void write_sor_tuple(sor)
    sor_tuple *sor;
{
    sor = sor;			/* to shut up lint */
}

void write_fun_tuple(fun)
    fun_tuple *fun;
{
    write_indexlist(&fun->sor_indlist);
    write_indexlist(&fun->return_list);
}

void write_atm_tuple(atm)
    atm_tuple *atm;
{
    write_indexlist(&atm->sor_indlist);
}

void write_pro_tuple(pro)
    pro_tuple *pro;
{
    write_indexlist(&pro->sor_indlist);
}

void write_set_tuple(set)
    set_tuple *set;
{
    if (set->ind.key == 0)
	(void) fprintf(output, "[%c.0] ", table_char(set->ind.table));
    else
	write_index(&set->ind);
    if (set->u_tag == 0)
	write_index(&set->construct.sort);
    else
	write_s_term(&set->construct.set_term);
}

void write_com_tuple(com)
    com_tuple *com;
{
    write_ae_term(&com->aet[0]);
    (void) fprintf(output, "  ");
    write_ae_term(&com->aet[1]);
    (void) fprintf(output, "  ");
    write_ae_term(&com->aet[2]);
}

void write_var_tuple(var)
    var_tuple *var;
{
    write_index(&var->ind);
}

void write_equ_tuple(equ)
    equ_tuple *equ;
{
    int c;

    write_ae_term(&equ->aet1);
    (void) fprintf(output, " = ");
    write_ae_term(&equ->aet2);
    if (equ->a > 0) {
	(void) fprintf(output, " <= %d  ", equ->a);
	for (c = 0; c < equ->a; c++) {
	    if (c)
		(void) fprintf(output, " , ");
	    write_ae_term(&equ->guard[c].aet1);
	    (void) fprintf(output, " = ");
	    write_ae_term(&equ->guard[c].aet2);
	}
    }
}

void write_def_tuple(def)
    def_tuple *def;
{
    write_ae_term(&def->ae_t);
    (void) fprintf(output, " = ");
    write_pr_expr(&def->p_expr);
}

static void write_freeformat(ff)
freeformat ff;
{
#ifdef NEWFF
    (void) fprintf(output, "\t{");
    for ( ; ff != NULL; ff = ff->next) {
	(void) fprintf(output, " <%s>%s", ff->tag, ff->info);
    }
    (void) fprintf(output, " }\n");
#else
    (void) fprintf(output, "\t{%s}\n", ff);
#endif
}

void write_tuple(table, key)
    tabletype table;
    keytype key;
{
    record_as_printed(table, key);	/* avoids circularity */
    switch (table) {
    case ADM:
#ifdef THIS_IS_ITIL_TIL
	assert(0);		/* there are no ADM tuples in ITIL */
#else
	(void) fprintf(output, "[%c.%d]\t", table_char(table), key);
	write_freeformat(my_mod->adm[key].ff);
#endif

	break;

    case SOR:
#ifdef WRITEDBU
	write_subs_sor_tuple(&my_mod->sor[key]);
#endif
	(void) fprintf(output, "[%c.%d]\t", table_char(table), key);
	write_sor_tuple(&my_mod->sor[key]);

#ifdef THIS_IS_ITIL_TIL
	(void) fprintf(output, "\t{ <n>%s <o>%u <v>%c }\n",
		       my_mod->sor[key].ff,
		       my_mod->sor[key].o,
			(my_mod->sor[key].t == EXPORTS) ? 'e' : 'h');
#else
	write_freeformat(my_mod->sor[key].ff);
#endif

	break;

    case FUN:
#ifdef WRITEDBU
	write_subs_fun_tuple(&my_mod->fun[key]);
#endif
	(void) fprintf(output, "[%c.%d]\t", table_char(table), key);
	write_fun_tuple(&my_mod->fun[key]);

#ifdef THIS_IS_ITIL_TIL
	(void) fprintf(output, "\t{ <n>%s <o>%u <v>%c }\n",
		       my_mod->fun[key].ff,
		       my_mod->fun[key].o,
			(my_mod->fun[key].t == EXPORTS) ? 'e' : 'h');

#else
	write_freeformat(my_mod->fun[key].ff);
#endif

	break;

    case ATM:
#ifdef WRITEDBU
	write_subs_atm_tuple(&my_mod->atm[key]);
#endif
	(void) fprintf(output, "[%c.%d]\t", table_char(table), key);
	write_atm_tuple(&my_mod->atm[key]);

#ifdef THIS_IS_ITIL_TIL
	(void) fprintf(output, "\t{ <n>%s <o>%u <v>%c }\n",
		       my_mod->atm[key].ff,
		       my_mod->atm[key].o,
			(my_mod->atm[key].t == EXPORTS) ? 'e' : 'h');
#else
	write_freeformat(my_mod->atm[key].ff);
#endif

	break;

    case PRO:
#ifdef WRITEDBU
	write_subs_pro_tuple(&my_mod->pro[key]);
#endif
	(void) fprintf(output, "[%c.%d]\t", table_char(table), key);
	write_pro_tuple(&my_mod->pro[key]);

#ifdef THIS_IS_ITIL_TIL
	(void) fprintf(output, "\t{ <n>%s <o>%u <v>%c }\n",
		       my_mod->pro[key].ff,
		       my_mod->pro[key].o,
			(my_mod->pro[key].t == EXPORTS) ? 'e' : 'h');
#else
	write_freeformat(my_mod->pro[key].ff);
#endif

	break;

    case SET:
#ifdef WRITEDBU
	write_subs_set_tuple(&my_mod->set[key]);
#endif
	(void) fprintf(output, "[%c.%d]\t", table_char(table), key);
	write_set_tuple(&my_mod->set[key]);

#ifdef THIS_IS_ITIL_TIL
	(void) fprintf(output, "\t{ <n>%s <o>%u <v>%c }\n",
		       my_mod->set[key].ff,
		       my_mod->set[key].o,
			(my_mod->set[key].t == EXPORTS) ? 'e' : 'h');
#else
	write_freeformat(my_mod->set[key].ff);
#endif

	break;

    case COM:
#ifdef WRITEDBU
	write_subs_com_tuple(&my_mod->com[key]);
#endif
	(void) fprintf(output, "[%c.%d]\t", table_char(table), key);
	write_com_tuple(&my_mod->com[key]);

#ifdef THIS_IS_ITIL_TIL
	(void) fprintf(output, "\t{ <o>%u }\n",
		       my_mod->com[key].o);
#else
	write_freeformat(my_mod->com[key].ff);
#endif

	break;

    case VAR:
#ifdef WRITEDBU
	write_subs_var_tuple(&my_mod->var[key]);
#endif
	(void) fprintf(output, "[%c.%d]\t", table_char(table), key);
	write_var_tuple(&my_mod->var[key]);

#ifdef THIS_IS_ITIL_TIL
	(void) fprintf(output, "\t{ <n>%s <o>%u }\n",
		       my_mod->var[key].ff,
		       my_mod->var[key].o);
#else
	write_freeformat(my_mod->var[key].ff);
#endif

	break;

    case EQU:
#ifdef WRITEDBU
	write_subs_equ_tuple(&my_mod->equ[key]);
#endif
	(void) fprintf(output, "[%c.%d]\t", table_char(table), key);
	write_equ_tuple(&my_mod->equ[key]);

#ifdef THIS_IS_ITIL_TIL
	(void) fprintf(output, "\t{ <n>%s <o>%u }\n",
		       my_mod->equ[key].ff,
		       my_mod->equ[key].o);
#else
	write_freeformat(my_mod->equ[key].ff);
#endif

	break;

    case DEF:
#ifdef WRITEDBU
	write_subs_def_tuple(&my_mod->def[key]);
#endif
	(void) fprintf(output, "[%c.%d]\t", table_char(table), key);
	write_def_tuple(&my_mod->def[key]);

#ifdef THIS_IS_ITIL_TIL
	(void) fprintf(output, "\t{ <o>%u }\n",
		       my_mod->def[key].o);
#else
	write_freeformat(my_mod->def[key].ff);
#endif

	break;
    }

}


void write_module(a_mod, fp)
    struct module *a_mod;
    FILE *fp;
{
    tabletype i;
    keytype j;

    my_mod = a_mod;
    set_output_file(fp);

#ifdef THIS_IS_ITIL_TIL		/* then my_mod->entries_table[ADM] == 0, so,
				 * we'll have to derive the ADM tuples from the
				 * MOD table */
    for (j = 1; j <= my_mod->entries_table[MOD]; j++) {
	fprintf(output, "[0.%d]\t{ <m>%d %s }\n", j, j, my_mod->mod[j].ff);
    }
#endif

    for (i = 0; i <= DEF; i++) {
	if (!my_mod->entries_table[i])
	    continue;
	for (j = 1; j <= my_mod->entries_table[i]; j++) {
	    switch (i) {
	    case ADM:
		if (my_mod->adm[j].ff != NULL && !printed(i, j))
		    write_tuple(i, j);
		break;
	    case SOR:
		if (my_mod->sor[j].ff != NULL && !printed(i, j))
		    write_tuple(i, j);
		break;
	    case FUN:
		if (my_mod->fun[j].ff != NULL && !printed(i, j))
		    write_tuple(i, j);
		break;
	    case ATM:
		if (my_mod->atm[j].ff != NULL && !printed(i, j))
		    write_tuple(i, j);
		break;
	    case PRO:
		if (my_mod->pro[j].ff != NULL && !printed(i, j))
		    write_tuple(i, j);
		break;
	    case SET:
		if (my_mod->set[j].ff != NULL && !printed(i, j))
		    write_tuple(i, j);
		break;
	    case COM:
		if (my_mod->com[j].ff != NULL && !printed(i, j))
		    write_tuple(i, j);
		break;
	    case VAR:
		if (my_mod->var[j].ff != NULL && !printed(i, j))
		    write_tuple(i, j);
		break;
	    case EQU:
		if (my_mod->equ[j].ff != NULL && !printed(i, j))
		    write_tuple(i, j);
		break;
	    case DEF:
		if (my_mod->def[j].ff != NULL && !printed(i, j))
		    write_tuple(i, j);
		break;
	    }
	}
    }
    destroy_record_printed();	/* cleanup */
    set_output_file(stdout);
}

void write_til_init()
{
    set_output_file(stdout);
}
