/*
 * Equation handling part of PSF rewrite system
 * 
 * Functions: Create an Equation Manager Add an Equation to a Manager Delete an
 * Equation from a Manager
 * 
 */

#include <stdio.h>
#include <string.h>
#include "eqm.h"
#include "tiltype.h"
#include "eqm_local.h"

#ifdef PRTILPSF
#include "prtilparts.h"
#else
#define print_ae_term(s)
#endif

#include "psf_malloc.h"

static eqm_t *last_eqm = NULL;
static int eqm_options = 0;

#ifdef EQM_STATS
static unsigned int rewrite_count, ntries;

static void print_stats(eq)
    eq_list_t *eq;
{
    if (eq->nsuccess == 0 && eq->ntries == 0)
	return;
    printf("%8u%8u ", eq->nsuccess, eq->ntries);
    rewrite_count += eq->nsuccess;
    ntries += eq->ntries;
    if (eq->cheap)
	putchar('*');
    else
	putchar(' ');
    print_ae_term(&eq->this_eq.aet1);
    printf(" = ");
    print_ae_term(&eq->this_eq.aet2);
    if (eq->this_eq.a > 0)
	printf(" when ...");
    putchar('\n');
}

void eqm_print_stats()
{
    eqm_t *eqm;
    int i, j;
    eq_list_t *eq;
    eq_fun_list_t *fun;

    if (!(eqm_options & EQM_PRINT_STATS))
	return;

    for (eqm = last_eqm; eqm != NULL; eqm = eqm->next_eqm) {
	rewrite_count = 0;
	ntries = 0;
	printf("#matches  #tries rule\n\n");
	for (i = 1; i <= eqm->fun_cnt; i++) {
	    if (eqm->fun_list[i] == NULL)
		continue;
	    fun = eqm->fun_list[i];
	    for (j = 0; j < PRIME1; j++) {
		eq = fun->eq_list[j];
		while (eq != NULL) {
		    print_stats(eq);
		    eq = eq->next_eq;
		}
	    }
	    eq = fun->var_eq;
	    while (eq != NULL) {
		print_stats(eq);
		eq = eq->next_eq;
	    }
	}
	printf("\n# of rewrites: %u, # of tries %u\n\n", rewrite_count, ntries);
    }
}

void eqm_clean(/*eqm*/)
    /*eqm_t *eqm;*/
{
    int i, j;
    eq_list_t *eq;
    eq_fun_list_t *fun;
    eqm_t *eqm;

    eqm = last_eqm;
    for (i = 1; i <= eqm->fun_cnt; i++) {
	if (eqm->fun_list[i] == NULL)
	    continue;
	fun = eqm->fun_list[i];
	for (j = 0; j < PRIME1; j++) {
	    eq = fun->eq_list[j];
	    while (eq != NULL) {
		eq->ntries = eq->nsuccess = 0;
		eq = eq->next_eq;
	    }
	}
	eq = fun->var_eq;
	while (eq != NULL) {
	    eq->ntries = eq->nsuccess = 0;
	    eq = eq->next_eq;
	}
    }
}

#endif

static int count_vars(t)
    ae_term *t;
{
    int i, result = 0;

    if (term_is_variable(t))
	return 1;

    for (i = t->a; i-- > 0;)
	result += count_vars(t->ae_list + i);

    return result;
}

eqm_t *
 create_eqm(nfuns)
    unsigned int nfuns;
{
    eqm_t *new = PSF_MALLOC(eqm_t);
    char *opt, *getenv();
    int i;

    if (new == NULL)
	return NULL;

    new->fun_list = PSF_NMALLOC(eq_fun_list_t *, nfuns + 1);
    new->fun_cnt = nfuns;

    if (new->fun_list == NULL)
	return NULL;

    for (i = 0; i <= nfuns; i++)
	new->fun_list[i] = NULL;

    opt = getenv("EQMOPTIONS");
    if (opt) {
	int options = 0;

	if (strstr(opt, "stats") != NULL)
	    options |= EQM_PRINT_STATS;
	if (strstr(opt, "reverse") != NULL)
	    options |= EQM_REVERSE;
	eqm_setoptions(options);
    }

#if defined(HAS_ON_EXIT) && defined(EQM_STATS)
    if (last_eqm == NULL) {
	on_exit(eqm_print_stats, (char *) NULL);
    }
#endif

    new->next_eqm = last_eqm;
    last_eqm = new;
    return new;
}

eqm_t *
 add_equation(eqm, eq)
    eqm_t *eqm;
    equ_tuple eq;
{
    eq_list_t *new = PSF_MALLOC(eq_list_t);
    eq_fun_list_t *fun;
    int cheap;

    if (new == NULL)
	return NULL;

    fun = eqm->fun_list[eq.aet1.ind.key];
    new->ntries = new->nsuccess = 0;
    new->this_eq = eq;
    new->cheap = cheap = eq.a == 0 && count_vars(&eq.aet1) > 0 &&
	count_vars(&eq.aet2) == 0;

    if (fun == NULL) {
	int i;

	fun = PSF_MALLOC(eq_fun_list_t);
	if (fun == NULL)
	    return NULL;
	for (i = 0; i < PRIME1; i++)
	    fun->eq_list[i] = NULL;
	fun->var_eq = NULL;
	eqm->fun_list[eq.aet1.ind.key] = fun;
    }
    if (eq.aet1.a == 0 || term_is_variable(&eq.aet1.ae_list[0])) {
	int i;

	if (fun->var_eq == NULL || eqm_options & EQM_REVERSE) {
	    new->next_eq = fun->var_eq;
	    fun->var_eq = new;
	} else {
	    eq_list_t *tmp = fun->var_eq;

	    while (tmp->next_eq != NULL)
		tmp = tmp->next_eq;
	    new->next_eq = NULL;
	    tmp->next_eq = new;
	}

	for (i = 0; i < PRIME1; i++) {
	    new = PSF_MALLOC(eq_list_t);
	    new->ntries = new->nsuccess = 0;
	    new->cheap = cheap;
	    new->this_eq = eq;

	    if (fun->eq_list[i] == NULL || eqm_options & EQM_REVERSE) {
		new->next_eq = fun->eq_list[i];
		fun->eq_list[i] = new;
	    } else {
		eq_list_t *tmp;

		tmp = fun->eq_list[i];
		while (tmp->next_eq != NULL)
		    tmp = tmp->next_eq;
		new->next_eq = NULL;
		tmp->next_eq = new;
	    }
	}
    } else {
	int hash1 = FUN1HASH(eq.aet1.ae_list[0].ind);

	if (fun->eq_list[hash1] == NULL || eqm_options & EQM_REVERSE) {
	    new->next_eq = fun->eq_list[hash1];
	    fun->eq_list[hash1] = new;
	} else {
	    eq_list_t *tmp;

	    tmp = fun->eq_list[hash1];
	    while (tmp->next_eq != NULL)
		tmp = tmp->next_eq;
	    new->next_eq = NULL;
	    tmp->next_eq = new;
	}
    }
    return eqm;
}

/*
 * eqm_t * del_equation(eqm, eq) eqm_t	 *eqm; equ_tuple eq; { return eqm; }
 */

/*
 * Defines the order of two equations.
 */
#define CRIT(t1,t2)	(((t1)->cheap < (t2)->cheap && (t2)->nsuccess != 0) ||\
			  ((t1)->cheap == (t2)->cheap && \
			    (t1)->nsuccess < (t2)->nsuccess) || \
			  (t1)->nsuccess == 0)

void eqm_sort(eqm)
    eqm_t *eqm;
{
    int i;

    for (i = 1; i <= eqm->fun_cnt; i++) {
	eq_fun_list_t *fun;
	eq_list_t *last, *one, *two, *pone;
	int j;

	if (eqm->fun_list[i] == NULL)
	    continue;

	fun = eqm->fun_list[i];

#if 1
	last = NULL;
	if (fun->var_eq != NULL && fun->var_eq->next_eq != NULL) {
	    while (last != fun->var_eq->next_eq) {
		pone = NULL;
		one = fun->var_eq;
		two = one->next_eq;
		while (two != last) {
		    if (CRIT(one, two)) {
			one->next_eq = two->next_eq;
			two->next_eq = one;
			if (pone == NULL)
			    fun->var_eq = two;
			else
			    pone->next_eq = two;
			pone = two;
			two = one->next_eq;
		    } else {
			pone = one;
			one = one->next_eq;
			two = two->next_eq;
		    }
		}
		last = one;
	    }
	}
#endif

	for (j = 0; j < PRIME1; j++) {
	    if (fun->eq_list[j] == NULL || fun->eq_list[j]->next_eq == NULL)
		continue;

	    last = NULL;
	    while (last != fun->eq_list[j]->next_eq) {
		pone = NULL;
		one = fun->eq_list[j];
		two = one->next_eq;
		while (two != last) {
		    if (CRIT(one, two)) {
			one->next_eq = two->next_eq;
			two->next_eq = one;
			if (pone == NULL)
			    fun->eq_list[j] = two;
			else
			    pone->next_eq = two;
			pone = two;
		    } else {
			pone = one;
			one = one->next_eq;
		    }
		    two = one->next_eq;
		}
		last = one;
	    }
	}
    }
}

void eqm_setoptions(options)
    int options;
{
    eqm_options = options;
}
