#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <ctype.h>
#include "psf_prototype.h"
#include "tiltype.h"
#include "fieldex.h"
#include "prtilparts.h"
#include "psf_malloc.h"
#include "psf_standards.h"

bool duplicate_names_allowed = FALSE;
bool Option_skipnr = FALSE;

static struct module *current_module;
static FILE *output;

void set_print_current_module(mod)
    struct module *mod;
{
    current_module = mod;
}

void set_print_output_file(fp)
    FILE *fp;
{
    output = fp;
}

void print_freeformat(table, key)
    tabletype table;
    keytype key;
{
    freeformat ff;

    switch (table) {
    case SOR:
	ff = current_module->sor[key].ff;
	break;
    case FUN:
	ff = current_module->fun[key].ff;
	break;
    case ATM:
	ff = current_module->atm[key].ff;
	break;
    case PRO:
	ff = current_module->pro[key].ff;
	break;
    case SET:
	ff = current_module->set[key].ff;
	break;
    case COM:
	ff = current_module->com[key].ff;
	break;
    case VAR:
	ff = current_module->var[key].ff;
	break;
    case EQU:
	ff = current_module->equ[key].ff;
	break;
    case DEF:
	ff = current_module->def[key].ff;
	break;
    }
    print_ffname(ff, table, key);
}

static char opnum[] = "!@$%^&+*:?~/|\\_.";

void print_ffname(ff, table, key)
    freeformat ff;
    tabletype table;
    keytype key;
{
    char *p, *underscore, *post, *t;
    bool operator, dot;
    int i;

    p = psf_strdup(field_extract("n", ff, table, key));
    if (duplicate_names_allowed) {
	operator = FALSE;
	dot = FALSE;
	if (table == FUN)
	    for (i = 0; i < sizeof opnum; i++)
		if (*p == opnum[i]) {
		    operator = TRUE;
		    break;
		}
	post = psf_strdup(field_extract("o", ff, table, key));
	if (operator) {
	    if (p[strlen(p) - 1] == '_') {
		p[strlen(p) - 1] = '\0';
		if (p[strlen(p) - 1] == '.') {
		    p[strlen(p) - 1] = '\0';
		    underscore = "._";
		    dot = TRUE;
		} else
		    underscore = "_";
	    } else
		underscore = "";
	    if (!dot) {
		for (t = post; *t != '\0'; t++)
		    if (isalnum(*t))
			*t = opnum[(*t - '0') % 10];
	    }
	    if (p[strlen(p)-1] == '-')
		(void) fprintf(output, "%s~%s%s", p, post, underscore);
	    else
		(void) fprintf(output, "%s-%s%s", p, post, underscore);
	} else {
	    if (p[strlen(p)-1] == '-')
		(void) fprintf(output, "%s~%s", p, post);
	    else
		(void) fprintf(output, "%s-%s", p, post);
	}
	PSF_FREE(post);
    } else
	(void) fprintf(output, "%s", p);
    PSF_FREE(p);
}

void print_term_name(index)
    struct indextype index;
{
    freeformat ff;
    char *p, *post, *t, *underscore;
    bool operator, dot;
    int i;

    switch (index.table) {
    case SOR:
	ff = current_module->sor[index.key].ff;
	break;
    case FUN:
	ff = current_module->fun[index.key].ff;
	break;
    case ATM:
	ff = current_module->atm[index.key].ff;
	break;
    case PRO:
	ff = current_module->pro[index.key].ff;
	break;
    case SET:
	ff = current_module->set[index.key].ff;
	break;
    case VAR:
	ff = current_module->var[index.key].ff;
	break;
    default:
	assert(THIS_CANNOT_HAPPEN);
	break;
    }
    p = psf_strdup(field_extract("n", ff, index.table, index.key));
    if (duplicate_names_allowed) {
	operator = FALSE;
	dot = FALSE;
	if (index.table == FUN)
	    for (i = 0; i < sizeof opnum; i++)
		if (*p == opnum[i]) {
		    operator = TRUE;
		    break;
		}
	post = psf_strdup(field_extract("o", ff, index.table, index.key));
	if (operator) {
	    if (p[strlen(p) - 1] == '_') {
		p[strlen(p) - 1] = '\0';
		if (p[strlen(p) - 1] == '.') {
		    p[strlen(p) - 1] = '\0';
		    underscore = ".";
		    dot = TRUE;
		} else
		    underscore = "";
	    } else
		underscore = "";
	    if (!dot) {
		for (t = post; *t != '\0'; t++)
		    if (isalnum(*t))
			*t = opnum[(int) (*t - '0') % (int)strlen(opnum)];
	    }
	    if (p[strlen(p)-1] == '-')
		(void) fprintf(output, "%s~%s%s", p[0] == '_' ? p + 1 : p,
			       post, underscore);
	    else
		(void) fprintf(output, "%s-%s%s", p[0] == '_' ? p + 1 : p,
			       post, underscore);
	} else if (p[strlen(p)-1] == '-')
	    (void) fprintf(output, "%s~%s", p, post);
	else
	    (void) fprintf(output, "%s-%s", p, post);
	PSF_FREE(post);
    } else {
	if (p[strlen(p) - 1] == '_')
	    p[strlen(p) - 1] = '\0';
	(void) fprintf(output, "%s", p[0] == '_' ? p + 1 : p);
    }
    PSF_FREE(p);
}

void print_onlyffname(ff, table, key)
    freeformat ff;
    tabletype table;
    keytype key;

{
    (void) fprintf(output, "%s", field_extract("n", ff, table, key));
}

void print_fforigin(ff, table, key)
    freeformat ff;
    tabletype table;
    keytype key;

{
    (void) fprintf(output, "%s", field_extract("o", ff, table, key));
}

void print_sorlist(a, ind, sep)
    arity a;
    struct indextype *ind;
    char *sep;

{
    int i, el;

    if (!a)
	return;
    el = 0;
    for (i = 0; i < a; i++) {
	if (el)
	    (void) fprintf(output, "%s", sep);
	if (ind[i].table == SOR && ind[i].key == 0)
	    fprintf(output, "atoms");
	else
	    print_freeformat(ind[i].table, ind[i].key);
	el++;
    }
}

void printline()
{
    (void) fprintf(output, "\n");
}

static bool infix_operator(t, k)
    tabletype t;
    keytype k;
{
    if (t != FUN)
	return (FALSE);
    return field_extract("n", current_module->fun[k].ff, t, k)[0] == '_';
}

void print_ae_term(term)
    ae_term *term;
{
    int i;
    int infix;

    if (term->t == TUPLE) {
	(void) fprintf(output, "<");
	for (i = 0; i < term->a; i++) {
	    if (i)
		(void) fprintf(output, ", ");
	    print_ae_term(&term->ae_list[i]);
	}
	(void) fprintf(output, ">");
    } else if (term->t != TERM) {	/* Sun got the assert() macro wrong */
	assert(THIS_CANNOT_HAPPEN);
    } else if (term->a == 0) {
	if (term->ind.table == SOR && term->ind.key == 0)
	    fprintf(output, "atoms");
	else
	    print_term_name(term->ind);
    } else if (infix_operator(term->ind.table, term->ind.key)) {
	infix = infix_operator(term->ae_list[0].ind.table,
	    term->ae_list[0].ind.key);
	if (infix)
	    fprintf(output, "(");
	print_ae_term(&term->ae_list[0]);
	if (infix)
	    fprintf(output, ")");
	fprintf (output, " ");
	print_term_name(term->ind);
	fprintf (output, " ");
	for (i = 1; i < term->a; i++) {
	    if (i > 1)
		fprintf(output, ", ");
	    infix = infix_operator(term->ae_list[i].ind.table,
		term->ae_list[i].ind.key);
	    if (infix)
		fprintf(output, "(");
	    print_ae_term(&term->ae_list[i]);
	    if (infix)
		fprintf(output, ")");
	}
    } else {
	print_term_name(term->ind);
	(void) fprintf(output, "(");
	for (i = 0; i < term->a; i++) {
	    if (i)
		fprintf(output, ", ");
	    print_ae_term(&term->ae_list[i]);
	}
	(void) fprintf(output, ")");
    }
}

static int first_placeholder;
static int *var_placeholder;
static char *sep_placeholders;

void print_placeholder(key)
    keytype key;
{
    var_tuple *var_t;

    var_t = &current_module->var[key];
    print_ffname(var_t->ff, VAR, key);
    (void) fprintf(output, " in ");
    print_freeformat(var_t->ind.table, var_t->ind.key);
}

void print_placeholders(term)
    ae_term *term;
{
    int i;

    if (term->ind.table == VAR) {
	if (!var_placeholder[term->ind.key]) {
	    if (first_placeholder) {
		(void) fprintf(output, "%s",
			       sep_placeholders);
		first_placeholder = 0;
	    } else
		(void) fprintf(output, ", ");
	    print_placeholder(term->ind.key);
	    var_placeholder[term->ind.key] = 1;
	}
    }
    if (term->a) {
	for (i = 0; i < term->a; i++)
	    print_placeholders(&term->ae_list[i]);
    }
}

void list_placeholders(a, term, sep)
    arity a;
    union terms *term;
    char *sep;
{
    int i;
    int ent;

    first_placeholder = 1;
    sep_placeholders = sep;
    ent = current_module->entries_table[VAR];
    var_placeholder = PSF_NMALLOC(int, ent + 1);
    for (i = 0; i <= ent; i++)
	var_placeholder[i] = 0;
    if (a)
	for (i = 0; i < a; i++)
	    print_placeholders(term[i].ae_t);
    free((char *) var_placeholder);
}

static int *MarkVariable;

void mark_placeholders(mod)
    struct module *mod;
{
    unsigned int ent;
    int i, j;

    ent = mod->entries_table[VAR];
    MarkVariable = PSF_NMALLOC(int, ent + 1);
    for (i = 0; i <= ent; i++)
	MarkVariable[i] = 0;

    ent = mod->entries_table[SET];
    for (i = 1; i <= ent; i++) {
	if (mod->set[i].u_tag)
	    mark_s_term(&mod->set[i].construct.set_term);
    }

    ent = mod->entries_table[COM];
    for (i = 1; i <= ent; i++) {
	mark_ae_term(&mod->com[i].aet[0]);
	mark_ae_term(&mod->com[i].aet[1]);
	mark_ae_term(&mod->com[i].aet[2]);
    }

    ent = mod->entries_table[DEF];
    for (i = 1; i <= ent; i++) {
	mark_process_expr(&mod->def[i].p_expr);
    }

    ent = mod->entries_table[EQU];
    for (i = 1; i <= ent; i++) {
	unmark_ae_term(&mod->equ[i].aet1);
	unmark_ae_term(&mod->equ[i].aet2);
	if (mod->equ[i].a) {
	    for (j = 0; j < mod->equ[i].a; j++) {
		unmark_ae_term(&mod->equ[i].guard[j].aet1);
		unmark_ae_term(&mod->equ[i].guard[j].aet2);
	    }
	}
    }
}

void mark_s_term(term)
    struct s_term *term;
{
    int i;

    for (i = 0; i < term->a; i++) {
	if (term->u_tag[i] == 0)
	    mark_ae_term(term->arr[i].ae_t);
	else
	    mark_s_term(term->arr[i].s_t);
    }
}

void mark_ae_term(term)
    ae_term *term;
{
    int i;

    if (term->ind.table == VAR)
	MarkVariable[term->ind.key] = 1;
    for (i = 0; i < term->a; i++)
	mark_ae_term(&term->ae_list[i]);
}

void mark_process_expr(pe)
    struct process_expr *pe;
{
    int i;
    unsigned int narg;

    switch (pe->fun) {
    case SKP:
	break;
    case ALT:
    case SEQ:
    case PAR:
    case INTR:
    case DISR:
    case STAR:
    case SHARP:
	narg = pe->proc_expr.pe3.a;
	for (i = 0; i < narg; i++)
	    mark_process_expr(&pe->proc_expr.pe3.pe[i]);
	break;
    case SUM:
    case MRG:
	MarkVariable[pe->proc_expr.pe1.ind.key] = 1;
	mark_process_expr(pe->proc_expr.pe1.pe);
    case ENC:
    case HID:
	mark_process_expr(pe->proc_expr.pe1.pe);
	break;
    case DLK:
	break;
    case IF:
	mark_process_expr(pe->proc_expr.pe4.pe);
	break;
    case PRIO:
	mark_process_expr(pe->proc_expr.pe5.pe);
	break;
    case AET:
	break;
    }
}

void unmark_ae_term(term)
    ae_term *term;
{
    int i;

    if (term->ind.table == VAR)
	MarkVariable[term->ind.key] = 0;
    for (i = 0; i < term->a; i++)
	unmark_ae_term(&term->ae_list[i]);
}

void print_s_term(term)
    struct s_term *term;
{
    int i, el;

    if (term->fun == ENU)
	(void) fprintf(output, "{");
    el = 0;
    for (i = 0; i < term->a; i++) {
	if (el) {
	    switch (term->fun) {
	    case ENU:
		(void) fprintf(output, ", ");
		break;
	    default:
		(void) fprintf(output, " %c ", term->fun);
		break;
	    }
	}
	if (term->u_tag[i] == 0)
	    print_ae_term(term->arr[i].ae_t);
	else
	    print_s_term(term->arr[i].s_t);
	el++;
    }
    if (term->fun == ENU) {
	list_placeholders(term->a, term->arr, " | ");
	(void) fprintf(output, "}");
    }
}

void print_sor_tuple(tuple, key)
    sor_tuple *tuple;
    keytype key;
{
    print_ffname(tuple->ff, SOR, key);
}

void print_fun_tuple(tuple, key)
    fun_tuple *tuple;
    keytype key;
{
    print_ffname(tuple->ff, FUN, key);
    (void) fprintf(output, "\t: ");
    print_sorlist(tuple->sor_indlist.a,
		  tuple->sor_indlist.indlist, " # ");
    (void) fprintf(output, "\t-> ");
    print_sorlist(tuple->return_list.a,
		  tuple->return_list.indlist, " # ");
}

void print_atmpro_tuple(tuple, table, key)
    atm_tuple *tuple;
    tabletype table;
    keytype key;
{
    print_ffname(tuple->ff, table, key);
    if (tuple->sor_indlist.a) {
	(void) fprintf(output, "\t: ");
	print_sorlist(tuple->sor_indlist.a,
		      tuple->sor_indlist.indlist, " # ");
    }
}

void print_exp_section(mod)
    struct module *mod;
{
    sor_tuple *sor_t;
    fun_tuple *fun_t;
    int i, el, sel;
    unsigned int ent;
    char *f;
    int key;

    sel = 0;
    sor_t = mod->sor;
    if (ent = mod->entries_table[SOR]) {
	el = 0;
	for (i = 1; i <= ent; i++)
	    if (sor_t[i].ff != NULL) {
		if (!sel)
		    (void) fprintf(output, "exports\nbegin\n");
		if (el)
		    (void) fprintf(output, "\n\t\t");
		else
		    (void) fprintf(output, "\tsorts\n\t\t");
		print_sor_tuple(&sor_t[i], i);
		if (i < ent)
		    fprintf(output, ",");
		if ((f = get_ff_field("enum", sor_t[i].ff)) != NULL) {
		    sscanf(f, "%d", &key);
		    fprintf(output, "\t-- enumeration: ");
		    print_ffname(current_module->set[key].ff, SET, key);
		} 
		el++;
		sel++;
	    }
	if (el) {
	    printline();
	    printline();
	}
    }
    fun_t = mod->fun;
    if (ent = mod->entries_table[FUN]) {
	el = 0;
	for (i = 1; i <= ent; i++)
	    if (fun_t[i].ff != NULL) {
		if (!sel)
		    (void) fprintf(output, "exports\nbegin\n");
		if (!el)
		    (void) fprintf(output, "\tfunctions\n");
		(void) fprintf(output, "\t\t");
		print_fun_tuple(&fun_t[i], i);
		el++;
		sel++;
		printline();
	    }
	if (el)
	    printline();
    }
    if (sel)
	(void) fprintf(output, "end\n\n");
}

void print_sor_section(ent, tuple)
    unsigned int ent;
    sor_tuple *tuple;
{
    int i, el;
    char *f;
    int key;

    if (!ent)
	return;
    el = 0;
    for (i = 1; i <= ent; i++)
	if (tuple[i].ff != NULL) {
	    if (i == 1)
		(void) fprintf(output, "sorts\n");
	    (void) fprintf(output, "\t");
	    print_sor_tuple(&tuple[i], i);
	    if (i < ent)
		(void) fprintf(output, ",");
	    if ((f = get_ff_field("enum", tuple[i].ff)) != NULL) {
		sscanf(f, "%d", &key);
		fprintf(output, "\t-- enumeration: ");
		print_ffname(current_module->set[key].ff, SET, key);
	    } 
	    fprintf(output, "\n");
	    el++;
	}
    if (el) {
	printline();
    }
}

void print_fun_section(ent, tuple)
    unsigned int ent;
    fun_tuple *tuple;
{
    int i, el;

    if (!ent)
	return;
    el = 0;
    for (i = 1; i <= ent; i++)
	if (tuple[i].ff != NULL) {
	    if (!el)
		(void) fprintf(output, "functions\n");
	    (void) fprintf(output, "\t");
	    print_fun_tuple(&tuple[i], i);
	    printline();
	    el++;
	}
    if (el)
	printline();
}

void print_atmpro_tuples(ent, tuple, id, table)
    unsigned int ent;
    atm_tuple *tuple;
    char *id;
    tabletype table;
{
    int i, el;

    if (!ent)
	return;
    el = 0;
    for (i = 1; i <= ent; i++)
	if (tuple[i].ff != NULL) {
	    if (!el)
		(void) fprintf(output, "%s\n", id);
	    (void) fprintf(output, "\t");
	    print_atmpro_tuple(&tuple[i], table, i);
	    printline();
	    el++;
	}
    if (el)
	printline();
}

void print_atm_section(ent, tuple)
    unsigned int ent;
    atm_tuple *tuple;
{
    print_atmpro_tuples(ent, tuple, "atoms", ATM);
}

void print_pro_section(ent, tuple)
    unsigned int ent;
    pro_tuple *tuple;
{
    print_atmpro_tuples(ent, tuple, "processes", PRO);
}

void print_set_section(ent, tuple)
    unsigned int ent;
    set_tuple *tuple;
{
    int i, el;

    if (!ent)
	return;
    el = 0;
    for (i = 1; i <= ent; i++)
	if (tuple[i].ff != NULL) {
	    if (!el)
		(void) fprintf(output, "sets\n");
	    (void) fprintf(output, "\tof ");
	    if (tuple[i].ind.key == 0)
		(void) fprintf(output, "atoms\n\t\t");
	    else {
		print_freeformat(tuple[i].ind.table,
				 tuple[i].ind.key);
		(void) fprintf(output, "\n\t\t");
	    }
	    print_ffname(tuple[i].ff, SET, i);
	    (void) fprintf(output, " = ");
	    if (tuple[i].u_tag)
		print_s_term(&tuple[i].construct.set_term);
	    else
		print_freeformat(tuple[i].construct.sort.table,
				 tuple[i].construct.sort.key);
	    printline();
	    el++;
	}
    if (el)
	printline();
}

void print_com_section(ent, tuple)
    unsigned int ent;
    com_tuple *tuple;
{
    int i;
    union terms ut[3];

    if (!ent)
	return;
    (void) fprintf(output, "communications\n");
    for (i = 1; i <= ent; i++)
	if (tuple[i].ff != NULL) {
	    (void) fflush(output);
	    (void) fprintf(output, "\t");
	    print_ae_term(&tuple[i].aet[0]);
	    (void) fprintf(output, " | ");
	    print_ae_term(&tuple[i].aet[1]);
	    (void) fprintf(output, " = ");
	    print_ae_term(&tuple[i].aet[2]);
	    (void) fflush(output);
	    ut[0].ae_t = &tuple[i].aet[0];
	    ut[1].ae_t = &tuple[i].aet[1];
	    ut[2].ae_t = &tuple[i].aet[2];
	    list_placeholders((arity) 3, ut, " for ");
	    printline();
	}
    printline();
}

int check_mod_type(ff, c, table, key)
    freeformat ff;
    char c;
    tabletype table;
    keytype key;
{
    unsigned int ent;
    keytype i;
    char *p;
    unsigned int onr, mnr;

    p = field_extract("o", ff, table, key);
    if (sscanf(p, "%u", &onr) != 1)
	onr = 0;
    ent = current_module->entries_table[ADM];
    for (i = 1; i <= ent; i++)
	if (current_module->adm[i].ff != NULL) {
	    if (!(p = field_extract("m", current_module->adm[i].ff, ADM, i)))
		continue;
	    if (sscanf(p, "%u", &mnr) != 1)
		mnr = 0;
	    if (onr != mnr)
		continue;
	    p = field_extract("t", current_module->adm[i].ff, ADM, i);
	    while (*p == ' ')
		p++;
	    if (*p == c)
		return (1);
	}
    return (0);
}

void print_var_section(ent, tuple, type)
    unsigned int ent;
    var_tuple *tuple;
    char type;
{
    int i, el;

    if (!ent)
	return;
    el = 0;
    for (i = 1; i <= ent; i++)
	if (tuple[i].ff != NULL) {
	    if (MarkVariable[i])
		continue;
	    if (!check_mod_type(tuple[i].ff, type, VAR, i))
		continue;
	    if (!el)
		(void) fprintf(output, "variables\n");
	    (void) fprintf(output, "\t");
	    print_ffname(tuple[i].ff, VAR, i);
	    (void) fprintf(output, "\t:\t-> ");
	    print_freeformat(tuple[i].ind.table, tuple[i].ind.key);
	    printline();
	    el++;
	}
    if (el)
	printline();
}

void print_equation(aet1, aet2)
    ae_term *aet1;
    ae_term *aet2;
{
    print_ae_term(aet1);
    (void) fprintf(output, " = ");
    print_ae_term(aet2);
}

void print_equ_section(ent, tuple)
    unsigned int ent;
    equ_tuple *tuple;
{
    int i, j;
    char *s;

    if (!ent)
	return;
    (void) fprintf(output, "equations\n");
    for (i = 1; i <= ent; i++)
	if (tuple[i].ff != NULL) {
	    s = field_extract("n", tuple[i].ff, EQU, i);
	    if (*s == '[') /* old TIL-code */
		(void) fprintf(output, "%s", s);
	    else
		(void) fprintf(output, "[%s]", s);
	    (void) fprintf(output, "\t");
	    print_equation(&tuple[i].aet1, &tuple[i].aet2);
	    if (tuple[i].a) {
		for (j = 0; j < tuple[i].a; j++) {
		    if (j)
			(void) fprintf(output, ",\n\t\t\t");
		    else
			(void) fprintf(output, "    when\n\t\t\t");
		    print_equation(&tuple[i].guard[j].aet1,
				   &tuple[i].guard[j].aet2);
		}
	    }
	    printline();
	}
    printline();
}

static int precedence(fun)
char fun;
{
    switch (fun) {
    case ALT:
	return(1);
    case MRG:
    case STAR:
    case SHARP:
	return(2);
    case SEQ:
	return(3);
    case AET:
	return(5); /* highest */
    default:
	return(4);
    }
}

void print_process_expr(pe)
    struct process_expr *pe;
{
    struct process_expr *pexpr;
    int hook;

    switch (pe->fun) {
    case SKP:
	(void) fprintf(output, "skip");
	if (Option_skipnr)
	    (void) fprintf(output, "<%u>", pe->proc_expr.nr);
	break;
    case ALT:
	print_process_expr(&pe->proc_expr.pe3.pe[0]);
	(void) fprintf(output, " + ");
	print_process_expr(&pe->proc_expr.pe3.pe[1]);
	break;
    case SEQ:
	if (precedence(pe->proc_expr.pe3.pe[0].fun) < precedence(pe->fun)) {
	    hook = 1;
	    (void) fprintf(output, "(");
	} else
	    hook = 0;
	print_process_expr(&pe->proc_expr.pe3.pe[0]);
	if (hook)
	    (void) fprintf(output, ")");
	fprintf(output, " . ");
	if (precedence(pe->proc_expr.pe3.pe[1].fun) <= precedence(pe->fun)) {
	    hook = 1;
	    (void) fprintf(output, "(");
	} else
	    hook = 0;
	print_process_expr(&pe->proc_expr.pe3.pe[1]);
	if (hook)
	    (void) fprintf(output, ")");
	break;
    case PAR:
	if (precedence(pe->proc_expr.pe3.pe[0].fun) < precedence(pe->fun)) {
	    hook = 1;
	    (void) fprintf(output, "(");
	} else
	    hook = 0;
	print_process_expr(&pe->proc_expr.pe3.pe[0]);
	if (hook)
	    (void) fprintf(output, ")");
	fprintf(output, " || ");
	if (precedence(pe->proc_expr.pe3.pe[1].fun) <= precedence(pe->fun)) {
	    hook = 1;
	    (void) fprintf(output, "(");
	} else
	    hook = 0;
	print_process_expr(&pe->proc_expr.pe3.pe[1]);
	if (hook)
	    (void) fprintf(output, ")");
	break;
    case INTR:
    case DISR:
	if (pe->fun == INTR)
	    fprintf(output, "interrupt(");
	else
	    fprintf(output, "disrupt(");
	print_process_expr(&pe->proc_expr.pe3.pe[0]);
	fprintf(output, ", ");
	print_process_expr(&pe->proc_expr.pe3.pe[1]);
	fprintf(output, ")");
	break;
    case STAR:
    case SHARP:
	if (precedence(pe->proc_expr.pe3.pe[0].fun) < precedence(pe->fun)) {
	    hook = 1;
	    (void) fprintf(output, "(");
	} else
	    hook = 0;
	print_process_expr(&pe->proc_expr.pe3.pe[0]);
	if (hook)
	    (void) fprintf(output, ")");
	if (pe->fun == STAR)
	    fprintf(output, " * ");
	else
	    fprintf(output, " # ");
	if (precedence(pe->proc_expr.pe3.pe[1].fun) <= precedence(pe->fun)) {
	    hook = 1;
	    (void) fprintf(output, "(");
	} else
	    hook = 0;
	print_process_expr(&pe->proc_expr.pe3.pe[1]);
	if (hook)
	    (void) fprintf(output, ")");
	break;
    case SUM:
	(void) fprintf(output, "sum(");
	print_placeholder(pe->proc_expr.pe1.ind.key);
	(void) fprintf(output, ", ");
	print_process_expr(pe->proc_expr.pe1.pe);
	(void) fprintf(output, ")");
	break;
    case MRG:
	(void) fprintf(output, "merge(");
	print_placeholder(pe->proc_expr.pe1.ind.key);
	(void) fprintf(output, ", ");
	print_process_expr(pe->proc_expr.pe1.pe);
	(void) fprintf(output, ")");
	break;
    case ENC:
	(void) fprintf(output, "encaps(");
	if (pe->proc_expr.pe1.ind.table == SOR && pe->proc_expr.pe1.ind.key
	    == 0)
	    fprintf(output, "atoms");
	else
	    print_freeformat(pe->proc_expr.pe1.ind.table,
		pe->proc_expr.pe1.ind.key);
	(void) fprintf(output, ", ");
	print_process_expr(pe->proc_expr.pe1.pe);
	(void) fprintf(output, ")");
	break;
    case HID:
	(void) fprintf(output, "hide(");
	if (pe->proc_expr.pe1.ind.table == SOR && pe->proc_expr.pe1.ind.key
	    == 0)
	    fprintf(output, "atoms");
	else
	    print_freeformat(pe->proc_expr.pe1.ind.table,
		pe->proc_expr.pe1.ind.key);
	(void) fprintf(output, ", ");
	print_process_expr(pe->proc_expr.pe1.pe);
	(void) fprintf(output, ")");
	break;
    case DLK:
	(void) fprintf(output, "delta");
	break;
    case IF:
	(void) fprintf(output, "[");
	print_ae_term(&pe->proc_expr.pe4.aex);
	(void) fprintf(output, " = ");
	print_ae_term(&pe->proc_expr.pe4.aey);
	(void) fprintf(output, "]->");
	pexpr = pe->proc_expr.pe4.pe;
	if (pexpr->fun == ALT || pexpr->fun == SEQ ||
		pexpr->fun == PAR) {
	    hook = 1;
	    (void) fprintf(output, "(");
	} else
	    hook = 0;
	print_process_expr(pe->proc_expr.pe4.pe);
	if (hook)
	    (void) fprintf(output, ")");
	break;
    case PRIO:
	(void) fprintf(output, "prio(");
	print_sorlist(pe->proc_expr.pe5.sets.a, pe->proc_expr.pe5.sets.indlist,
	    " > ");
	(void) fprintf(output, ", ");
	print_process_expr(pe->proc_expr.pe5.pe);
	(void) fprintf(output, ")");
	break;
    case AET:
	print_ae_term(&pe->proc_expr.pe2);
	break;
    }
}

void print_def_section(ent, tuple)
    unsigned int ent;
    def_tuple *tuple;
{
    int i;

    if (!ent)
	return;
    (void) fprintf(output, "definitions\n");
    for (i = 1; i <= ent; i++)
	if (tuple[i].ff != NULL) {
	    (void) fprintf(output, "\t");
	    print_ae_term(&tuple[i].ae_t);
	    (void) fprintf(output, " = ");
	    print_process_expr(&tuple[i].p_expr);
	    printline();
	}
    printline();
}
