#include <ctype.h>
#include <stdio.h>
#include <string.h>

#ifndef READXTIL
#include "tiltype.h"
#endif

#include "psf_prototype.h"
#include "psf_exits.h"
#include "psf_malloc.h"
#include "psf_standards.h"

#include "readexp.h"

extern FILE *input;
extern char *progname;
extern char *filename;

int lineno;
int linepos;

#ifdef SKIP_COMMENT_HAS_BEEN_REPAIRED
#ifndef READXTIL
/*
 * This routine tries to ungetc() both characters,when a '-' not followed by
 * another '-' has been read.  Its caller, skip_space, then tries to ungetc()
 * the '-' again.
 * If this routine didn't try to ungetc() the '-', skip_space would still fail
 * when it tries to ungetc() a character across a 8192-byte block boundary.
 *
 * Solution: make 1 character of look-ahead available in a global variable, so
 * that skip_comment can use the one character pushback that ungetc() provides.
 */

bool skip_comment()
{
    int i;

    i = getc(input);
    linepos++;

#ifdef READDEBUG
    (void) fprintf(stderr, "-");
    (void) fprintf(stderr, "%c", i);
#endif	/* READDEBUG */

    if (i == EOF)
	return TRUE;
    if (i != '-') {
	PSF_ASSERT(ungetc(i, input)!=EOF);
	PSF_ASSERT(ungetc('-', input)!=EOF);	/* one day, this will fail */
	linepos -= 2;
	return FALSE;
    }
    while (i != '\n' && i != EOF) {
	i = getc(input);
	linepos++;

#ifdef READDEBUG
	(void) fprintf(stderr, "%c", i);
#endif	/* READDEBUG */
    }
    lineno++;
    linepos = 0;
    return TRUE;
}
#endif	/* READXTIL */
#endif	/* SKIP_COMMENT_HAS_BEEN_REPAIRED */

void skip_space()
{
    int ic;
    bool space;

    space = TRUE;
    while (space) {
	ic = getc(input);
	linepos++;
	switch (ic) {
	case ' ':
	case '\t':
	case '\r':
	case '\f':

#ifdef READDEBUG
	    (void) fprintf(stderr, "%c", ic);
#endif	/* READDEBUG */

	    break;
	case '\n':
	    lineno++;
	    linepos = 0;

#ifdef READDEBUG
	    (void) fprintf(stderr, "%c", ic);
#endif	/* READDEBUG */

	    break;

#ifdef SKIP_COMMENT_HAS_BEEN_REPAIRED
#ifndef READXTIL
	case '-':
	    space = skip_comment();
	    break;
#endif	/* READXTIL */
#endif	/* SKIP_COMMENT_HAS_BEEN_REPAIRED */

	default:
	    space = FALSE;
	    break;
	}
    }

    if (ic != EOF) {
	PSF_ASSERT(ungetc(ic, input)!=EOF);
	linepos--;
    }
}

void read_char(c)
    char c;
{
    int ic;

    skip_space();
    ic = getc(input);
    linepos++;
    if (ic == EOF) {
	char s[sizeof "EOF found, 'y' expected"];
	(void) sprintf(s, "EOF found, \"%c\" expected", c);
	SYNTAX_ERROR(s);
	exit(EXIT_SYNTAX_ERR);
    }
    if (ic != c) {
	char s[sizeof "'x' found, 'y' expected"];
	(void) sprintf(s, "\"%c\" found, \"%c\" expected", ic, c);
	SYNTAX_ERROR(s);
	exit(EXIT_SYNTAX_ERR);
    }

#ifdef READDEBUG
    (void) fprintf(stderr, "%c", ic);
#endif

}

void read_string(s)
    char *s;

{
    int i;
    char *se;
    char *sl;
    char *sil;
    char *si;

    si = PSF_NMALLOC(char, strlen(s) + 1);
    skip_space();
    for (sl = s, sil = si;
	    *sl != '\0' && (i = getc(input)) != EOF && *sl == (char) i;
	    *sil++ = (char) i, sl++);
    if (*sl == '\0' || i == EOF)
	*sil = '\0';
    else {
	*sil++ = (char) i;
	*sil = '\0';
    }
    if (i == EOF) {
	se = PSF_NMALLOC(char, sizeof "EOF found, '' expected" + strlen(s));
	(void) sprintf(se, "EOF found, \"%s\" expected", s);
	SYNTAX_ERROR(se);
	exit(EXIT_SYNTAX_ERR);
    }
    if (strcmp(s, si) != 0) {
	se = PSF_NMALLOC(char,
		    sizeof "''found, '' expected" + strlen(si) + strlen(s));
	(void) sprintf(se, "\"%s\" found, \"%s\" expected", si, s);
	SYNTAX_ERROR(se);
	exit(EXIT_SYNTAX_ERR);
    }
    linepos += strlen(si);

#ifdef READDEBUG
    (void) fprintf(stderr, "%s", si);
#endif

}

unsigned read_nat()
{
    int ic;
    unsigned num;

    skip_space();
    ic = getc(input);
    linepos++;
    if (ic == EOF) {
	SYNTAX_ERROR("EOF found, natural expected");
	exit(EXIT_SYNTAX_ERR);
    }
    if (!isdigit(ic)) {
	char s[sizeof "'x' found, number expected"];
	(void) sprintf(s, "\"%c\" found, number expected", ic);
	SYNTAX_ERROR(s);
	exit(EXIT_SYNTAX_ERR);
    }
    num = 0;
    do {
	num = 10 * num + ic - '0';
	ic = getc(input);
	linepos++;
    } while (isdigit(ic));
    PSF_ASSERT(ungetc(ic, input)!=EOF);
    linepos--;

#ifdef READDEBUG
    (void) fprintf(stderr, "%u", num);
#endif

    return num;
}

#ifndef READXTIL
static char char_table[] = {
    AET, SKP, ALT, SEQ, PAR, SUM, MRG, ENC, HID, DLK, IF, INTR, DISR, PRIO,
    STAR, SHARP
};
static char *name_table[] = {
    "", "skip", "alt", "seq", "par", "sum",
    "merge", "encaps", "hide", "delta", "if",
    "interrupt", "disrupt", "prio", "star", "sharp"
};				/* below it is assumed (twice) that */
				/* "interrupt" */
				/* is the longest word in this table */

#define MAX_NAME_TABLE		(sizeof name_table/sizeof name_table[0])
#define MAX_TOKEN_LENGTH	(sizeof "<interrupt>" - 1)

static void syntax_error_in_exp_name(found, quotes)
    char *found;
    bool quotes;
{
    char* buf = PSF_NMALLOC(char, sizeof "found '', expecting one of" +
		    strlen(found) + MAX_NAME_TABLE * sizeof " '<interrupt>',"
		    + 2);
    (void) sprintf(buf, "found \"%s, expecting one of", found);
    (void) strcpy(buf, "found ");
    if (quotes) {
	(void) strcat(buf, "\"");
	(void) strcat(buf, found);
	(void) strcat(buf, "\"");
    } else
	(void) strcat(buf, found);
    (void) strcat(buf, ", expecting one of \"<skip>\", \"<alt>,\", ");
    (void) strcat(buf, "\"<seq>,\", \"<par>,\",\n\"<sum>\", \"<merge>\", ");
    (void) strcat(buf, "\"<encaps>\", \"<hide>\", \"<delta>\", \"<if>\", ");
    (void) strcat(buf, "\"<interrupt>\", \"<disrupt>\", \"<star>\", ");
    (void) strcat(buf, "\"<sharp>\"\n...");
    SYNTAX_ERROR(buf);
    exit(EXIT_SYNTAX_ERR);
}

char read_exp_name()
{
    char s[MAX_TOKEN_LENGTH];
    int i, ic;

    skip_space();
    i = 0;
    do {
	ic = getc(input);

#ifdef READDEBUG
	if (islower(ic) && i < MAX_TOKEN_LENGTH)
	    (void) fprintf(stderr, "%c", ic);
#endif

	linepos++;
	s[i++] = ic;
    } while (islower(ic) && i < MAX_TOKEN_LENGTH);
				/* note that islower(EOF) is false */
    if (ic == EOF)
	syntax_error_in_exp_name("EOF", FALSE);
    if (ic != '>' && ic != ',') {
	static char t[] = "'x'";
	t[1] = ic;
	syntax_error_in_exp_name(t, FALSE);
    }
    s[i - 1] = '\0';
    PSF_ASSERT(ungetc(ic, input)!=EOF);
    linepos--;
    for (i = 0; i < MAX_NAME_TABLE; i++) {
	if (strcmp(s, name_table[i]) == 0)
	    return (char_table[i]);
    }
    syntax_error_in_exp_name(s, TRUE);

    return '\0';		/* to shut up lint */
}

#endif
