#include <ctype.h>
#include <setjmp.h>
#include <stdio.h>
#include <string.h>

#include "psf_prototype.h"
#include "psf_exits.h"
#include "psf_malloc.h"
#include "psf_standards.h"

#include "readexp.h"
#include "tiltype.h"
#include "readtil.h"
#include "readtiltype.h"

bool no_illegal_chars = FALSE;
bool psf_double_quotes = FALSE;

static struct module *current_module;

static char_map def_char_map[] = {
    {' ', "'spa'"},
    {'!', "'exc'"},
    {'"', "'dqu'"},
    {'#', "'hsh'"},
    {'$', "'dlr'"},
    {'%', "'prc'"},
    {'&', "'amp'"},
    {'\'', "'squ'"},
    {'(', "'lpa'"},
    {')', "'rpa'"},
    {'*', "'ast'"},
    {'+', "'pls'"},
    {'.', "'dot'"},
    {',', "'cma'"},
    {'/', "'fsl'"},
    {':', "'col'"},
    {';', "'sco'"},
    {'<', "'lth'"},
    {'=', "'eql'"},
    {'>', "'grt'"},
    {'?', "'que'"},
    {'@', "'ats'"},
    {'[', "'lbr'"},
    {'\\', "'bsl'"},
    {']', "'rbr'"},
    {'^', "'hat'"},
    {'_', "'und'"},
    {'`', "'bqu'"},
    {'{', "'lac'"},
    {'|', "'bar'"},
    {'}', "'rac'"},
    {'~', "'tld'"},
};

#define REMAP_SIZE (sizeof(def_char_map)/sizeof(char_map))
#define ENTRY_SIZE 6		/* length of longest string in def_char_map,
				 * plus 1 */

static char remap_buffer[ENTRY_SIZE];

static char *remap(c)
    char c;
{
    int i;

    for (i = 0; i < REMAP_SIZE; i++)
	if (def_char_map[i].c == c)
	    return def_char_map[i].s;
    (void) sprintf(remap_buffer, "'%3.3u'", (unsigned char) c);
    return remap_buffer;
}

#ifdef NEWFF
static char *remap_info(info)
    char *info;
{
    char *ip, *bufp, *tmp, *remap_info_buffer;
    bool start_point;
    bool end_point;

    start_point = FALSE;
    end_point = FALSE;
    if (info == NULL)
	return NULL;
    remap_info_buffer = PSF_NMALLOC(char, ENTRY_SIZE * strlen(info) + 1);
    for (ip = info, bufp = remap_info_buffer; *ip; ip++) {
	if (isalnum(*ip))
	    *bufp++ = *ip;
	else {
	    if (*ip == '_')
		if (ip == info) {
		    tmp = "_.";
		    end_point = TRUE;
		} else if (*(ip + 1) == '\0') {
		    if (!end_point)
			start_point = end_point = TRUE;
		    tmp = "";
		} else
		    tmp = remap('_');
	    else
		tmp = remap(*ip);
	    while (*tmp)
		*bufp++ = *tmp++;
	}
    }
    if (end_point) {
	*bufp++ = '.';
	*bufp++ = '_';
    }
    *bufp = '\0';
    if (start_point) {
	tmp = PSF_NMALLOC(char, strlen(remap_info_buffer) + 2);
	tmp[0] = '.';
	strcpy(tmp + 1, remap_info_buffer);
	PSF_FREE(remap_info_buffer);
	remap_info_buffer = tmp;
    } else
	remap_info_buffer = PSF_REALLOC(remap_info_buffer, char,
					strlen(remap_info_buffer) + 1);
    PSF_FREE(info);
    return remap_info_buffer;
}

#else

#define IDLE 0			/* before first '<' */
#define TAG  1			/* between '<' and '>' */
#define IDENT 2			/* between '>' and '<' or '\0'  */

static char *remap_ff(ff)
    char *ff;
{
    char *ffp, *bufp, *tmp, *remap_ff_buffer;
    int state;

    if (ff == NULL)
	return NULL;
    remap_ff_buffer = PSF_NMALLOC(char, ENTRY_SIZE * strlen(ff) + 1);
    state = IDLE;
    for (ffp = ff, bufp = remap_ff_buffer; *ffp != '\0'; ffp++) {
	switch (state) {
	case IDLE:
	    *bufp++ = *ffp;
	    if (*ffp == '<')
		state = TAG;
	    break;
	case TAG:
	    *bufp++ = *ffp;
	    if (*ffp == '>')
		state = IDENT;
	    break;
	case IDENT:
	    if (*ffp == '_' &&
	     (*(ffp - 1) == '>' || *(ffp + 1) == '<' || *(ffp + 1) == '\0')) {
		if (*(ffp - 1) == '>') {
		    *bufp++ = *ffp;
		    *bufp++ = '.';
		} else {
		    *bufp++ = '.';
		    *bufp++ = *ffp;
		}
	    } else if (*ffp == '<') {
		*bufp++ = *ffp;
		state = TAG;
	    } else if (isalnum(*ffp))
		*bufp++ = *ffp;
	    else {
		tmp = remap(*ffp);
		while (*tmp)
		    *bufp++ = *tmp++;
	    }
	    break;
	default:
	    PSF_ASSERT(THIS_CANNOT_HAPPEN);
	}
    }
    *bufp = '\0';
    remap_ff_buffer = PSF_REALLOC(remap_ff_buffer, char,
				  strlen(remap_ff_buffer) + 1);
    PSF_FREE(ff);
    return remap_ff_buffer;
}
#endif				/* NEWFF */

void set_read_current_module(mod)
    struct module *mod;
{
    current_module = mod;
}

void read_index(index)
    struct indextype *index;
{
    bool undefined;

    read_char('[');
    index->table = read_nat();
    read_char('.');
    index->key = read_nat();
    read_char(']');

    undefined = FALSE;
    if (index->table != SOR & index->key != 0) {
	if (current_module->entries_table[index->table] < index->key)
	    undefined = TRUE;
	else
	    switch (index->table) {
	    case SOR:
		if (current_module->sor[index->key].ff == NULL)
		    undefined = TRUE;
		break;
	    case FUN:
		if (current_module->fun[index->key].ff == NULL)
		    undefined = TRUE;
		break;
	    case ATM:
		if (current_module->atm[index->key].ff == NULL)
		    undefined = TRUE;
		break;
	    case PRO:
		if (current_module->pro[index->key].ff == NULL)
		    undefined = TRUE;
		break;
	    case SET:
		if (current_module->set[index->key].ff == NULL)
		    undefined = TRUE;
		break;
	    case COM:
		if (current_module->com[index->key].ff == NULL)
		    undefined = TRUE;
		break;
	    case VAR:
		if (current_module->var[index->key].ff == NULL)
		    undefined = TRUE;
		break;
	    case EQU:
		if (current_module->equ[index->key].ff == NULL)
		    undefined = TRUE;
		break;
	    case DEF:
		if (current_module->def[index->key].ff == NULL)
		    undefined = TRUE;
		break;
	    default:
		{
		    char s[sizeof "unknown table number '1234567890'"];
		    (void) sprintf(s,
				"unknown table number '%u'", index->table);
		    SYNTAX_ERROR(s);
		    exit(EXIT_SYNTAX_ERR);
		}
	    }
	if (undefined) {

#ifndef READDBU
	    if (index->table != VAR && index->table != SET &&
		    index->table != SOR) {
		char s[sizeof "tuple [0.1234567890] not defined"];
		(void) sprintf(s, "tuple [%u.%u] not defined",
			    index->table, index->key);
		SYNTAX_ERROR(s);
		exit(EXIT_SYNTAX_ERR);
	    } else if (index->table != VAR && index->table != SET) {
		char s[sizeof "warning tuple [0.1234567890] not (yet) defined"];
		(void) sprintf(s, "warning tuple [%u.%u] not (yet) defined",
			    index->table, index->key);
		(void) fprintf(stderr,
			       "%s: %s in file %s, line %u, position %u\n",
			       progname, s, filename, lineno, linepos);
	    }
#else
	    char s[sizeof "tuple [0.1234567890] not defined"];
	    (void) sprintf(s, "tuple [%u.%u] not defined",
			index->table, index->key);
	    SYNTAX_ERROR(s);
	    exit(EXIT_SYNTAX_ERR);
#endif

	}
    }
}

struct indextype *read_indexarr(nr)
    unsigned int nr;
{
    int i;
    struct indextype *indlist;

    if (!nr)
	return (NULL);
    indlist = PSF_NMALLOC(struct indextype, nr);
    for (i = 0; i < nr; i++) {
	read_index(&indlist[i]);
    }
    return (indlist);
}

void read_indexlist(il)
    struct indexlist *il;
{
    il->a = read_nat();
    il->indlist = read_indexarr(il->a);
}

#define NO_SKIPPING 0
#define SKIPPING 1
#define BACK_SKIPPING 2

char *adjust_spaces(ff)
    char *ff;
{
    char *ffp, *s, *sp;
    int state;

    state = NO_SKIPPING;
    s = psf_strdup(ff);
    for (ffp = ff, sp = s; *ffp != '\0'; ffp++)
	switch (state) {
	case NO_SKIPPING:
	    *sp++ = *ffp;
	    if (*ffp == '>')
		state = SKIPPING;
	    break;
	case SKIPPING:
	    if (!isspace(*ffp)) {
		*sp++ = *ffp;
		state = BACK_SKIPPING;
	    }
	    break;
	case BACK_SKIPPING:
	    if (*ffp == '<') {
		state = NO_SKIPPING;
		while (isspace(*--sp));
		sp++;
	    }
	    *sp++ = *ffp;
	    break;
	default:
	    PSF_ASSERT(THIS_CANNOT_HAPPEN);
	}
    while (isspace(*--sp));
    *++sp = '\0';
    PSF_FREE(ff);
    return s;
}

#ifdef NEWFF
char *readtag()
{
    char *s;
    int i, bufsize;
    int ic;

    bufsize = 1;
    s = PSF_NMALLOC(char, bufsize + 1);
    i = 0;
    while ((ic = getc(input)) != '>') {
	if (ic == EOF || ic == '}')
	    read_char('>');

#ifdef READDEBUG
	(void) fprintf(stderr, "%c", ic);
#endif

	if (psf_double_quotes && ic == '"') {
	    linepos++;
	    continue;
	}
	if (ic == '\n') {
	    lineno++;
	    linepos = 0;
	} else
	    linepos++;
	s[i++] = ic;
	if (i >= bufsize) {
	    bufsize *= 2;
	    s = PSF_REALLOC(s, char, bufsize + 1);
	}
    }
    s[i] = '\0';

#ifdef READDEBUG
    (void) fprintf(stderr, "%c", ic);
#endif

    s = PSF_REALLOC(s, char, i + 1);
    return (s);
}

char *read_info()
{
    char *s;
    int i, bufsize;
    int ic;

    skip_space();
    bufsize = 1;
    s = PSF_NMALLOC(char, bufsize + 1);
    i = 0;
    while ((ic = getc(input)) != '}' && ic != '<') {
	if (ic == EOF)
	    read_char('}');

#ifdef READDEBUG
	(void) fprintf(stderr, "%c", ic);
#endif

	if (psf_double_quotes && ic == '"') {
	    linepos++;
	    continue;
	}
	if (ic == '\n') {
	    lineno++;
	    linepos = 0;
	} else
	    linepos++;
	s[i++] = ic;
	if (i >= bufsize) {
	    bufsize *= 2;
	    s = PSF_REALLOC(s, char, bufsize + 1);
	}
    }

#ifdef READDEBUG
    (void) fprintf(stderr, "%c", ic);
#endif

    while (--i >= 0 && isspace(s[i]))
	;
    s[i+1] = '\0';
    s = PSF_REALLOC(s, char, i + 2);
    PSF_ASSERT(ungetc(ic, input)!=EOF);
    linepos--;

    if (no_illegal_chars)
	return remap_info(s);
    else
	return (s);
}

freeformat read_freeformat2()
{
    freeformat ff;
    int ic;

    do {
	ic = getc(input);
	if (ic == '\n') {
	    lineno++;
	    linepos = 0;
	} else
	    linepos++;

#ifdef READDEBUG
	(void) fprintf(stderr, "%c", ic);
#endif

    } while (ic != EOF && ic != '}' && ic != '<');
    if (ic == '}')
	return NULL;
    if (ic == EOF)
	read_char('}');
    ff = PSF_MALLOC(struct new_freeformat);
    ff->tag = readtag();
    ff->info = read_info();
    ff->next = read_freeformat2();
    return ff;

}

freeformat
read_freeformat()
{
    freeformat ff;

    ff = PSF_MALLOC(struct new_freeformat);
    ff->tag = "";
    ff->info = "";
    read_char('{');
    ff->next = read_freeformat2();
    if (ff->next == NULL)
	return ff;
    else
	return ff->next;
}

#else

freeformat
read_freeformat()
{
    char *s;
    int i, bufsize;
    int ic;

    read_char('{');
    skip_space();
    bufsize = 1;
    s = PSF_NMALLOC(char, bufsize + 1);
    i = 0;
    while ((ic = getc(input)) != '}') {
	if (ic == EOF)
	    read_char('}');

#ifdef READDEBUG
	(void) fprintf(stderr, "%c", ic);
#endif

	if (psf_double_quotes && ic == '"') {
	    linepos++;
	    continue;
	}
	if (ic == '\n') {
	    lineno++;
	    linepos = 0;
	} else
	    linepos++;
	s[i++] = ic;
	if (i >= bufsize) {
	    bufsize *= 2;
	    s = PSF_REALLOC(s, char, bufsize + 1);
	}
    }
    s[i] = '\0';

#ifdef READDEBUG
    (void) fprintf(stderr, "%c", ic);
#endif

    s = PSF_REALLOC(s, char, i + 1);
    s = adjust_spaces(s);
    if (no_illegal_chars)
	return remap_ff(s);
    else
	return (s);
}

#endif				/* NEWFF */

void read_ae_term(term)
    ae_term *term;
{
    unsigned int nr;
    int i;

    skip_space();
    term->t = getc(input);
    PSF_ASSERT(ungetc(term->t, input)!=EOF);
    if (term->t == TUPLE) {	/* if the look-ahead character is TUPLE, it */
	read_char('<');		/* had better be '<', since TUPLE == '<' */
	term->a = read_nat();
	read_char(',');
	term->ae_list = PSF_NMALLOC(ae_term, term->a);
	for (i = 0; i < term->a; i++)
	    read_ae_term(&(term->ae_list[i]));
	read_char('>');
    } else {
	if (term->t != TERM) {
	    char s[sizeof "'x' found, '[' or '<' expected"];
	    (void) sprintf(s, "\"%c\" found, \"%c\" or \"%c\" expected",
			   term->t, TERM, TUPLE);
	    SYNTAX_ERROR(s);
	    exit(EXIT_SYNTAX_ERR);
	}
	read_index(&term->ind);
	if (term->ind.table == FUN)
	    nr = current_module->fun[term->ind.key].sor_indlist.a;
	else if (term->ind.table == ATM)
	    nr = current_module->atm[term->ind.key].sor_indlist.a;
	else if (term->ind.table == PRO)
	    nr = current_module->pro[term->ind.key].sor_indlist.a;
	else
	    nr = 0;
	term->a = nr;
	if (nr == 0) {
	    term->ae_list = NULL;
	    return;
	}
	read_char('(');
	term->ae_list = PSF_NMALLOC(ae_term, nr);

	for (i = 0; i < nr; i++)
	    read_ae_term(&(term->ae_list[i]));
	read_char(')');
    }
}

void read_s_term(term)
    struct s_term *term;
{
    int c;
    int i;

    read_char('<');
    term->fun = (char) getc(input);
    read_char(',');
    term->a = read_nat();
    read_char('>');
    read_char('(');

    term->u_tag = PSF_NMALLOC(int, term->a);
    term->arr = PSF_NMALLOC(union terms, term->a);
    for (i = 0; i < term->a; i++) {
	skip_space();
	c = getc(input);
	PSF_ASSERT(ungetc(c, input)!=EOF);
	if (c == '[') {
	    term->u_tag[i] = 0;
	    term->arr[i].ae_t = PSF_MALLOC(ae_term);
	    read_ae_term(term->arr[i].ae_t);
	} else {
	    term->u_tag[i] = 1;
	    term->arr[i].s_t = PSF_MALLOC(struct s_term);
	    read_s_term(term->arr[i].s_t);
	}
    }
    read_char(')');
}

static unsigned int skip_nr = 0;

void reset_skip_nr()
{
    skip_nr = 0;
}

void read_process_expr(p_expr)
    struct process_expr *p_expr;
{
    char fc;
    int i, c;
    arity arg;

    skip_space();
    c = getc(input);

    if (c == '[') {
	PSF_ASSERT(ungetc(c, input)!=EOF);
	p_expr->fun = AET;
	read_ae_term(&p_expr->proc_expr.pe2);
    } else {
	if (c != '<') {
	    char s[sizeof "'x' found, '[' or '<' expected"];
	    (void) sprintf(s, "\"%c\" found, \"[\" or \"<\" expected", c);
	    SYNTAX_ERROR(s);
	    exit(EXIT_SYNTAX_ERR);
	}

#ifdef READDEBUG
	(void) fprintf(stderr, "%c", c);
#endif

	fc = read_exp_name();
	p_expr->fun = fc;
	switch (fc) {
	case ALT:
	case SEQ:
	case PAR:
	case INTR:
	case DISR:
	case STAR:
	case SHARP:
	    read_char(',');
	    arg = read_nat();
	    read_char('>');
	    read_char('(');
	    p_expr->proc_expr.pe3.a = arg;
	    p_expr->proc_expr.pe3.pe =
		PSF_NMALLOC(struct process_expr, arg);
	    for (i = 0; i < arg; i++)
		read_process_expr(&p_expr->proc_expr.pe3.pe[i]);
	    read_char(')');
	    break;
	case SKP:
	    read_char('>');
	    p_expr->proc_expr.nr = skip_nr++;
	    break;
	case SUM:
	case MRG:
	case ENC:
	case HID:
	    read_char('>');
	    read_char('(');
	    read_index(&p_expr->proc_expr.pe1.ind);
	    p_expr->proc_expr.pe1.pe = PSF_MALLOC(struct process_expr);
	    read_process_expr(p_expr->proc_expr.pe1.pe);
	    read_char(')');
	    break;
	case DLK:
	    read_char('>');
	    break;
	case IF:
	    read_char('>');
	    read_char('(');
	    read_ae_term(&p_expr->proc_expr.pe4.aex);
	    read_ae_term(&p_expr->proc_expr.pe4.aey);
	    p_expr->proc_expr.pe4.pe = PSF_MALLOC(struct process_expr);
	    read_process_expr(p_expr->proc_expr.pe4.pe);
	    read_char(')');
	    break;
	case PRIO:
	    read_char('>');
	    read_char('(');
	    read_indexlist(&p_expr->proc_expr.pe5.sets);
	    p_expr->proc_expr.pe5.pe = PSF_MALLOC(struct process_expr);
	    read_process_expr(p_expr->proc_expr.pe5.pe);
	    read_char(')');
	    break;
	}
    }
}

void read_sor_tuple(sor_t)
    sor_tuple *sor_t;
{
    sor_t->ff = read_freeformat();
}

void read_fun_tuple(fun_t)
    fun_tuple *fun_t;
{
    read_indexlist(&fun_t->sor_indlist);
    read_indexlist(&fun_t->return_list);
    fun_t->ff = read_freeformat();
}

void read_atm_tuple(atm_t)
    atm_tuple *atm_t;
{
    read_indexlist(&atm_t->sor_indlist);
    atm_t->ff = read_freeformat();
}

void read_pro_tuple(pro_t)
    pro_tuple *pro_t;
{
    read_atm_tuple(pro_t);
}

void read_set_tuple(set_t)
    set_tuple *set_t;
{
    char c;

    read_index(&set_t->ind);
    skip_space();
    c = getc(input);
    PSF_ASSERT(ungetc(c, input)!=EOF);
    if (c == '[') {
	set_t->u_tag = 0;
	read_index(&set_t->construct.sort);
    } else {
	set_t->u_tag = 1;
	read_s_term(&set_t->construct.set_term);
    }
    set_t->ff = read_freeformat();
}

void read_com_tuple(com_t)
    com_tuple *com_t;
{
    read_ae_term(&com_t->aet[0]);
    read_ae_term(&com_t->aet[1]);
    read_ae_term(&com_t->aet[2]);
    com_t->ff = read_freeformat();
}

void read_var_tuple(var_t)
    var_tuple *var_t;
{
    read_index(&var_t->ind);
    var_t->ff = read_freeformat();
}

void read_equ_tuple(equ_t)
    equ_tuple *equ_t;
{
    int ch, i;

    read_ae_term(&equ_t->aet1);
    read_char('=');
    read_ae_term(&equ_t->aet2);
    skip_space();
    ch = getc(input);
    if (ch != '<') {
	PSF_ASSERT(ungetc(ch, input)!=EOF);
	equ_t->a = 0;
    } else {
	read_char('=');
	equ_t->a = read_nat();
	equ_t->guard = PSF_NMALLOC(struct equation, equ_t->a);
	for (i = 0; i < equ_t->a; i++) {
	    read_ae_term(&equ_t->guard[i].aet1);
	    read_char('=');
	    read_ae_term(&equ_t->guard[i].aet2);
	    if (i < equ_t->a - 1) {
		read_char(',');
	    }
	}
    }
    equ_t->ff = read_freeformat();
}

void read_def_tuple(def_t)
    def_tuple *def_t;
{
    read_ae_term(&def_t->ae_t);
    read_char('=');
    read_process_expr(&def_t->p_expr);
    def_t->ff = read_freeformat();
}

void read_adm_tuple(adm_t)
    adm_tuple *adm_t;
{
    adm_t->ff = read_freeformat();
}

#ifdef	EXTTIL
void read_dataed_tuple(dataed_t)
    dataed_tuple *dataed_t;
{
    read_ae_term(&dataed_t->aet);
    dataed_t->ff = read_freeformat();
}

void read_proced_tuple(proced_t)
    proced_tuple *proced_t;
{
    read_process_expr(&proced_t->p_expr);
    proced_t->ff = read_freeformat();
}

#endif
