#include <assert.h>
#include <signal.h>
#include <sys/wait.h>

#include "psf_prototype.h"
#include "psf_exits.h"
#include "psf_standards.h"
#include "main.h"

static sigset_t signal_mask;

static int catching[] = {SIGHUP, SIGINT, SIGABRT, SIGPIPE, SIGXCPU, SIGXFSZ, 0};

static void handler(sig)
    int sig;
{
    (void) fprintf(stderr, "%s: interrupted\n", progname);
    exit_status = EXIT_CTRL_C;
    cleanup();

#ifdef lint
    (void) sig;
#endif
}

void set_signal_handlers()
{
    int i;
    void (*old_handler)();

    sysCall(sigemptyset(&signal_mask), "initialize", "signal mask");
    for (i=0; catching[i]; i++)
        sysCall(sigaddset(&signal_mask, catching[i]), "add signal", "to mask");

    for (i=0; catching[i]; i++) {
        sysCall((long) (old_handler = signal(catching[i], SIG_IGN)),
                                "retrieve", "signal handler");
        if (old_handler != SIG_IGN)
            sysCall((long) signal(catching[i], handler),
                                "install", "signal handler");
    }
}

void signals_block()
{
    sigset_t old_mask;

    sysCall(sigprocmask(SIG_BLOCK, &signal_mask, &old_mask),
                "block", "signals");
}

void signals_unblock()
{
    sigset_t old_mask;

    sysCall(sigprocmask(SIG_UNBLOCK, &signal_mask, &old_mask),
                "unblock", "signals");
}

void do_wait(kid_name)
    char *kid_name;
{
    int status;

    sysCall(wait(&status), "wait for", kid_name);

    if (WIFEXITED(status)) {
	if (WEXITSTATUS(status) == EXIT_SUCCESS)
	    return;
    } else if (WIFSIGNALED(status)) {
	(void) fprintf(stderr, "%s: %s: ", progname, kid_name);
	switch (WTERMSIG(status)) {

#define	Case(sig,txt) case sig: (void)fprintf(stderr, txt); break

	    Case(SIGHUP,	"hangup");
	    Case(SIGINT,	"keyboard interrupt");
	    Case(SIGQUIT,	"quit signal");
	    Case(SIGILL,	"illegal instruction");
	    Case(SIGTRAP,	"trace trap");
	    Case(SIGABRT,	"abort");
#ifdef SIGEMT /* for Linux */
	    Case(SIGEMT,	"emulator trap");
#endif
	    Case(SIGFPE,	"arithmetic exception");
	    Case(SIGKILL,	"killed");
	    Case(SIGBUS,	"bus error");
	    Case(SIGSEGV,	"memory fault");
#ifdef SIGSYS /* for Linux */
	    Case(SIGSYS,	"bad system call");
#endif
	    Case(SIGPIPE,	"broken pipe");
	    Case(SIGALRM,	"alarm clock");
	    Case(SIGTERM,	"terminated");
	    Case(SIGURG,	"urgent i/o condition");
	    Case(SIGSTOP,	"stopped (signal)");
	    Case(SIGTSTP,	"stopped");
	    Case(SIGCONT,	"continued");
	    Case(SIGCHLD,	"child exited");
	    Case(SIGTTIN,	"stopped (tty input)");
	    Case(SIGTTOU,	"stopped (tty output)");
	    Case(SIGIO,		"i/o possible");
	    Case(SIGXCPU,	"cpu time limit exceeded");
	    Case(SIGXFSZ,	"file size limit exceeded");
	    Case(SIGVTALRM,	"virtual timer alarm");
	    Case(SIGPROF,	"profiling timer alarm");
	    Case(SIGWINCH,	"window size changed");

#ifdef SIGLOST
	    Case(SIGLOST,	"resource lost");
#endif

	    Case(SIGUSR1,	"user defined signal 1");
	    Case(SIGUSR2,	"user defined signal 2");

#undef	Case

	default:
	    assert(THIS_CANNOT_HAPPEN);
	}

/* seems to work different now, but NOT needed anyway
#ifndef WCOREFLG
#define WCOREFLG ((union wait *) & status)->w_coredump
#endif

	if (WCOREFLG)
	    (void) fprintf(stderr, " (core dumped)\n");
	else
*/
	    (void) fprintf(stderr, "\n");
    } else
	assert(THIS_CANNOT_HAPPEN);

    cleanup();
}
