#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/param.h>
#include <sys/types.h>

#include "psf_prototype.h"
#include "psf_exits.h"
#include "psf_malloc.h"
#include "psf_standards.h"
#include "version.h"
#include "main.h"
#include "imports.h"
#include "args.h"
#include "auxfunc.h"
#include "libs.h"
#include "process.h"
#include "split.h"
#include "tsort.h"
#include "all_mtil.h"
#include "all_itil.h"
#include "parcompiler.h"
#include "til.h"
#include "set.h"
#include "check.h"
#include "graphformat.h"

char *progname;
char cwd[MAXPATHLEN + 1];
unsigned l_cwd;

char *psfc_bindir;
char *psf_suffix;
unsigned l_psf_suffix;
char *til_suffix;
char *psf_tmpdir;
unsigned l_psf_tmpdir;
char *current_target = 0;
char *target_module = 0;
char *outputfile = 0;
static char *tilfile;
char *graphformat = 0;

int exit_status = EXIT_HELP;

static void version()
{
    (void) fprintf(stderr, "(toolbus)psf version %s, last compiled %s\n",
		   VERSION, __DATE__);
}

void cleanup()
{
    if (current_target) {
	if (unlink(current_target) == -1) {
	    if (errno != ENOENT) {
		(void) fprintf(stderr, "%s: can't remove ", progname);
		perror(current_target);
	    }
	} else {
	    (void) fprintf(stderr, "%s: %s removed\n", progname,
			   psf_basename(current_target));
	}
    }
    exit(exit_status);
}

static void init(argc, argv)
    int argc;
    char **argv;
{

#ifdef DEBUG
    extern int malloc_debug();

    (void) malloc_debug(2);
#endif

    if (argc == 0) {
	(void) fprintf(stderr, "who am I?\n");
	exit(EXIT_CMD_LINE_ERR);
    }
    progname = psf_basename(*argv);

    if (argc == 1)
	usage();

#ifdef SYSV
    stdioCall(getcwd(cwd, MAXPATHLEN),
			"name current directory (or a parent)", "");
#else /* SYSV */
    stdioCall(getcwd(cwd, MAXPATHLEN),
			"name current directory (or a parent)", "");
    /* Hmm, nowadays it is getcwd, but does this also work on BSD systems?
    stdioCall(getwd(cwd), "name current directory (or a parent)", "");
    */
#endif /* SYSV */

    (void) strcat(cwd, "/");
    l_cwd = strlen(cwd) + 1;
}

int main(argc, argv)
    int argc;
    char **argv;
{
    int i;

    init(argc, argv);
    parse_opts(argc, argv);
    if (option['v'])
	version();
    if (!target_module && (optind == argc))
	if (option['v'])
	    exit(EXIT_SUCCESS);
	else {
	    (void) fprintf(stderr,
		   "%s: give input file(s) or use the -m option\n", progname);
	    exit(EXIT_CMD_LINE_ERR);
	}
    parse_env();
    if (target_module)
	set_target(target_module);
    parse_args(argc, argv);
    if (!option['S']) {
	if (option['v'])
	    (void) fprintf(stderr, "searching libraries (when needed)\n");
	search_libraries();
	if (option['v'])
	    (void) fprintf(stderr, "sorting modules\n");
	tsort();
	if (target_module)
	    cut(target_module);
    }
    if (option['l']) {
	list_modules();
	exit(EXIT_SUCCESS);
    }
    /*
     * PSF Dependency Graph
     */
    if (graphformat) {
	if (outputfile)
	    stdioCall(freopen(outputfile, "w", stdout), "create", outputfile);
	if (GraphFormat(graphformat)) {
	    (void) fprintf(stderr, "%s: unknown graph-format \"%s\" for option -G\n",
		progname, graphformat);
	    exit(EXIT_CMD_LINE_ERR);
	}
	exit(EXIT_SUCCESS);
    }

    if (last_module == &module_list) {
	(void) fprintf(stderr, "%s: no modules found\n", progname);
	exit(EXIT_SYNTAX_ERR);
    }

    set_signal_handlers();

    split();
    if (option['S'])
	exit(EXIT_SUCCESS);
    if (option['x'] || option['X']) {
	if (option['v'])
	    (void) fprintf(stderr, "extracting modules\n");
	extract_modules(option['x']);
	exit(EXIT_SUCCESS);
    }
    sysCall(chdir(psf_tmpdir), "chdir to", psf_tmpdir);
    if (option['P'])
	parallel_compile(option['i'], option['c'], option['v']);
    else {
	all_mtil(option['i'], option['c'], option['v']);
	all_itil(option['i'], option['c'], option['v']);
    }

    {
	module m;
	char *in;
	char *s;
	m = (module) last_module;
	in = m->tnam;
	s = PSF_NMALLOC(char, strlen(m->name) + strlen(til_suffix) + 1);
	(void) strcpy(s, m->name);
	(void) strcat(s, til_suffix);
	m->tnam = s;
	if (!option['s'])
	    if (outputfile)
		mkAbsPath(tilfile, outputfile);
	    else
		absPath(tilfile, s);
	else
	    tilfile = s;
	if (must_recompile(in, tilfile, option['c'])) {
	    if (option['v'])
		(void) fprintf(stderr, "compiling %s\n", in);
	    til(in, tilfile);
	}
    }

    if (option['s']) {
	char *in;
	in = ((module) last_module)->tnam;
	if (outputfile)
	    mkAbsPath(tilfile, outputfile);
	else
	    absPath(tilfile, in);
	if (must_recompile(in, tilfile, option['c'])) {
	    if (option['v'])
		(void) fprintf(stderr, "compiling %s\n", in);
	    simpp(in, tilfile, option['v']);
	}
    }

    if (option['t']) {
	if (option['v'])
	    (void) fprintf(stderr, "checking equations in %s\n",
		psf_basename(tilfile));
	trs_check(tilfile);
    }
    if (option['c']) {
	if (option['v'])
	    (void) fprintf(stderr, "removing intermediate files\n");
	rm_tmpdir(option['s']);
    }
    return EXIT_SUCCESS;
}
