/*
 * routines to access the set table used by the PSF parser
 */

#include <stdio.h>
#include <string.h>
#include "psf_prototype.h"
#include "psf_malloc.h"
#include "global.h"
#include "tabtypes.h"

#include "main.h"
#include "yimports.h"

static SENODE *septr = NULL;
static int seindex = 1;
static SNODE *set_sort = NULL;	/* points to base sort for data sets, NULL for
				 * sets of atoms */

void init_set_table()
{
    septr = NULL;
}

BOOL set_present()
{
    return septr != NULL;
}

SENODE *lookup_set(id)
    ID id;
{
    SENODE *tmp;

    for (tmp = septr; tmp != NULL; tmp = tmp->next)
	if (strcmp(tmp->id, id) == 0)
	    return (tmp);

    return (NULL);
}

SENODE *simple_ins_set(id)
    ID id;
{
    SENODE *tmp;

    if (lookup_set(id) != NULL) {
	(void) fprintf(stderr, "%s: in module\"%s\", the set \"%s\" is used ",
		       progname, module_name, id);
	(void) fprintf(stderr, "before it is declared\n");
	errorseen = TRUE;
    }
    tmp = PSF_MALLOC(SENODE);

    tmp->index = seindex++;
    tmp->id = psf_strdup(id);
    tmp->tag = get_tag_flag();
    tmp->trm = NULL;
    tmp->sort = NULL;

    tmp->next = septr;
    septr = tmp;

    if (get_tag_flag() == PARAMETER)
	ins_par_item(SET, tmp->index);

    return (tmp);
}

SNODE *get_set_sort()
{
    return set_sort;
}

static void check_redefined_sort(id)
    ID id;
{
    SNODE *old_sort;

    if ((old_sort = lookup_sort(id)) != NULL) {
	if (old_sort->tag == IMPORTS) {
	    old_sort->tag = RDS;
	    old_sort->redefined_as = get_set_sort();
	}
    }
}

void ins_set(id, ptr, sptr)
    ID id;
    TNODE *ptr;
    SNODE *sptr;
{
    SENODE *tmp;

    push_tag_flag();
    set_tag_flag(IMPORTS);
    type_tree(ptr, SET);
    set_tag_flag(pop_tag_flag());

    /* if a set is used before in the same definition, type_tree will
     * define it, and lookup_set(id) won't return NULL
     */
    if (lookup_set(id) != NULL) {
	(void) fprintf(stderr, "%s: in module \"%s\", the set \"%s\" is used ",
		       progname, module_name, id);
	(void) fprintf(stderr, "before it is declared\n");
	errorseen = TRUE;
    }
    if (lookup_sort(id) != NULL) {
	(void) fprintf(stderr, "%s: in module \"%s\", \"%s\" is used ",
		       progname, module_name, id);
	(void) fprintf(stderr, "both as a set name and as a sort name\n");
	errorseen = TRUE;
    }

    clear_ph();
    tmp = PSF_MALLOC(SENODE);
    tmp->index = seindex++;
    tmp->id = psf_strdup(id);
    tmp->tag = get_tag_flag();
    tmp->trm = ptr;
    tmp->sort = sptr;
    tmp->next = septr;
    septr = tmp;

    check_redefined_sort(id);

    if (get_tag_flag() == PARAMETER)
	ins_par_item(SET, tmp->index);

}

void set_set_sort(id)
    ID id;
{
    if (id == NULL) {
	set_sort = NULL;
    } else {
	set_sort = lookup_sort(id);
	if (set_sort == NULL) {
	    ins_sort(id);
	    set_sort = lookup_sort(id);
	}
    }
}

static void print_set(ptr)
    SENODE *ptr;
{
    if (ptr == NULL) {
	printf("# %d\n", seindex - 1);
	return;
    }
    print_set(ptr->next);
    printf("[%c.%d] %c ", SET, ptr->index, ptr->tag);
    if (ptr->sort != NULL)
	printf("[%c.%d] ", SOR, ptr->sort->index);
    else
	printf("[%c.0] ", SOR);
    print_node(ptr->trm);
    printf("\t{ %s }\n", ptr->id);
}

void show_set()
{
    print_set(septr);
}

void clear_set()
{
    SENODE *tmp, *next;

    for (tmp = septr; tmp != NULL; tmp = next) {
	next = tmp->next;
	clear_tree(tmp->trm);
	PSF_FREE(tmp->id);
	PSF_FREE(tmp);
    }
    septr = NULL;
    seindex = 1;
}
