#include <assert.h>
#include <errno.h>
#include <signal.h>
#include <stdio.h>
#include <string.h>
#include <sys/stat.h>

#include "psf_prototype.h"
#include "psf_exits.h"
#include "psf_malloc.h"
#include "psf_redefs.h"
#include "psf_standards.h"
#include "defaults.h"
#include "psf.h"

FILE *split_stream;
static char *temp_name;
static char *previous_target = 0;

void change_module(name)
    char *name;
{
    module m;
    struct stat buf;
    sigset_t old_mask;

    for (m = module_list; m && ! strEqual(m->name, name); m = m->next)
	;
    if (m && option['S'] && !option['f']) {
	if (stat(m->tnam, &buf) == -1) {
	    if (errno != ENOENT)
		Complain("stat", m->tnam);
	} else {
	    fprintf(stderr, "%s: won't overwrite file %s\n",
		    	progname, psf_basename(m->tnam));
	    exit_status = EXIT_CMD_LINE_ERR;
	    cleanup();
	}
    }

    sysCall(sigprocmask(SIG_BLOCK, &signal_mask, &old_mask),
		"block", "control-C");
    if (m) {
	sysCall(rename(temp_name, m->tnam), "rename temp file to", m->tnam);
	current_target = m->tnam;
    } else {
	sysCall(unlink(temp_name), "unlink", temp_name);
	current_target = 0;
    }
    sysCall(sigprocmask(SIG_UNBLOCK, &signal_mask, &old_mask),
		"unblock", "control-C");
    previous_target = 0;
}

void change_file()
{
    stdioCall(!fclose(split_stream), "close", current_target);
    previous_target = current_target;
    current_target = temp_name;
    stdioCall(split_stream = fopen(temp_name, "w+"), "create", temp_name);
}

static void mk_symlink(f)
    file f;
{
    char *flt = f->last->tnam, *absname;

    if (unlink(flt) == -1 && errno != ENOENT)
	Complain("remove", flt);

    if (*f->name != '/') {
	absPath(absname, f->name);
	sysCall(symlink(absname, flt), "symlink", flt);
	free(absname);
    } else {
	sysCall(symlink(f->name, flt), "symlink", flt);
    }
}

static void psf_split(in)
    char *in;
{
    if (option['v'])
	(void) fprintf(stderr, "splitting %s\n", in);
    stdioCall(freopen(in, "r", yyin), "read", in);
    current_target = temp_name;
    stdioCall(split_stream = fopen(temp_name, "w+"), "create", temp_name);
    (void) yylex();
    if (previous_target) {
	FILE *last;
	int c;

	stdioCall(!fseek(split_stream, 0L, 0), "rewind", temp_name);
	stdioCall(last = fopen(previous_target, "a"),
		  "append to", previous_target);
	while ((c = getc(split_stream)) != EOF)
	    (void) putc(c, last);
	stdioCall(!ferror(split_stream), "read back from", temp_name);
	stdioCall(!ferror(last), "write to", previous_target);
	stdioCall(!fclose(last), "close", previous_target);
	sysCall(unlink(temp_name), "unlink", temp_name);
    }
    stdioCall(!fclose(split_stream), "close", current_target);
    current_target = 0;
}

void split()
{
    file f;

    splitting = TRUE;
    temp_name = PSF_NMALLOC(char, l_psf_tmpdir + l_psf_suffix + 10);
    (void) sprintf(temp_name, "%spart.%d%s", psf_tmpdir, (int) getpid(),
	psf_suffix);

    for (f = file_list; f; f = f->next) {
	if (f->num_modules == 0)
	    (void) fprintf(stderr, "%s: warning: no modules in file %s\n",
			   progname, f->name);
	else if (option['S'])
	    psf_split(f->name);
	else if (f->separandum)
	    if (f->num_modules == 1)
		mk_symlink(f);
	    else
		psf_split(f->name);
    }

    free(temp_name);
}
