#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <sys/stat.h>

#include "psf_prototype.h"
#include "psf_exits.h"
#include "psf_malloc.h"
#include "psf_standards.h"
#include "psf.h"

extern int yylineno;

module module_list = 0;
module *last_module = &module_list;
file file_list = 0;
static file *last_file = &file_list;
static file this_file;
static struct stat sbuf;
char *inputfile;
int startlineno;
bool unfinished_module = FALSE;

static import *last_uses;

void next_module(name)
    char *name;
{
    module m;
    struct stat tbuf;

    for (m = module_list; m && strcmp(name, m->name) != 0; m = m->next)
	;
    if (m) {
	if (m->where) {
	    (void) fprintf(stderr,
			"%s: module \"%s\" defined in %s and again in %s\n",
			progname, m->name, m->where->name, this_file->name);
	    exit(EXIT_SYNTAX_ERR);
	}
    } else {
	m = PSF_MALLOC(struct module);
	m->next = 0;
	m->name = psf_strdup(name);
	*last_module = m;
	last_module = &m->next;
    }

    m->where = (file) last_file;
    m->where->num_modules++;
    m->status = alive;
    m->imports = 0;
    m->whence = 0;
    m->from_stdlib = FALSE;
    last_uses = &m->imports;
    this_file->last = m;

    m->tnam = PSF_NMALLOC(char, l_psf_tmpdir + strlen(name) + l_psf_suffix);
    (void) strcpy(m->tnam, psf_tmpdir);
    (void) strcat(m->tnam, name);
    (void) strcat(m->tnam, psf_suffix);

    if (option['c'] || option['f']) {
	if (lstat(m->tnam, &tbuf) == -1) {
	    if (errno != ENOENT)
		Complain("stat", psf_basename(m->tnam));
	} else {
	    if (sbuf.st_dev == tbuf.st_dev && sbuf.st_ino == tbuf.st_ino) {
		fprintf(stderr,
			"%s: input %s is output -- that's too dangerous\n",
			progname, psf_basename(m->tnam));
		exit(EXIT_CMD_LINE_ERR);
	    } else
		sysCall(unlink(m->tnam) == -1, "remove", psf_basename(m->tnam));
	}
    } else if (!this_file->separandum)
	if (stat(m->tnam, &tbuf) == -1)
	    if (errno == ENOENT)
		this_file->separandum = TRUE;
	    else
		Complain("stat", m->tnam);
	else
	    this_file->separandum = sbuf.st_mtime > tbuf.st_mtime;
}

void check_id(name)
    char *name;
{
    if (strcmp(name, this_file->last->name) != 0) {
	(void) fprintf(stderr,
	       "%s: %s line %d: \"end %s\" unexpected; \"end %s\" expected\n",
		  progname, this_file->name, yylineno - startlineno + 1, name,
		       this_file->last->name);
	exit(EXIT_SYNTAX_ERR);
    }
}

void next_imports(name)
    char *name;
{
    module m;
    import i;

    if (this_file->num_modules == 0) {
	(void) fprintf(stderr,
		       "%s: %s line %d: \"imports %s\" not in any module\n",
		 progname, this_file->name, yylineno - startlineno + 1, name);
	exit(EXIT_SYNTAX_ERR);
    }
    for (m = module_list; m && ! strEqual(name, m->name); m = m->next)
	;
    if (!m) {
	m = PSF_MALLOC(struct module);
	m->next = 0;
	m->name = psf_strdup(name);
	m->where = 0;
	m->status = alive;
	m->imports = 0;
	m->whence = 0;
	m->from_stdlib = FALSE;

	*last_module = m;
	last_module = &m->next;
    }
    i = PSF_MALLOC(struct import);
    i->next = 0;
    i->uses = m;
    *last_uses = i;
    last_uses = &i->next;
}

void set_target(name)
    char *name;
{
    module m;

    m = PSF_MALLOC(struct module);
    m->next = 0;
    m->name = name;
    m->where = 0;
    m->status = alive;
    m->imports = 0;
    m->whence = 0;
    m->from_stdlib = FALSE;

    m->tnam = PSF_NMALLOC(char, l_psf_tmpdir + strlen(name) + l_psf_suffix);
    (void) strcpy(m->tnam, psf_tmpdir);
    (void) strcat(m->tnam, name);
    (void) strcat(m->tnam, psf_suffix);

    *last_module = m;
    last_module = &m->next;
}

static void next_file(name)
    char *name;
{
    struct file *f;

    inputfile = name;
    startlineno = yylineno;

    f = PSF_MALLOC(struct file);

    f->next = 0;
    f->name = name;
    f->num_modules = 0;
    f->last = 0;
    f->separandum = option['c'] || option['f'];

    this_file = f;
    *last_file = f;
    last_file = &f->next;

    sysCall(fstat(fileno(yyin), &sbuf), "fstat", name);
}

void list_modules()
{
    module m;

    if (outputfile) {
	if (option['v'])
	    (void) fprintf(stderr, "listing modules\n");
	stdioCall(freopen(outputfile, "w", stdout), "create", outputfile);
    }
    for (m = module_list; m; m = m->next)
	(void) printf("%s\tfrom\t%s\n",
		      m->name, m->where ? m->where->name : "???");
}

void dump_all()
{
    module m;
    import i;
    file f;

    (void) printf("modules:\n");
    for (m = module_list; m; m = m->next) {
	(void) printf("\t%s\t%s\n",
		      m->name, m->where ? m->where->name : "nowhere");
	if (m->where)
	    for (i = m->imports; i; i = i->next) {
		(void) printf("\t\t%s\n", i->uses->name);
	    }
    }

    (void) printf("files:\n");
    for (f = file_list; f; f = f->next) {
	(void) printf("\t%s\n", f->name);
    }
}

#include "psfDOT.c"
#include "psfGP.c"
#include "psfdaV.c"

void call_yylex(name)
    char *name;
{
    next_file(name);
    lex_inputfile_change();
    (void) yylex();
    if (unfinished_module) {
	if (this_file->last)
	    (void) fprintf(stderr,
		       "%s: module \"%s\" in file %s does not end properly\n",
			   progname, this_file->last->name, name);
	else
	    (void) fprintf(stderr,
			   "%s: file %s contains an incomplete module\n",
			   progname, name);
	exit(EXIT_SYNTAX_ERR);
    }
}
