#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <malloc.h>
#include <stdio.h>
#include <string.h>
#include <sys/param.h>
#include <sys/stat.h>

#include "psf_prototype.h"
#include "psf_exits.h"
#include "psf_fopen.h"
#include "psf_malloc.h"
#include "psf_redefs.h"
#include "defaults.h"
#include "env_variables.h"
#include "psf.h"

bool *option;
static bool psf_tmpdir_needed = TRUE;

static char *psfpath;

void usage()
{
    (void) fprintf(stderr,
      	"usage: %s [-cfhilsStvxX] [-o outputfile] [-m module] [-G format] [files]\n",
		   progname);
    exit(EXIT_CMD_LINE_ERR);
}

static void check_combinations()
{
    static char either_one_of[] = "lSxX";
    static char or_any_of[] = "cist";

    int n;
    char *s, *t;

    if (option['f'] && !option['S']) {
	fprintf(stderr, "%s: -f can only be combined with -S\n", progname);
	exit(EXIT_CMD_LINE_ERR);
    }
    if (option['m'] && option['S']) {
	fprintf(stderr, "%s: can't combine -m and -S\n", progname);
	exit(EXIT_CMD_LINE_ERR);
    }
    n = 0;
    for (s = either_one_of; *s; s++)
	if (option[*s])
	    n++;
    if (n > 1) {
	fprintf(stderr, "%s: use at most one of -%s\n", progname,
		either_one_of);
	exit(EXIT_CMD_LINE_ERR);
    } else if (option['l'] || option['S'])
	psf_tmpdir_needed = FALSE;

    for (s = either_one_of; *s; s++)
	if (option[*s])
	    for (t = or_any_of; *t; t++)
		if (option[*t]) {
		    fprintf(stderr, "%s: can't combine -%c and -%c\n",
			    progname, *s, *t);
		    exit(EXIT_CMD_LINE_ERR);
		}
}

void parse_opts(argc, argv)
    int argc;
    char **argv;
{
    char *s;
    int opt;
    static bool actual_option_array[UCHAR_MAX];

    option = actual_option_array - CHAR_MIN;

    if (s = getenv("PSFOPTIONS")) {
	if (*s == '-')
	    s++;
	while (*s) {
	    if (*s == '-' || isspace(*s))
		s++;
	    else if (strchr("cfhilsStvxX", *s))
		option[*s++] = TRUE;
	    else {
		(void) fprintf(stderr, *s == 'm' || *s == 'o' ?
			       "%s: -%c cannot be set from %s\n" :
			       "%s: unknown option -%c in %s\n",
			       progname, *s, "PSFOPTIONS");
		exit(EXIT_CMD_LINE_ERR);
	    }
	}
    }
    while ((opt = getopt(argc, argv, "cfG:hilL:m:o:sStvxX")) != -1) {
	option[(char) opt] = TRUE;
	if (opt == '?' || opt == 'h')
	    usage();		/* does not return */
	else if (opt == 'L')
	    psfpath = optarg;
	else if (opt == 'm')
	    target_module = optarg;
	else if (opt == 'o')
	    outputfile = optarg;
	else if (opt == 'G')
	    graphformat = optarg;
    }

    check_combinations();
}

static void set_psf_tmpdir()
{
    char *s, *t;
    int fd, len;
    struct stat buf;

    if (psf_tmpdir_needed) {
	if (!(s = getenv("PSFTMPDIR"))) {
	    s = PSF_NMALLOC(char, MAXPATHLEN);
	    if ((fd = open(DOTPSFRC, O_RDONLY)) != -1) {
		sysCall(len = read(fd, s, MAXPATHLEN), "read", DOTPSFRC);
		(void) close(fd);
		s[len - 1] = '\0';
	    } else {
		if (errno != ENOENT)
		    Complain("read", DOTPSFRC);
		(void) sprintf(s, "%s%d\n", DEFTMPDIR, (int) getpid());
		len = strlen(s);
		if ((fd = open(DOTPSFRC, O_WRONLY | O_CREAT, 0666)) == -1) {
		    (void) fprintf(stderr, "%s: can't create ", progname);
		    perror(DOTPSFRC);
		    (void) fprintf(stderr,
				"%s: try setting the environment ", progname);
		    (void) fprintf(stderr,
				"variable PSFTMPDIR and using the -o option\n");
		    cleanup();
		}
		sysCall(write(fd, s, len), "write to", DOTPSFRC);
		sysCall(close(fd), "close", DOTPSFRC);
		s[len - 1] = '\0';
	    }
	}
    } else if (option['S']) {
	if (outputfile)
	    s = outputfile;
	else
	    s = ".";
    } else {
	psf_tmpdir = "";
	l_psf_tmpdir = 0;
	return;
    }

    if (stat(s, &buf) == -1) {
	if (errno != ENOENT)
	    Complain2("stat", s);
	if (option['v'])
	    (void) fprintf(stderr, "creating directory %s\n", s);
	sysCall(mkdir(s, 0755), "mkdir", s);
    } else if (!S_ISDIR(buf.st_mode)) {
	(void) fprintf(stderr, "%s: %s: %s\n",
				progname, s, strerror(ENOTDIR));
	exit(EXIT_IO_ERR);
    } else if (geteuid() != buf.st_uid) {
	fprintf(stderr, "%s: you're not the owner of %s\n", progname, s);
	fprintf(stderr, "%s: try setting the environment variable PSFTMPDIR\n",
				progname);
	exit(EXIT_IO_ERR);
    }
    psf_tmp_dev = buf.st_dev;
    psf_tmp_ino = buf.st_ino;
    mkAbsPath(t, s);
    l_psf_tmpdir = strlen(t) + 2;
    psf_tmpdir = PSF_NMALLOC(char, l_psf_tmpdir);
    (void) strcpy(psf_tmpdir, t);
    (void) strcat(psf_tmpdir, "/");
}

void parse_env()
{
    char *s;

    set_psf_tmpdir();		/* must do this before set_libraries() */

    if (!option['S']) {
	if (option['L'])
	    s = psfpath;
	else
	    if (!(s = getenv("PSFPATH")))
		s = ".";
	set_libraries(s);
    }
    if (!(psf_suffix = getenv(PSFSUFFIX)))
	psf_suffix = PSFSUFFIX_DEFAULT;
    l_psf_suffix = strlen(psf_suffix) + 1;

    if (!(til_suffix = getenv(TILSUFFIX)))
	til_suffix = TILSUFFIX_DEFAULT;

    if (s = getenv("PSFBINDIR")) {
	psfc_bindir = PSF_NMALLOC(char, strlen(s) + 2);
	(void) strcpy(psfc_bindir, s);
	(void) strcat(psfc_bindir, "/");
    } else {
	psfc_bindir = DEFPSFBINDIR;
    }
}

void parse_args(argc, argv)
    int argc;
    char **argv;
{
    int i;
    psf_file pf;
    static suffix suffices[] = {{PSFSUFFIX, PSFSUFFIX_DEFAULT}, {0, 0}};

    if (outputfile) {
	for (i = optind; i < argc; i++) {
	    if (strEqual(argv[i], outputfile)) {
		(void) fprintf(stderr, "%s: input %s is output\n",
			       		progname, outputfile);
		exit(EXIT_CMD_LINE_ERR);
	    }
	}
    }
    for (i = optind; i < argc; i++) {
	pf = psf_fopen(argv[i], suffices);
	yyin = pf.fp;
	call_yylex(pf.name);
    }
}
