#include <stdio.h>
#include <string.h>
#include "psf_prototype.h"
#include "psf_malloc.h"
#include "dll.h"
#include "error.h"
#include "symtab.h"
#include "ystructs.h"
#include "tabout.h"
#include "typecheck.h"


#define DEBUG_FIND	0	/* show debug info Yes/No ? */

#define ANY		-1


/* memory locations */


extern DLL object_table;
extern DLL parse_table;
extern DLL set_table;
extern DLL variable_table;
extern int set_index;

DLL variables;
DLL scope;
int counter[TIL_ENTRIES];
int Bool_index, True_index, False_index;

void **til[TIL_ENTRIES];



void count_objects()
{
    DLL tab1, tab2;
    DLL_ITEM ptr, ptr1;
    struct specification *spec;
    struct func_decl *func;
    struct act_decl *act;
    struct proc_decl *proc;

    counter[SET] = set_index;	/* global value */

    DLL_FORALL(parse_table, ptr) {
	spec = (struct specification *) dll_inspect(ptr);

	switch (spec->type) {

	case SRT:
	    counter[SOR] += dll_count(spec->spec);
	    break;

	case FNC:
	    DLL_FORALL(spec->spec, ptr1) {
		func = (struct func_decl *) dll_inspect(ptr1);
		counter[FUN] += dll_count(func->functions);
	    }
	    break;

	case ACT:
	    DLL_FORALL(spec->spec, ptr1) {
		act = (struct act_decl *) dll_inspect(ptr1);
		counter[ATM] += dll_count(act->actions);
	    }
	    break;

	case CMM:
	    counter[COM] += dll_count(spec->spec);
	    break;

	case REW:
	    ptr1 = dll_go_last(spec->spec);
	    tab1 = (DLL) dll_inspect(ptr1);
	    counter[EQU] += dll_count(tab1);

	    ptr1 = dll_go_first(spec->spec);
	    tab1 = (DLL) dll_inspect(ptr1);

	    DLL_FORALL(tab1, ptr1) {
		tab2 = (DLL) dll_inspect(ptr1);
		counter[VAR] += dll_count(tab2) - 1;
	    }
	    break;

	case PRC:
	    counter[PRO] += dll_count(spec->spec);
	    counter[DEF] += dll_count(spec->spec);

	    DLL_FORALL(spec->spec, ptr1) {
		proc = (struct proc_decl *) dll_inspect(ptr1);
		counter[VAR] += dll_count(proc->vars);
	    }

	    break;


	default:
	    break;

	}
    }
}


int find_sort(name)
    char *name;
{
    int i;


#if DEBUG_FIND
    (void) printf("looking for sort <%s>\n", name);
#endif

    for (i = 1; i <= counter[SOR] && til[SOR][i] != NULL; i++) {
	/*
	 * (void)printf("looking @<%s>\n",((OBJ_PTR)til[SOR][i])->name);
	 */
	if (strcmp(name, ((OBJ_PTR) til[SOR][i])->name) == 0) {
	    return (i);
	}
    }
    return (0);
}


void type_sorts()
{
    DLL_ITEM ptr, ptr1;
    struct specification *spec;
    OBJ_PTR info;
    int cnt = 0, declared;

    DLL_FORALL(parse_table, ptr) {
	spec = (struct specification *) dll_inspect(ptr);

	if (spec->type == SRT) {
	    DLL_FORALL(spec->spec, ptr1) {
		info = (OBJ_PTR) dll_inspect(ptr1);
		declared = find_sort(info->name);
		if (declared) {
		    error_line(info->line_nr);
		    (void) fprintf(stderr, "sort '%s' redeclared\n", info->name);
		}
		info->class = SOR;
		info->index = ++cnt;
		til[info->class][info->index] = (void *) info;
	    }
	}
    }
}


int find_function(name, arity, itype)
    char *name;
    int arity, *itype;
{
    int i, j;
    struct fun_info *the_fun;

#if DEBUG_FIND
    (void) printf("looking for function <%s> %d\n", name, arity);
#endif

    for (i = 1; i <= counter[FUN] && til[FUN][i] != NULL; i++) {
	the_fun = (struct fun_info *) til[FUN][i];

#if DEBUG_FIND
	(void) printf("  comparing with %s\n", the_fun->obj->name);
#endif

	if (strcmp(name, the_fun->obj->name) == 0) {
	    if (arity == the_fun->arity) {
		if (arity == 0) {
		    return (i);
		}
		for (j = 0; itype[j] == the_fun->i_type[j] && j < arity; j++);
		if (j == arity) {
		    return (i);
		}
	    }
	}
    }
    return (0);
}


void type_Booleans()
{

    Bool_index = find_sort("Bool");
    if (!Bool_index) {
	error_line(0);
	(void) fprintf(stderr, "standard sort 'Bool' not declared\n");
    } else {

	True_index = find_function("T", 0, (int *) NULL);
	if (!True_index) {
	    error_line(0);
	    (void) fprintf(stderr, "standard function 'T' not declared\n");
	} else {
	    if (Bool_index != ((struct fun_info *) til[FUN][True_index])->o_type) {
		error_line(0);
		(void) fprintf(stderr, "standard function 'T' must be of type 'Bool'\n");
	    }
	}

	False_index = find_function("F", 0, (int *) NULL);
	if (!False_index) {
	    error_line(0);
	    (void) fprintf(stderr, "standard function 'F' not declared\n");
	} else {
	    if (Bool_index != ((struct fun_info *) til[FUN][False_index])->o_type) {
		error_line(0);
		(void) fprintf(stderr, "standard function 'F' must be of type 'Bool'\n");
	    }
	}
    }
}


void type_functions()
{
    DLL_ITEM ptr, ptr1, ptr2;
    struct specification *spec;
    int *in_type, arity, out_type;
    struct func_decl *info;
    OBJ_PTR info1;
    int cnt = 0, sort_index, func_index, j;
    struct fun_info *tmp;

    DLL_FORALL(parse_table, ptr) {
	spec = (struct specification *) dll_inspect(ptr);

	if (spec->type == FNC) {

	    DLL_FORALL(spec->spec, ptr1) {
		info = (struct func_decl *) dll_inspect(ptr1);
		arity = dll_count(info->input_type);
		in_type = PSF_NMALLOC(int, arity);
		info1 = (OBJ_PTR) info->output_type;
		out_type = find_sort(info1->name);
		if (!out_type) {
		    error_line(info1->line_nr);
		    (void) fprintf(stderr, "sort '%s' not declared\n", info1->name);
		}
		/* prepare array containing input type */
		j = 0;
		DLL_FORALL(info->input_type, ptr2) {
		    info1 = (OBJ_PTR) dll_inspect(ptr2);
		    sort_index = find_sort(info1->name);
		    if (!sort_index) {
			error_line(info1->line_nr);
			(void) fprintf(stderr, "sort '%s' not declared\n", info1->name);
		    }
		    in_type[j++] = sort_index;
		}

		DLL_FORALL(info->functions, ptr2) {
		    info1 = (OBJ_PTR) dll_inspect(ptr2);
		    func_index = find_function(info1->name, arity, in_type);

		    if (func_index) {
			error_line(info1->line_nr);
			(void) fprintf(stderr, "function '%s' redeclared\n", info1->name);
		    }
		    tmp = PSF_MALLOC(struct fun_info);

		    tmp->arity = arity;
		    tmp->i_type = in_type;

		    tmp->o_type = out_type;
		    tmp->obj = info1;
		    info1->class = FUN;
		    info1->index = ++cnt;
		    til[info1->class][info1->index] = (void *) tmp;
		}
	    }
	}
    }
}


int find_atom(name, arity, type)
    char *name;
    int arity, *type;
{
    int i, j;
    struct atm_info *the_act;

#if DEBUG_FIND
    (void) printf("looking for atom <%s>\n", name);
#endif

    for (i = 1; i <= counter[ATM] && til[ATM][i] != NULL; i++) {
	the_act = (struct atm_info *) til[ATM][i];

#if DEBUG_FIND
	(void) printf("  comparing with <%s>\n", the_act->obj->name);
#endif

	if (strcmp(name, the_act->obj->name) == 0) {
	    if (arity == ANY) {	/* just match on name */
		return (i);
	    }
	    if (arity == the_act->arity) {
		for (j = 0; j < arity; j++) {
		    if (type[j] != the_act->type[j]) {
			break;
		    }
		}
		if (j == arity) {
		    return (i);
		}
	    }
	}
    }
    return (0);
}


void type_actions()
{
    DLL_ITEM ptr, ptr1, ptr2;
    struct specification *spec;
    int *type, arity;
    struct act_decl *info;
    OBJ_PTR info1;
    int cnt = 0, sort_index, func_index, atom_index, j;
    struct atm_info *tmp;

    DLL_FORALL(parse_table, ptr) {
	spec = (struct specification *) dll_inspect(ptr);

	if (spec->type == ACT) {
	    DLL_FORALL(spec->spec, ptr1) {
		info = (struct act_decl *) dll_inspect(ptr1);
		arity = dll_count(info->type);
		type = PSF_NMALLOC(int, arity);

		/* prepare array containing input type */
		j = 0;
		DLL_FORALL(info->type, ptr2) {
		    info1 = (OBJ_PTR) dll_inspect(ptr2);
		    sort_index = find_sort(info1->name);
		    if (!sort_index) {
			error_line(info1->line_nr);
			(void) fprintf(stderr, "sort '%s' not declared\n", info1->name);
		    }
		    type[j++] = sort_index;
		}

		DLL_FORALL(info->actions, ptr2) {
		    info1 = (OBJ_PTR) dll_inspect(ptr2);

		    func_index = find_function(info1->name, arity, type);
		    if (func_index) {
			error_line(info1->line_nr);
			(void) fprintf(stderr, "action '%s' clashes with function declaration on line %d.\n",
				       info1->name, ((struct fun_info *) til[FUN][func_index])->obj->line_nr);
		    }
		    atom_index = find_atom(info1->name, arity, type);
		    if (atom_index) {
			error_line(info1->line_nr);
			(void) fprintf(stderr, "action '%s' redeclared\n", info1->name);
		    }
		    tmp = PSF_MALLOC(struct atm_info);

		    tmp->arity = arity;
		    tmp->type = type;

		    tmp->obj = info1;
		    info1->class = ATM;
		    info1->index = ++cnt;
		    til[info1->class][info1->index] = (void *) tmp;
		}
	    }
	}
    }
}


int find_process(name, arity, type)
    char *name;
    int arity, *type;
{
    int i, j;
    struct pro_info *the_pro;

#if DEBUG_FIND
    (void) printf("looking for process <%s>\n", name);
#endif

    for (i = 1; i <= counter[PRO] && til[PRO][i] != NULL; i++) {
	the_pro = (struct pro_info *) til[PRO][i];
	if (strcmp(name, the_pro->obj->name) == 0) {
	    if (arity == the_pro->arity) {
		for (j = 0; j < arity; j++) {
		    if (type[j] != the_pro->type[j]) {
			break;
		    }
		}
		if (j == arity) {
		    return (i);
		}
	    }
	}
    }
    return (0);
}


DLL_ITEM find_var_name(name, var_list_list)
    char *name;
    DLL var_list_list;
{
    DLL var_list;
    DLL_ITEM ptr1, ptr2;
    OBJ_PTR curr_var;

    DLL_FORALL(var_list_list, ptr1) {
	var_list = (DLL) dll_inspect(ptr1);
	ptr2 = dll_go_first(var_list);
	for (ptr2 = dll_go_fw(ptr2); ptr2 != NULL; ptr2 = dll_go_fw(ptr2)) {
	    curr_var = (OBJ_PTR) dll_inspect(ptr2);
	    if (strcmp(curr_var->name, name) == 0) {
		return (ptr2);
	    }
	}
    }
    return (NULL);
}


void unique_var_names(var_list_list)
    DLL var_list_list;
{
    DLL var_list;
    DLL_ITEM ptr1, ptr2;
    OBJ_PTR curr_var;

    DLL_FORALL(var_list_list, ptr1) {
	var_list = (DLL) dll_inspect(ptr1);
	ptr2 = dll_go_first(var_list);
	for (ptr2 = dll_go_fw(ptr2); ptr2 != NULL; ptr2 = dll_go_fw(ptr2)) {
	    curr_var = (OBJ_PTR) dll_inspect(ptr2);
	    if (ptr2 != find_var_name(curr_var->name, var_list_list)) {
		error_line(curr_var->line_nr);
		(void) fprintf(stderr, "variable '%s' redeclared\n", curr_var->name);
	    }
	}
    }
}


void type_processes()
{
    DLL_ITEM ptr, ptr1, ptr2, ptr3;
    struct specification *spec;
    int *type, arity;
    struct proc_decl *info;
    OBJ_PTR info1;
    DLL info2;
    int cnt = 0, sort_index, func_index, atom_index, proc_index, j;
    struct pro_info *tmp;

    DLL_FORALL(parse_table, ptr) {
	spec = (struct specification *) dll_inspect(ptr);

	if (spec->type == PRC) {
	    DLL_FORALL(spec->spec, ptr1) {
		info = (struct proc_decl *) dll_inspect(ptr1);

#if DEBUG_FIND
		(void) printf("extracted process: <%s>\n", info->process->name);
#endif

		unique_var_names(info->vars);

		arity = dll_count(info->vars);
		type = PSF_NMALLOC(int, arity);

#if DEBUG_FIND
		(void) printf("#vars : <%d>\n", arity);
#endif

		/* prepare array containing input type */
		j = 0;
		DLL_FORALL(info->vars, ptr2) {
		    info2 = (DLL) dll_inspect(ptr2);
		    ptr3 = dll_go_first(info2);
		    info1 = (OBJ_PTR) dll_inspect(ptr3);
		    sort_index = find_sort(info1->name);

		    /*
		     * can be skipped, checks are made in type_variables if
		     * (!sort_index) { error_line(info1->line_nr);
		     * (void)fprintf(stderr,"sort '%s' not
		     * declared\n",info1->name); }
		     */

		    type[j++] = sort_index;
		}

		func_index = find_function(info->process->name, arity, type);
		if (func_index) {
		    error_line(info->process->line_nr);
		    (void) fprintf(stderr, "process '%s' clashes with function declaration on line %d.\n",
				   info->process->name,
		    ((struct fun_info *) til[FUN][func_index])->obj->line_nr);
		}
		atom_index = find_atom(info->process->name, arity, type);
		if (atom_index) {
		    error_line(info->process->line_nr);
		    (void) fprintf(stderr, "process '%s' clashes with action declaration on line %d.\n",
				   info->process->name,
		    ((struct atm_info *) til[ATM][atom_index])->obj->line_nr);
		}
		proc_index = find_process(info->process->name, arity, type);
		if (proc_index) {
		    error_line(info->process->line_nr);
		    (void) fprintf(stderr, "process '%s' redeclared\n", info->process->name);
		}
		tmp = PSF_MALLOC(struct pro_info);

		tmp->arity = arity;
		tmp->type = type;

		tmp->obj = info->process;
		info->process->class = PRO;
		info->process->index = ++cnt;
		til[info->process->class][info->process->index] = (void *) tmp;
	    }
	}
    }
}


#define DEBUG_ALLTYPE 0

int *all_atoms(name)		/* return a list with all indexes for given
				 * name */
    char *name;
{
    int i, cnt = 0, *types;
    struct atm_info *the_act;
    DLL tmp_list;
    DLL_ITEM ptr;

#if DEBUG_ALLTYPE
    (void) printf("looking for atom <%s>\n", name);
#endif

    tmp_list = dll_create();

    for (i = 1; i <= counter[ATM] && til[ATM][i] != NULL; i++) {
	the_act = (struct atm_info *) til[ATM][i];

#if DEBUG_ALLTYPE
	(void) printf("  comparing with <%s>\n", the_act->obj->name);
#endif

	if (strcmp(name, the_act->obj->name) == 0) {
	    dll_append(tmp_list, (DLL_INFO) i);
	    cnt++;
	}
    }
    types = PSF_NMALLOC(int, cnt + 1);
    types[0] = cnt;
    i = 0;
    DLL_FORALL(tmp_list, ptr) {
	types[++i] = (int) dll_inspect(ptr);
    }

    dll_dispose(tmp_list);
    return (types);
}


int equal_type(x, y)		/* check equality of type of actions */
    int x, y;
{
    int i;
    struct atm_info *xp, *yp;

    xp = (struct atm_info *) til[ATM][x];
    yp = (struct atm_info *) til[ATM][y];

    if (xp->arity != yp->arity) {
	return (0);
    }
    for (i = 0; i < xp->arity; i++) {
	if (xp->type[i] != yp->type[i]) {
	    return (0);
	}
    }
    return (1);
}


void print_type(type, len)
    int *type, len;
{
    int i;

    for (i = 0; i < len; i++) {
	if (i != 0) {
	    (void) fprintf(stderr, " # ");
	}
	if (type[i] == 0) {
	    (void) fprintf(stderr, "?");
	} else {
	    (void) fprintf(stderr, "%s", ((OBJ_PTR) til[SOR][type[i]])->name);
	}
    }
}


void comm_compatible(line, s, r, c)	/* check compatibility in comm.
					 * function */
    int line, *s, *r, *c;
{
    int i, j, k;
    int compatible1, compatible2;
    struct atm_info *the_atom;

    for (i = 1; i <= s[0]; i++) {
	compatible1 = 0;
	for (j = 1; j <= r[0]; j++) {
	    if (equal_type(s[i], r[j])) {
		compatible1 = 1;
		compatible2 = 0;
		for (k = 1; k <= c[0]; k++) {
		    if (equal_type(r[j], c[k])) {
			compatible2 = 1;
			break;
		    }
		}
		if (compatible2) {
		    break;
		} else {
		    if (c[0] != 0) {	/* not an empty list */
			the_atom = (struct atm_info *) til[ATM][c[1]];
			error_line(line);
			(void) fprintf(stderr, "missing communication action '%s(",
				       the_atom->obj->name);
			the_atom = (struct atm_info *) til[ATM][s[1]];
			print_type(the_atom->type, the_atom->arity);
			(void) fprintf(stderr, ")'\n");
		    }
		}
	    }
	}
	if (compatible1) {
	    continue;
	} else {
	    if (r[0] != 0) {	/* not an empty list */
		the_atom = (struct atm_info *) til[ATM][r[1]];
		error_line(line);
		(void) fprintf(stderr, "missing communication action '%s(",
			       the_atom->obj->name);
		the_atom = (struct atm_info *) til[ATM][s[1]];
		print_type(the_atom->type, the_atom->arity);
		(void) fprintf(stderr, ")'\n");
	    }
	}
    }
}


void init_varlist()
{
    variables = dll_create();
    scope = dll_create();
}


int add_variable(name, type)	/* add variable to list only if 'new' */
    char *name;
    int type;
{
    DLL_ITEM ptr;
    int index = 0;
    struct var_info *var_info;

    DLL_FORALL(variables, ptr) {
	index++;
	var_info = (struct var_info *) dll_inspect(ptr);
	if (var_info->type == type) {
	    if (strcmp(var_info->name, name) == 0) {
		return (index);
	    }
	}
    }

    var_info = PSF_MALLOC(struct var_info);
    var_info->name = name;
    var_info->type = type;
    var_info->index = ++index;
    dll_append(variables, (DLL_INFO) var_info);

    return (index);
}


int var_index(type, ptr)	/* look for variable with type from ptr on */
    int type;
    DLL_ITEM ptr;
{
    struct var_info *var_info;

    for (; ptr != NULL; ptr = dll_go_fw(ptr)) {
	var_info = (struct var_info *) dll_inspect(ptr);
	if (var_info->type == type) {
	    return (var_info->index);
	}
    }
    return (0);
}


void print_variable_table()
{
    DLL_ITEM list_ptr, name_ptr;
    DLL var_list;
    OBJ_PTR info;
    int sort_index;


    (void) printf("### Variable Table Dump ###\n");
    DLL_FORALL(variable_table, list_ptr) {
	var_list = (DLL) dll_inspect(list_ptr);
	name_ptr = dll_go_first(var_list);
	info = (OBJ_PTR) dll_inspect(name_ptr);
	sort_index = find_sort(info->name);

	for (name_ptr = dll_go_fw(name_ptr);
		name_ptr != NULL;
		name_ptr = dll_go_fw(name_ptr)) {
	    info = (OBJ_PTR) dll_inspect(name_ptr);
	    (void) printf("line %d. var: <%s> type-index: <%d>\n", info->line_nr, info->name, sort_index);
	}
    }
    (void) printf("### End of Variable Table Dump ###\n");
}


void type_variables()
{
    DLL_ITEM list_ptr, name_ptr;
    DLL var_list;
    OBJ_PTR info;
    int sort_index, func_index, atom_index, proc_index;


    DLL_FORALL(variable_table, list_ptr) {
	var_list = (DLL) dll_inspect(list_ptr);
	name_ptr = dll_go_first(var_list);
	info = (OBJ_PTR) dll_inspect(name_ptr);
	sort_index = find_sort(info->name);

	if (!sort_index) {
	    error_line(info->line_nr);
	    (void) fprintf(stderr, "sort '%s' not declared\n", info->name);
	}
	for (name_ptr = dll_go_fw(name_ptr);
		name_ptr != NULL;
		name_ptr = dll_go_fw(name_ptr)) {
	    info = (OBJ_PTR) dll_inspect(name_ptr);

	    func_index = find_function(info->name, 0, (int *) NULL);
	    if (func_index) {
		error_line(info->line_nr);
		(void) fprintf(stderr, "variable '%s' clashes with function declaration on line %d.\n",
			       info->name,
		    ((struct fun_info *) til[FUN][func_index])->obj->line_nr);
	    }
	    atom_index = find_atom(info->name, 0, (int *) NULL);
	    if (atom_index) {
		error_line(info->line_nr);
		(void) fprintf(stderr, "variable '%s' clashes with action declaration on line %d.\n",
			       info->name,
		    ((struct atm_info *) til[ATM][atom_index])->obj->line_nr);
	    }
	    proc_index = find_process(info->name, 0, (int *) NULL);
	    if (proc_index) {
		error_line(info->line_nr);
		(void) fprintf(stderr, "variable '%s' clashes with process declaration on line %d.\n",
			       info->name,
		    ((struct pro_info *) til[PRO][proc_index])->obj->line_nr);
	    }
	    info->class = VAR;
	    info->index = add_variable(info->name, sort_index);
	}
    }
}


void comm_consistency()
{
    DLL_ITEM ptr, ptr1;
    struct specification *spec;
    struct comm_decl *info;
    int *left_types, *right_types, *result_types;


    DLL_FORALL(parse_table, ptr) {
	spec = (struct specification *) dll_inspect(ptr);

	if (spec->type == CMM) {
	    DLL_FORALL(spec->spec, ptr1) {
		info = (struct comm_decl *) dll_inspect(ptr1);

		left_types = all_atoms(info->left->name);
		right_types = all_atoms(info->right->name);
		result_types = all_atoms(info->result->name);

		comm_compatible(info->left->line_nr,
				left_types, right_types, result_types);
		comm_compatible(info->right->line_nr,
				right_types, left_types, result_types);

		free(left_types);
		free(right_types);
		free(result_types);
	    }
	}
    }
}


void type_sets()
{				/* items in set_list: <index,action-list> */
    DLL_ITEM ptr, ptr1;
    DLL the_set, the_atoms;
    int index, length, cnt, i;
    int *tmp, *atm_list;
    struct set_info *tmp_set;
    OBJ_PTR the_atom;


    the_atoms = dll_create();

    DLL_FORALL(set_table, ptr) {
	the_set = (DLL) dll_inspect(ptr);

	ptr1 = dll_go_first(the_set);
	index = (int) dll_inspect(ptr1);
	length = 0;

	for (ptr1 = dll_go_fw(ptr1); ptr1 != NULL; ptr1 = dll_go_fw(ptr1)) {
	    the_atom = (OBJ_PTR) dll_inspect(ptr1);
	    tmp = all_atoms(the_atom->name);
	    if (tmp[0] == 0) {	/* no atoms with this name */
		error_line(the_atom->line_nr);
		(void) fprintf(stderr, "action '%s' not declared\n", the_atom->name);
	    }
	    dll_append(the_atoms, tmp);	/* save list of atoms */
	    length += tmp[0];
	}

	tmp = PSF_NMALLOC(int, length);

	cnt = 0;
	DLL_FORALL(the_atoms, ptr1) {
	    atm_list = (int *) dll_inspect(ptr1);
	    for (i = 1; i <= atm_list[0]; i++) {
		tmp[cnt++] = atm_list[i];
	    }
	}

	dll_flush(the_atoms);

	tmp_set = PSF_MALLOC(struct set_info);
	tmp_set->length = length;
	tmp_set->set = tmp;

	til[SET][index] = (void *) tmp_set;
    }
}



void type_communications()
{
    DLL_ITEM ptr, ptr1;
    struct specification *spec;
    struct comm_decl *info;
    int cnt = 0;
    struct com_info *tmp;

    DLL_FORALL(parse_table, ptr) {
	spec = (struct specification *) dll_inspect(ptr);

	if (spec->type == CMM) {
	    DLL_FORALL(spec->spec, ptr1) {
		info = (struct comm_decl *) dll_inspect(ptr1);

		tmp = PSF_MALLOC(struct com_info);

		tmp->line = info->left->line_nr;

		tmp->left = find_atom(info->left->name, ANY, (int *) NULL);
		if (!tmp->left) {
		    error_line(info->left->line_nr);
		    (void) fprintf(stderr, "action '%s' not declared\n", info->left->name);
		}
		tmp->right = find_atom(info->right->name, ANY, (int *) NULL);
		if (!tmp->right) {
		    error_line(info->right->line_nr);
		    (void) fprintf(stderr, "action '%s' not declared\n", info->right->name);
		}
		tmp->result = find_atom(info->result->name, ANY, (int *) NULL);
		if (!tmp->result) {
		    error_line(info->result->line_nr);
		    (void) fprintf(stderr, "action '%s' not declared\n", info->result->name);
		}
		til[COM][++cnt] = (void *) tmp;
	    }
	}
    }
}


int find_variable(name, type)
    char *name;
    int type;
{
    DLL_ITEM ptr;
    struct var_info *the_var;

    DLL_FORALL(variables, ptr) {
	the_var = (struct var_info *) dll_inspect(ptr);
	if (type == the_var->type) {
	    if (strcmp(name, the_var->name) == 0) {
		return (the_var->index);
	    }
	}
    }
    return (0);
}


void delete_from_scope()
{
    (void) dll_del_front(scope);
}


void print_scope()
{
    DLL var_list_list, var_list;
    DLL_ITEM ptr, ptr1, ptr2;
    OBJ_PTR curr_type, curr_var;

    (void) printf("### Scope Dump ###\n");

    DLL_FORALL(scope, ptr) {
	var_list_list = (DLL) dll_inspect(ptr);
	DLL_FORALL(var_list_list, ptr1) {
	    var_list = (DLL) dll_inspect(ptr1);
	    ptr2 = dll_go_first(var_list);
	    curr_type = (OBJ_PTR) dll_inspect(ptr2);
	    (void) printf("type <%s>\n", curr_type->name);
	    for (ptr2 = dll_go_fw(ptr2); ptr2 != NULL; ptr2 = dll_go_fw(ptr2)) {
		curr_var = (OBJ_PTR) dll_inspect(ptr2);
		(void) printf("variable <%s>\n", curr_var->name);
	    }
	}
    }

    (void) printf("### End of Scope Dump ###\n");
}


int var_in_scope(name)		/* look for 'name' var-list, return type */
    char *name;
{
    DLL var_list_list, var_list;
    DLL_ITEM ptr, ptr1, ptr2;
    OBJ_PTR curr_type, curr_var;
    int sort_index;

    DLL_FORALL(scope, ptr) {
	var_list_list = (DLL) dll_inspect(ptr);
	DLL_FORALL(var_list_list, ptr1) {
	    var_list = (DLL) dll_inspect(ptr1);
	    ptr2 = dll_go_first(var_list);
	    curr_type = (OBJ_PTR) dll_inspect(ptr2);
	    for (ptr2 = dll_go_fw(ptr2); ptr2 != NULL; ptr2 = dll_go_fw(ptr2)) {
		curr_var = (OBJ_PTR) dll_inspect(ptr2);
		if (strcmp(name, curr_var->name) == 0) {
		    sort_index = find_sort(curr_type->name);
		    if (sort_index) {
			return (sort_index);
		    } else {
			return (-1);	/* var in scope, but cannot be typed */
		    }
		}
	    }
	}
    }
    return (0);
}


void add_to_scope(list)
    DLL list;
{
    DLL_ITEM ptr, ptr1;
    DLL var_list;
    OBJ_PTR the_var;


    DLL_FORALL(list, ptr) {
	var_list = (DLL) dll_inspect(ptr);

	ptr1 = dll_go_first(var_list);
	for (ptr1 = dll_go_fw(ptr1); ptr1 != NULL; ptr1 = dll_go_fw(ptr1)) {
	    the_var = (OBJ_PTR) dll_inspect(ptr1);
	    if (var_in_scope(the_var->name)) {
		error_line(the_var->line_nr);
		(void) fprintf(stderr, "scoping conflict on variable '%s'\n", the_var->name);
	    }
	}
    }
    dll_insert(scope, (DLL_INFO) list);
}


int type_data_term(trm)
    struct data_term *trm;
{
    DLL_ITEM ptr;
    int *type, i = 0, func_index, the_index, arg_cnt, var_type;

    arg_cnt = dll_count(trm->arg);
    type = PSF_NMALLOC(int, arg_cnt);

#define TYPE_DT 0

#if TYPE_DT
    (void) printf("%s %d\n", trm->head->name, arg_cnt);
#endif

    DLL_FORALL(trm->arg, ptr) {
	type[i] = type_data_term((struct data_term *) dll_inspect(ptr));
	if (type[i++] == 0) {	/* error in sub term */
	    return (0);
	}
    }
    func_index = find_function(trm->head->name, arg_cnt, type);

    if (!func_index) {
	if (arg_cnt == 0) {	/* could be a variable */
	    var_type = var_in_scope(trm->head->name);
	    if (var_type > 0) {
		the_index = find_variable(trm->head->name, var_type);
		trm->head->class = VAR;
		trm->head->index = the_index;
		return (var_type);
	    } else {
		if (var_type == 0) {	/* suppress superflous error report */
		    error_line(trm->head->line_nr);
		    (void) fprintf(stderr, "constant function or variable '%s' not declared\n",
				   trm->head->name);
		}
		return (0);
	    }
	} else {
	    error_line(trm->head->line_nr);
	    (void) fprintf(stderr, "function '%s(", trm->head->name);
	    print_type(type, arg_cnt);
	    (void) fprintf(stderr, ")' not declared\n");
	    return (0);
	}
    } else {
	trm->head->class = FUN;
	trm->head->index = func_index;
	return (((struct fun_info *) til[FUN][func_index])->o_type);
    }

}


void type_equations()
{
    DLL_ITEM ptr, ptr1, ptr2, ptr3;
    struct specification *spec;
    DLL vars, eq_list, equation;
    struct data_term *lhs, *rhs;
    struct equ_info *tmp;
    int cnt = 0, lhs_type, rhs_type;

    DLL_FORALL(parse_table, ptr) {
	spec = (struct specification *) dll_inspect(ptr);

	if (spec->type == REW) {
	    ptr1 = dll_go_first(spec->spec);
	    vars = (DLL) dll_inspect(ptr1);

	    unique_var_names(vars);
	    add_to_scope(vars);

	    ptr1 = dll_go_fw(ptr1);
	    eq_list = (DLL) dll_inspect(ptr1);

	    DLL_FORALL(eq_list, ptr2) {
		tmp = PSF_MALLOC(struct equ_info);
		equation = (DLL) dll_inspect(ptr2);
		ptr3 = dll_go_first(equation);
		lhs = (struct data_term *) dll_inspect(ptr3);
		tmp->lhs = lhs;
		ptr3 = dll_go_fw(ptr3);
		rhs = (struct data_term *) dll_inspect(ptr3);
		tmp->rhs = rhs;
		lhs_type = type_data_term(lhs);
		rhs_type = type_data_term(rhs);
		if (lhs_type != rhs_type && lhs_type != 0 && rhs_type != 0) {
		    error_line(lhs->head->line_nr);
		    (void) fprintf(stderr, "type clash in equation. <lhs>:'%s' vs. <rhs>:'%s'\n",
				   ((OBJ_PTR) til[SOR][lhs_type])->name,
				   ((OBJ_PTR) til[SOR][rhs_type])->name);
		}
		til[EQU][++cnt] = (void *) tmp;
	    }
	    delete_from_scope();
	}
    }
}


void issue_warning(class, line)
    int class, line;
{
    (void) fprintf(stderr, "WARNING line %d. process declaration contains unsupported construct: ", line);
    switch (class) {
    case LMG:
	(void) fprintf(stderr, "left merge");
	break;
    case CMG:
	(void) fprintf(stderr, "communication merge");
	break;
    case DLK:
	(void) fprintf(stderr, "deadlock");
	break;
    case REN:
	(void) fprintf(stderr, "renaming");
	break;
    }
    (void) fprintf(stderr, "\n");
}


void type_var_list(list)
    DLL list;
{
    DLL_ITEM ptr, ptr1;
    DLL the_var;
    OBJ_PTR the_name, the_type;

    DLL_FORALL(list, ptr) {
	the_var = (DLL) dll_inspect(ptr);
	ptr1 = dll_go_first(the_var);
	the_type = (OBJ_PTR) dll_inspect(ptr1);
	ptr1 = dll_go_fw(ptr1);
	the_name = (OBJ_PTR) dll_inspect(ptr1);
	the_name->class = VAR;
	the_name->index = find_variable(the_name->name, find_sort(the_type->name));
    }
}


void type_proc_expr(expr, line)
    struct proc_expr *expr;
    int line;
{
    DLL the_args;
    DLL_ITEM ptr, ptr1;
    struct data_term *the_term;
    int dt_type, i = 0, arg_cnt, *type, the_index;
    OBJ_PTR the_obj;

    switch (expr->class) {
    case LMG:
    case CMG:
    case REN:
	issue_warning(expr->class, line);
	break;
    case ALT:
    case PAR:
    case SEQ:
	type_proc_expr(expr->pe1, line);
	type_proc_expr(expr->pe2, line);
	break;
    case HIF:
	ptr = dll_go_first(expr->list);
	the_term = (struct data_term *) dll_inspect(ptr);

	dt_type = type_data_term(the_term);
	if (dt_type != Bool_index && dt_type != 0) {
	    error_line(the_term->head->line_nr);
	    (void) fprintf(stderr, "condition in 'if' must be of type 'Bool'\n");
	}
	type_proc_expr(expr->pe1, line);
	type_proc_expr(expr->pe2, line);
	break;
    case DLK:
    case TAU:
	break;
    case ENC:
    case HID:
	type_proc_expr(expr->pe1, line);
	break;
    case SUM:
	type_var_list(expr->list);
	add_to_scope(expr->list);
	type_proc_expr(expr->pe1, line);
	delete_from_scope();
	break;
    case PRY:
    case PRN:
	ptr = dll_go_first(expr->list);
	the_obj = (OBJ_PTR) dll_inspect(ptr);
	ptr = dll_go_fw(ptr);
	the_args = (DLL) dll_inspect(ptr);
	arg_cnt = dll_count(the_args);
	type = PSF_NMALLOC(int, arg_cnt);

	DLL_FORALL(the_args, ptr1) {
	    type[i] = type_data_term((struct data_term *) dll_inspect(ptr1));
	    if (type[i++] == 0) {
		return;
	    }
	}
	if (arg_cnt == i) {
	    the_index = find_atom(the_obj->name, arg_cnt, type);
	    if (the_index) {	/* it is an atomic action */
		the_obj->class = ATM;
		the_obj->index = the_index;
	    } else {
		the_index = find_process(the_obj->name, arg_cnt, type);
		if (the_index) {/* it is a process */
		    the_obj->class = PRO;
		    the_obj->index = the_index;
		} else {
		    error_line(the_obj->line_nr);
		    (void) fprintf(stderr, "action or process '%s(", the_obj->name);
		    print_type(type, arg_cnt);
		    (void) fprintf(stderr, ")' not declared\n");
		}
	    }
	}
	break;
    default:
	break;
    }
}


void type_definitions()
{
    DLL_ITEM ptr, ptr1;
    struct specification *spec;
    struct proc_decl *the_def;

    DLL_FORALL(parse_table, ptr) {
	spec = (struct specification *) dll_inspect(ptr);
	if (spec->type == PRC) {
	    DLL_FORALL(spec->spec, ptr1) {
		the_def = (struct proc_decl *) dll_inspect(ptr1);
		type_var_list(the_def->vars);
		add_to_scope(the_def->vars);
		type_proc_expr(the_def->expr, the_def->process->line_nr);
		delete_from_scope();
	    }
	}
    }
}


void empty_sort_check()
{
    int i, j, k, *empty, changed;
    struct fun_info *the_fun;

    empty = PSF_NMALLOC(int, counter[SOR] + 1);
    for (i = 1; i <= counter[SOR]; i++) {
	empty[i] = 1;		/* initially all sorts empty */
    }

    do {
	changed = 0;
	for (j = 1; j <= counter[FUN]; j++) {
	    the_fun = (struct fun_info *) til[FUN][j];
	    if (empty[the_fun->o_type]) {	/* only interested in empty
						 * sorts */
		for (k = 0; k < the_fun->arity && !empty[the_fun->i_type[k]]; k++);
		if (k == the_fun->arity) {
		    empty[the_fun->o_type] = 0;
		    changed = 1;
		}
	    }
	}
    } while (changed);

    for (i = 1; i <= counter[SOR]; i++) {
	if (empty[i]) {
	    error_line(0);	/* global error */
	    (void) fprintf(stderr, "sort '%s' is empty\n", ((OBJ_PTR) til[SOR][i])->name);
	}
    }
}


#define DEADLOCK 0

void print_assoc(a1, a2, a3, res)
    int a1, a2, a3, res;
{
    (void) fprintf(stderr, "    (%s|%s)|%s = ",
		   ((struct atm_info *) til[ATM][a1])->obj->name,
		   ((struct atm_info *) til[ATM][a2])->obj->name,
		   ((struct atm_info *) til[ATM][a3])->obj->name);
    if (res == DEADLOCK) {
	(void) fprintf(stderr, "DEADLOCK");
    } else {
	(void) fprintf(stderr, "%s",
		       ((struct atm_info *) til[ATM][res])->obj->name);
    }
    (void) fprintf(stderr, "\n");
}


#define ct(x,y) com_tab[x*dimension+y]

void comm_assoc_check()
{
    int *com_tab;		/* communication table */
    int dimension;		/* dimension of communication table */
    int i, j, k;
    int left, right, result;
    int left_assoc, right_assoc, mid_assoc;
    struct com_info *the_comm;

    dimension = counter[ATM] + 1;
    com_tab = PSF_NMALLOC(int, dimension * dimension);	/* create table */

    for (i = 0; i < dimension; i++) {	/* ... and initialize it */
	for (j = 0; j < dimension; j++) {
	    ct(i, j) = DEADLOCK;
	}
    }

    for (i = 1; i <= counter[COM]; i++) {	/* check communication is
						 * function */
	the_comm = (struct com_info *) til[COM][i];
	left = the_comm->left;
	right = the_comm->right;
	result = the_comm->result;
	if (ct(left, right) != DEADLOCK) {
	    error_line(the_comm->line);
	    if (ct(left, right) == result) {
		(void) fprintf(stderr, "redeclaration of communication function\n");
	    } else {
		(void) fprintf(stderr, "communication function is not commutative\n");
	    }
	}
	ct(left, right) = result;	/* and fill communication table */
	ct(right, left) = result;
    }

    for (i = 1; i < dimension; i++) {
	for (j = i; j < dimension; j++) {
	    for (k = j; k < dimension; k++) {
		left_assoc = ct(ct(i, j), k);
		right_assoc = ct(ct(j, k), i);
		mid_assoc = ct(ct(k, i), j);
		if (!((left_assoc == mid_assoc) && (mid_assoc == right_assoc))) {
		    error_line(0);
		    (void) fprintf(stderr, "communication function is not associative\n");
		    print_assoc(i, j, k, left_assoc);
		    if (i != k) {	/* bypass superflous error report */
			print_assoc(j, k, i, right_assoc);
		    }
		    if ((i != j) && (j != k)) {
			print_assoc(k, i, j, mid_assoc);
		    }
		}
	    }
	}
    }
}


void clear_counters()
{
    int i;

    for (i = 0; i < TIL_ENTRIES; i++) {
	counter[i] = 0;
    }
}


void alloc_tables()
{
    int i, j;

    for (i = 0; i < TIL_ENTRIES; i++) {
	til[i] = PSF_NMALLOC(void *, (counter[i] + 1));
	for (j = 0; j <= counter[i]; j++) {
	    til[i][j] = NULL;
	}
    }
    init_varlist();
}


void type_check()
{

    clear_counters();
    count_objects();
    alloc_tables();

    type_sorts();
    type_functions();

    type_Booleans();		/* check for 'Boolean' standard package */

    type_actions();
    type_processes();

    type_variables();

    type_sets();
    type_communications();
    comm_consistency();
    type_equations();
    type_definitions();

    empty_sort_check();
    comm_assoc_check();
    /*
     * exit(1);
     */

    if (nr_of_errors()) {
	(void) fprintf(stderr, "input file contained %d semantical error(s)\n",
		       nr_of_errors());
    } else {
	print_tables();
    }

}
