/*
 * dll.c : doubly linked list structures
 */

#include <stdio.h>
#include "psf_prototype.h"
#include "psf_malloc.h"

#include "dll.h"

DLL dll_create()
{				/* create a doubly linked list */
    DLL ptr;

    ptr = (DLL) PSF_MALLOC(DLL_BLOCK);
    ptr->first = ptr->last = NULL;
    return ptr;
}


void dll_flush(ptr)		/* delete all elements from a doubly linked
				 * list */
    DLL ptr;
{
    for (; !dll_empty(ptr);) {
	dll_del_front(ptr);
    }
}


void dll_dispose(ptr)		/* dispose a doubly linked list */
    DLL ptr;
{
    dll_flush(ptr);
    free((char *) ptr);
}


DLL dll_copy(list)		/* create a copy of a dll */
    DLL list;
{
    DLL new_list;
    DLL_ITEM curr;

    new_list = dll_create();

    for (curr = dll_go_first(list); curr != NULL; curr = dll_go_fw(curr)) {
	dll_append(new_list, dll_inspect(curr));
    }

    return (new_list);
}


DLL_ITEM create_item(info)	/* create a doubly linked list item */
    DLL_INFO info;
{
    DLL_ITEM ptr;

    ptr = (DLL_ITEM) PSF_MALLOC(DLL_ITBL);
    ptr->fw = ptr->bw = NULL;
    ptr->info = info;
    return ptr;
}

void dispose_item(ptr)		/* dispose a doubly linked list item */
    DLL_ITEM ptr;
{
    free((char *) ptr);
}


int dll_empty(list)		/* check whether list empty */
    DLL list;
{
    return ((list->first == NULL) && (list->last == NULL));
}


int dll_first(list, curr)	/* is current first element? */
    DLL list;
    DLL_ITEM curr;
{
    return (curr == list->first);
}


int dll_last(list, curr)	/* is current last element? */
    DLL list;
    DLL_ITEM curr;
{
    return (curr == list->last);
}


DLL_INFO dll_inspect(curr)	/* get the current element of the list */
    DLL_ITEM curr;
{
    if (curr == NULL) {
	return (NULL);
    }
    return (curr->info);
}


DLL_ITEM dll_go_first(list)	/* go to first element of list */
    DLL list;
{
    return (list->first);
}


DLL_ITEM dll_go_last(list)	/* go to last element of list */
    DLL list;
{
    return (list->last);
}


DLL_ITEM dll_go_fw(curr)	/* go to next element of list */
    DLL_ITEM curr;
{
    if (curr != NULL) {
	curr = curr->fw;
    }
    return (curr);
}


DLL_ITEM dll_go_bw(curr)	/* go to previous element of list */
    DLL_ITEM curr;
{
    if (curr != NULL) {
	curr = curr->bw;
    }
    return (curr);
}


void dll_insert(list, info)	/* insert info at the front of list */
    DLL list;
    DLL_INFO info;
{
    DLL_ITEM item;

    item = create_item(info);

    if (list->first == NULL) {	/* list is empty */
	list->first = item;
	list->last = list->first;
    } else {
	list->first->bw = item;
	item->fw = list->first;
	list->first = item;
    }
}


void dll_insert_at(list, curr, info)	/* insert info at current element */
    DLL list;
    DLL_ITEM curr;
    DLL_INFO info;
{
    DLL_ITEM item;

    if (curr == NULL) {
	(void) fprintf(stderr, "dll_insert_at: illegal current pointer\n");
	exit(1);
    }
    item = create_item(info);

    if (curr == list->first) {	/* current is first? */
	list->first->bw = item;
	item->fw = list->first;
	list->first = item;
    } else {
	item->fw = curr;
	item->bw = curr->bw;
	item->fw->bw = item;
	item->bw->fw = item;
    }
}


void dll_append(list, info)	/* append info at the back of list */
    DLL list;
    DLL_INFO info;
{
    DLL_ITEM item;

    item = create_item(info);

    if (list->last == NULL) {	/* list is empty */
	list->last = item;
	list->first = list->last;
    } else {
	list->last->fw = item;
	item->bw = list->last;
	list->last = item;
    }
}


void dll_append_at(list, curr, info)	/* append info at current element */
    DLL list;
    DLL_ITEM curr;
    DLL_INFO info;
{
    DLL_ITEM item;

    if (curr == NULL) {
	(void) fprintf(stderr, "dll_insert_at: illegal current pointer\n");
	exit(1);
    }
    item = create_item(info);

    if (curr == list->last) {	/* current is last? */
	list->last->fw = item;
	item->bw = list->last;
	list->last = item;
    } else {
	item->bw = curr;
	item->fw = curr->fw;
	item->bw->bw = item;
	item->fw->fw = item;
    }
}


DLL_INFO del_item(list, ptr)	/* delete item pointed to from list */
    DLL list;
    DLL_ITEM ptr;
{
    DLL_INFO info;

    if (dll_empty(list)) {	/* empty list */
	(void) fprintf(stderr, "del_item: list empty!\n");
	exit(1);
    }
    if (list->first->fw == NULL) {	/* one element left */
	info = list->first->info;
	dispose_item(list->first);
	list->first = list->last = NULL;
	return (info);
    }
    if (ptr == list->first) {	/* delete first element */
	info = ptr->info;
	ptr->fw->bw = ptr->bw;
	list->first = list->first->fw;
	dispose_item(ptr);
	return (info);
    }
    if (ptr == list->last) {	/* delete last element */
	info = ptr->info;
	ptr->bw->fw = ptr->fw;
	list->last = list->last->bw;
	dispose_item(ptr);
	return (info);
    }
    info = ptr->info;		/* element in the middle */
    ptr->fw->bw = ptr->bw;
    ptr->bw->fw = ptr->fw;
    dispose_item(ptr);
    return (info);
}


DLL_INFO dll_del_front(list)	/* retrieve and delete the front element */
    DLL list;
{
    return (del_item(list, list->first));
}


DLL_INFO dll_del_rear(list)	/* retrieve and delete the rear element */
    DLL list;
{
    return (del_item(list, list->last));
}


DLL_INFO dll_del_at(list, curr)	/* retrieve and delete the current element */
    DLL list;
    DLL_ITEM curr;
{
    return (del_item(list, curr));
}


void dll_apply(list, f)		/* apply function 'f' to all elements from list */
    DLL list;
    void (*f) ();
{
    DLL_ITEM curr;

    for (curr = dll_go_first(list); curr != NULL; curr = dll_go_fw(curr)) {
	(*f) (dll_inspect(curr));
    }
}


void dll_insert_sorted(list, info, cmp)	/* insert into sorted list */
    DLL list;
    DLL_INFO info;
    int (*cmp) ();
{
    DLL_ITEM curr;

    for (curr = dll_go_first(list);
	    (curr != NULL) && ((*cmp) (dll_inspect(curr), info) < 0);
	    curr = dll_go_fw(curr));

    if (dll_inspect(curr) == NULL) {
	dll_append(list, info);
    } else {
	dll_insert_at(list, curr, info);
    }
}


void dll_insert_sorted_unique(list, info, cmp)	/* insert into sorted list */
    DLL list;
    DLL_INFO info;
    int (*cmp) ();
{
    DLL_ITEM curr;

    for (curr = dll_go_first(list);
	    (curr != NULL) && ((*cmp) (dll_inspect(curr), info) < 0);
	    curr = dll_go_fw(curr));

    if ((*cmp) (dll_inspect(curr), info) == 0)	/* already in list */
	return;

    if (dll_inspect(curr) == NULL) {
	dll_append(list, info);
    } else {
	dll_insert_at(list, curr, info);
    }
}


DLL_INFO dll_delete_sorted(list, info, cmp)	/* delete from sorted list */
    DLL list;
    DLL_INFO info;
    int (*cmp) ();
{
    DLL_ITEM curr;
    int the_result;

    for (curr = dll_go_first(list);
      (curr != NULL) && (the_result = ((*cmp) (dll_inspect(curr), info)) < 0);
	    curr = dll_go_fw(curr));

    if ((dll_inspect(curr) == NULL) || (the_result != 0)) {
	return (NULL);
    }
    return (dll_del_at(list, curr));
}


int dll_compare(x, y, f)	/* compare two lists; read as "x - y" */
    DLL x, y;			/* 1: x bigger / 0: equal / -1: y bigger */
    int (*f) ();
{
    DLL_ITEM x_curr, y_curr;
    int the_result;

    for (x_curr = dll_go_first(x), y_curr = dll_go_first(y);
	    (x_curr != NULL) || (y_curr != NULL);
	    x_curr = dll_go_fw(x_curr), y_curr = dll_go_fw(y_curr)) {

	if (x_curr == NULL)
	    return (-1);
	if (y_curr == NULL)
	    return (1);

	if ((the_result = (*f) (dll_inspect(x_curr), dll_inspect(y_curr))) != 0) {
	    return (the_result);
	}
    }
    return (0);
}


int dll_count(list)		/* count the items in the list */
    DLL list;
{
    DLL_ITEM ptr;
    int cnt;

    for (cnt = 0, ptr = dll_go_first(list);
	    ptr != NULL;
	    cnt++, ptr = dll_go_fw(ptr));
    return (cnt);
}
