/*
 * utilities to generate fc output file
 */

#include <stdio.h>
#include <string.h>
#include "psf_prototype.h"
#include "psf_malloc.h"
#include "tiltype.h"
#include "fieldex.h"
#include "constants.h"
#include "options.h"
#include "objects.h"
#include "main.h"
#include "memory.h"
#include "expressions.h"
#include "definitions.h"
#include "states.h"
#include "transitions.h"
#include "blocks.h"
#include "ltsfc.h"
#include "std.h"

#define Inets 0
#define Inet 1
#define Ibehavs 2
#define Ibehav 3
#define Istructs 4
#define Istruct 5
#define Ivertice 6
#define Ivertex 7
#define Iedges 8
#define Iedge 9
#define Ilogic 10
#define Ihook 11

static char *longfc[] = {
    "nets", "net", "behavs", "behav", "structs", "struct",
    "vertice", "vertex", "edges", "edge", "logic", "hook"
};

static char *shortfc[] = {
    "N", "n", "B", "b", "S", "s",
    "V", "v", "E", "e", "l", "h"
};

char **fctable;

int *fc_translation;

void write_fc_file(mod, top, file)
    struct module *mod;
    char *top;
    FILE *file;
{
    int i;

    if (Option_Shorthand == TRUE) {
	fctable = shortfc;
    } else {
	fctable = longfc;
    }
    fprintf(file, "%% Transition system generated by 'lts_fc'\n");
    fprintf(file, "%% %d nets, %d states, %d transitions\n", 1, NR_PROCESSES, NR_DEFS);

    fprintf(file, "%s %d\n", fctable[Inets], 1);
    fprintf(file, "%s \"main\" > \"%d\"\n", fctable[Ihook], top ? process_index(top) - 1 : 0);
    for (i = 0; i < 1; i++) {
	write_fc_net(i, top, file);
    }
}

/* this would not be necessary if readtil would do it right ! PP */
char *stripquotes(s)
    char **s;
{
    if ((*s)[0] == '"')
	(*s)++;
    if ((*s)[strlen(*s) - 1] == '"')
	(*s)[strlen(*s) - 1] = '\0';
    return (*s);
}

void write_fc_net(nbr, top, file)
    int nbr;
    char *top;
    FILE *file;
{
    int i, topnr;

    fprintf(file, "%s %d\n", fctable[Inet], nbr);
    fprintf(file, "%s %d\n", fctable[Ibehavs], NR_ATOMS);
    for (i = 1; i <= NR_ATOMS; i++) {
	fprintf(file, ":%d \"%s\"\n", i - 1, stripquotes(&(atom[i]->name)));
    }
    fprintf(file, "%s %d\n", fctable[Istructs], NR_PROCESSES);
    topnr = -1;
    for_all_processes(i) {
	fprintf(file, ":%d \"%s\"\n", i - 1, stripquotes(&(process[i]->name)));
	if (top) {
	    if (strcmp(top,process[i]->name) == 0) {
		topnr = i;
	    }
	}
    }
    if ((top) && (topnr == -1)) {
	fprintf(stderr,"write_fc_net: Specified top does not exist.\n");
    }
    fprintf(file, "%s \"initial\" > %d\t\t%% \"%s\"\n", fctable[Ilogic], top ? topnr-1 : 0, top ? top : "default");
    fprintf(file, "%s %d\n", fctable[Ivertice], NR_PROCESSES);
    for_all_processes(i) {
	fprintf(file, "%s%d %s %d\t%% \"%s\"", fctable[Ivertex], i-1, fctable[Istruct], i-1, stripquotes(&(process[i]->name)));
	if (i == topnr) {
	    fprintf(file, " %s \"initial\"", fctable[Ilogic]);
	}
	fprintf(file, "\n");
	write_fc_edges(i, file);
    }
}

void write_fc_edges(nbr, file)
    int nbr;
    FILE *file;
{
    int i = 0;

    fprintf(file, "%s %d\n", fctable[Iedges], count_expressions(process[nbr]->exp));
    /* display all outgoing edges for this process */
    write_fc_transitions(file, process[nbr]->exp, &i);
}

void write_fc_transitions(file, expr, cnt)
/* copied from fprint_expression in expressions.c */
    int *cnt;
    FILE *file;
    struct expression *expr;
{
    switch (expr->type) {
    case SEQ_EXP:
	fprintf(file, "%s%d %s ", fctable[Iedge], *cnt, fctable[Ibehav]);
	switch (expr->left->type) {
	case ATM_EXP:
	    fprintf(file, "%d", expr->left->index - 1);
	    break;
	case SKP_EXP:
	    fprintf(file, "%s", "tau" );
	    break;
	case TCK_EXP:
	    fprintf(file, "%s", "quit" );
	    break;
	default:
	    fprintf(stderr, "write_fc_transitions: not implemented expression type\n");
	    ProgrammerError;
	    break;
        }
	fprintf(file, " -> ");
	fprintf(file, "%d", expr->right->index - 1);
	fprintf(file, "\t%% \"");
	write_fc_transitions(file, expr->left, cnt);
	fprintf(file, "\" -> \"");
	write_fc_transitions(file, expr->right, cnt);
	fprintf(file, "\"\n");
	(*cnt)++;
	break;
    case ALT_EXP:
	write_fc_transitions(file, expr->left, cnt);
	write_fc_transitions(file, expr->right, cnt);
	break;
    case ATM_EXP:
    case PRC_EXP:
    case SKP_EXP:
    case TCK_EXP:
	fprint_expression(file,expr);
	break;
    case DLK_EXP:
    case EPS_EXP:
    case PAR_EXP:
    case LMG_EXP:
    case COM_EXP:
    case ENC_EXP:
    case HID_EXP:
	fprintf(stderr, "write_fc_transitions: not implemented expression type\n");
	ProgrammerError;
	break;
    default:
	fprintf(stderr, "write_fc_transitions: unknown expression type\n");
	ProgrammerError;
	break;
    }
}

int count_expressions(expr)
    struct expression *expr;
{
    int i = 0;

    if (expr) {
	switch (expr->type) {
	case SEQ_EXP:
	    i++;
	    break;
	case ALT_EXP:
	    i = i + count_expressions(expr->left);
	    i = i + count_expressions(expr->left);
	    break;
	default:
	    break;
	}
    }
    return (i);
}

int process_index(name)
    char *name;
{
    int i;

    if (name) {
	for_all_processes(i) {
	    if (!strcmp(name, process[i]->name)) {
		return i;
	    }
	}
    }
    return -1;
}
