/*
 * dll_check.c : input argument checking routines for DLL package
 */

#include <stdio.h>
#include <stdlib.h>
#include "psf_prototype.h"
#include "sign.h"
#include "constants.h"
#include "_dll.h"


#if DLL_CHECK
/* list of issued item blocks */
static DLL_BLOCK i_blk = {NULL, NULL, TRUE, FALSE, cmp_pointers, 0, ITEM_ID};
DLL _issued_items = &i_blk;

/* list of issued dll blocks */
static DLL_BLOCK l_blk = {NULL, NULL, TRUE, FALSE, cmp_pointers, 0, LIST_ID};
DLL _issued_lists = &l_blk;

int _item_guard = TRUE;		/* guard for inserting on issued list */
int _list_guard = TRUE;		/* guard for inserting on issued list */


/*
 * Routines checking input arguments
 * 
 * only active when DLL_CHECK is not defined else '#defined' to empty statement!
 * 
 * check_item           : DLL_ITEM -> check_list           : DLL ->
 * check_non_empty_list : DLL -> check_order_fns      : DLL # DLL ->
 * check_ordinary_dll   : DLL
 */


void check_item(item)		/* check for legal item */
    DLL_ITEM item;
{

    if (item) {
	if (_dll_search(_issued_items, (DLL_INFO) item)) {
	    switch (item->id) {
	    case ITEM_ID:
		return;
		break;
	    case FREED_ITEM_ID:
		fprintf(stderr, "dll: item already freed\n");
		_dll_exit(DLL_ERROR);
		break;
	    }
	} else {
	    fprintf(stderr, "dll: item not issued\n");
	    _dll_exit(DLL_ERROR);
	}
    }
    fprintf(stderr, "dll: illegal item\n");
    _dll_exit(DLL_ERROR);
}


void check_list(list)		/* check for legal list */
    DLL list;
{

    if (list) {
	if (_dll_search(_issued_lists, (DLL_INFO) list)) {
	    switch (list->id) {
	    case LIST_ID:
		return;
		break;
	    case FREED_LIST_ID:
		fprintf(stderr, "dll: list already freed\n");
		_dll_exit(DLL_ERROR);
		break;
	    }
	} else {
	    fprintf(stderr, "dll: list %ld not issued\n", list);
	    _dll_exit(DLL_ERROR);
	}
    }
    fprintf(stderr, "dll: illegal list\n");
    _dll_exit(DLL_ERROR);
}


void check_non_empty_list(list)	/* check for non-empty list */
    DLL list;
{
    check_list(list);
    if (_dll_empty(list)) {	/* empty list */
	fprintf(stderr, "dll: empty list\n");
	_dll_exit(DLL_ERROR);
    }
}


void check_order_fns(x, y)
    DLL x, y;
{
    if (x->sorted && y->sorted) {
	if (x->cmp != y->cmp) {
	    fprintf(stderr, "dll: arguments have incompatible order functions\n");
	    _dll_exit(DLL_ERROR);
	}
    } else {
	fprintf(stderr, "dll: both arguments must be sorted lists\n");
	_dll_exit(DLL_ERROR);
    }
}


void check_non_sorted(list)	/* check for non-sorted list */
    DLL list;
{
    if (list->sorted) {
	fprintf(stderr, "dll: operation not permitted on sorted list\n");
	_dll_exit(DLL_ERROR);
    }
}


void check_on_list(list, item)	/* check for item on list */
    DLL list;
    DLL_ITEM item;
{
    if (!_dll_search(list, (DLL_INFO) item)) {
	fprintf(stderr, "dll: item not element of list\n");
	_dll_exit(DLL_ERROR);
    }
}


void check_list_types(x, y)	/* check for equal types of lists */
    DLL x, y;
{
    if (!((x->sorted == y->sorted) && (x->unique == y->unique))) {
	fprintf(stderr, "dll: lists are of different type\n");
	_dll_exit(DLL_ERROR);
    }
}


int cmp_pointers(p1, p2)
    DLL_INFO *p1, *p2;
{
    return (sign((size_t) p1 - (size_t) p2));
}


#else

#define check_item(i) ;
#define check_list(l) ;
#define check_non_empty_list(l) ;
#define check_order_fns(l1,l2) ;
#define check_non_sorted(l) ;
#define check_on_list(l,i) ;
#define check_list_types(l1,l2) ;
#define cmp_pointers(p1,p2) ;
#endif
