/*
 * utilities to manipulate blocks
 */

#include <stdio.h>
#include <string.h>
#include <assert.h>
#include "psf_prototype.h"
#include "memory.h"
#include "sign.h"
#include "std.h"
#include "constants.h"
#include "options.h"
#include "objects.h"
#include "expressions.h"
#include "states.h"
#include "transitions.h"
#include "closure.h"
#include "blocks.h"
#include "dll.h"


struct block *the_block;

int NR_BLOCKS;
int REACHABLE_BLOCKS;


int cmp_block_transitions(trans1, trans2)
    DLL_INFO trans1, trans2;
{
    struct transition *t1, *t2;

    t1 = (struct transition *) trans1;
    t2 = (struct transition *) trans2;

    switch (sign(cmp_expressions(t1->label, t2->label))) {
    case ZERO:
	return (sign(t1->target->block - t2->target->block));
	break;
    case NEGATIVE:
	return (NEGATIVE);
	break;
    case POSITIVE:
	return (POSITIVE);
	break;
    default:
	ProgrammerError;
	break;
    }
    ProgrammerError;
    return (UNDEFINED);
}


void make_blocks()
{
    int i = 0;
    DLL_ITEM closure_ptr;

    NR_BLOCKS = dll_count(the_closures);	/* #blocks = #closures */
    the_block = alloc_array(struct block, NR_BLOCKS);

    DLL_FORALL(the_closures, closure_ptr) {
	the_block[i].index = i;
	the_block[i].processed = FALSE;
	the_block[i].final = FALSE;
	the_block[i].deadlock = FALSE;
	the_block[i].reachable = FALSE;
	the_block[i].elements = (DLL) dll_inspect(closure_ptr);
	the_block[i].transitions = dll_su_create(cmp_block_transitions);
	the_block[i].exp = NULL;
	the_block[i].name = NULL;
	i++;
    }
}


void make_trans_for_blocks()
{
    DLL_ITEM state_ptr, trans_ptr, closure_ptr;
    struct state *the_state, *closure_element;
    struct transition *the_transition;

    NR_TRANSITIONS = 0;

    DLL_FORALL(the_states, state_ptr) {
	the_state = (struct state *) dll_inspect(state_ptr);

	the_block[the_state->block].deadlock =
	    the_block[the_state->block].deadlock || the_state->deadlock;
	the_block[the_state->block].final =
	    the_block[the_state->block].final || the_state->final;
	the_block[the_state->block].non_epsilon =
	    the_block[the_state->block].non_epsilon || the_state->non_epsilon;

	if (!the_block[the_state->block].processed) {
	    the_block[the_state->block].processed = TRUE;
	    the_block[the_state->block].exp = the_state->exp;
	    DLL_FORALL(the_state->epsilon_closure, closure_ptr) {
		closure_element = (struct state *) dll_inspect(closure_ptr);
		DLL_FORALL(closure_element->transitions, trans_ptr) {
		    the_transition = (struct transition *) dll_inspect(trans_ptr);
		    if (the_transition->label->type != EPS_EXP) {

#if 0
			printf("B%d -(%s)-> B%d\n",
			       the_state->block,
			       label_name(the_transition->label),
			       the_transition->target->block);
#endif

			dll_add(the_block[the_state->block].transitions, (DLL_INFO) the_transition);
			NR_TRANSITIONS++;
		    }
		}
	    }
	}
    }
}


void print_trans_for_blocks(reachability)
    int reachability;
{
    DLL_ITEM trans_ptr;
    struct transition *the_transition;
    int i;

    for (i = 0; i < NR_BLOCKS; i++) {
	if ((!reachability) || the_block[i].reachable) {
	    DLL_FORALL(the_block[i].transitions, trans_ptr) {
		the_transition = (struct transition *) dll_inspect(trans_ptr);
		printf("B%d -(", i);
		print_expression(the_transition->label);
		printf(")-> B%d\n", the_transition->target->block);
	    }
	}
    }
}


void print_trans_for_pref_names()
{
    DLL_ITEM trans_ptr;
    struct transition *the_transition;
    int i;

    for (i = 0; i < NR_BLOCKS; i++) {
	DLL_FORALL(the_block[i].transitions, trans_ptr) {
	    the_transition = (struct transition *) dll_inspect(trans_ptr);
	    printf("{B%d:", i);
	    print_expression(the_block[i].exp);
	    if (the_block[i].name) {
		printf(",%s", the_block[i].name);
	    }
	    printf("} -(");
	    print_expression(the_transition->label);
	    printf(")-> {B%d:", the_transition->target->block);
	    print_expression(the_block[the_transition->target->block].exp);
	    if (the_block[the_transition->target->block].name) {
		printf(",%s", the_block[the_transition->target->block].name);
	    }
	    printf("}\n");
	}
    }
}


void preferred_names()
{
    int i;

    for (i = 0; i < NR_BLOCKS; i++) {
	if (the_block[i].deadlock && !the_block[i].non_epsilon) {
	    the_block[i].name = "Deadlock-State";
	} else {
	    switch (the_block[i].exp->type) {
	    case PRC_EXP:
		the_block[i].name = process[the_block[i].exp->index]->name;
		break;
	    case TCK_EXP:
		the_block[i].name = "Final-State";
		break;
	    case DLK_EXP:
		the_block[i].name = "Deadlock-State";
		break;
	    default:
		the_block[i].name = NULL;
		break;
	    }
	}
    }
}


void reachable_blocks()
{
    int i;

    REACHABLE_BLOCKS = REACHABLE_TRANSITIONS = 0;

    for (i = 0; i < NR_BLOCKS; i++) {
	if (!the_block[i].reachable) {
	    if (the_block[i].exp->type == PRC_EXP) {
		reachable_block(i);
	    }
	}
    }
}


void reachable_block(index)
    int index;
{
    struct transition *the_transition;
    DLL_ITEM trans_ptr;

    if (!the_block[index].reachable) {
	the_block[index].reachable = TRUE;
	REACHABLE_BLOCKS++;

	DLL_FORALL(the_block[index].transitions, trans_ptr) {
	    the_transition = (struct transition *) dll_inspect(trans_ptr);
	    REACHABLE_TRANSITIONS++;
	    reachable_block(the_transition->target->block);
	}
    }
}


void print_blocks(reachability)
    int reachability;
{
    int i;

    for (i = 0; i < NR_BLOCKS; i++) {
	if ((!reachability) || the_block[i].reachable) {
	    printf("B%-3d: ", i);
	    if (the_block[i].name) {
		printf("%s", the_block[i].name);
	    } else {
		print_expression(the_block[i].exp);
	    }
	    printf("\n");
	}
    }
    printf("\n");
}


void print_reachable_blocks_and_trans()
{
    DLL_ITEM trans_ptr;
    struct transition *the_transition;
    int i;

    for (i = 0; i < NR_BLOCKS; i++) {
	if (the_block[i].reachable) {
	    DLL_FORALL(the_block[i].transitions, trans_ptr) {
		the_transition = (struct transition *) dll_inspect(trans_ptr);
		printf("{B%d:", i);
		if (the_block[i].name) {
		    printf("%s", the_block[i].name);
		} else {
		    print_expression(the_block[i].exp);
		}
		printf("} -(");
		print_expression(the_transition->label);
		printf(")-> {B%d:", the_transition->target->block);
		if (the_block[the_transition->target->block].name) {
		    printf("%s", the_block[the_transition->target->block].name);
		} else {
		    print_expression(the_block[the_transition->target->block].exp);
		}
		printf("}\n");
	    }
	}
    }
    printf("\n");
}
