#include <stdio.h>
#include "psf_prototype.h"
#include "psf_standards.h"
#include "psf_malloc.h"
#include "dump.h"
#include "main.h"
#include "symtab.h"
#include "lexyacc.h"

#define Iany 0
#define Ibehav 1
#define Ibehavs 2
#define Iconstant 3
#define Ideclarations 4
#define Iedge 5
#define Iedges 6
#define Igraph 7
#define Igraphs 8
#define Ihook 9
#define Ihooks 10
#define Iindirect 11
#define Iinfix 12
#define Ilogic 13
#define Ilogics 14
#define Inet 15
#define Inets 16
#define Iprefix 17
#define Iresult 18
#define Iresults 19
#define Iunary 20
#define Ivertex 21
#define Ivertice 22
#define Istruct 23
#define Istructs 24

char **fctable;

static char *longfc[] = {
    "any", "behav", "behavs", "constant ", "declarations", "edge ",
    " edges ", "graph ", "graphs ", "hook", "hooks", " indirect ",
    "infix", "logic", "logics ", "net", "nets ", "prefix",
    " -> ", " results ", "unary", "vertex", "vertice", "struct",
    "structs"
};

static char *shortfc[] = {
    "any", "b", "B", "C", "D", "e",
    "E", "g", "G", "h", "H", "&",
    "I", "l", "L", "n", "N", "P",
    "r", "R", "U", "v", "V", "s",
    "S"
};

#define Icolon 0
#define Iident 1
#define Iinfix0 2
#define Iinfix1 3
#define Iinfix2 4
#define Iinfix3 5
#define Iinfix4 6
#define Iinfix5 7
#define Iint 8
#define Isymbol 9
#define Itabletype 10
#define Istar 11
#define Istring 12
#define Iversion 13
#define Iop 14
#define Icp 15
#define Iref 16
#define Iminus 17
#define Iundef 18
#define Ieof 19
#define Iunknown 20

static char *dumpfc[] = {
    ":", "ident", "infix0", "infix1", "infix2", "infix3", "infix4",
    "infix5", "int", "symbol", "table_type", "*", "string", "version",
    "(", ")", "@", "-", "undef", "end_of_file", "UNKNOWN"
};


char *typestr(val)
int val;
{
    switch (val) {
    case ANY:
	return(fctable[Iany]);
    case ARROW:
	return(longfc[Iresult]);
    case BEHAV:
	return(fctable[Ibehav]);
    case BEHAVS:
	return(fctable[Ibehavs]);
    case COLON:
	return(dumpfc[Icolon]);
    case CONSTANT:
	return(fctable[Iconstant]);
    case DECLARATIONS:
	return(fctable[Ideclarations]);
    case EDGE:
	return(fctable[Iedge]);
    case EDGES:
	return(fctable[Iedges]);
    case HOOK:
	return(fctable[Ihook]);
    case HOOKS:
	return(fctable[Ihooks]);
    case IDENT:
	return(dumpfc[Iident]);
    case INFIX:
	return(fctable[Iinfix]);
    case INFIX5:
	return(dumpfc[Iinfix5]);
    case INFIX4:
	return(dumpfc[Iinfix4]);
    case INFIX3:
	return(dumpfc[Iinfix3]);
    case INFIX2:
	return(dumpfc[Iinfix2]);
    case INFIX1:
	return(dumpfc[Iinfix1]);
    case INFIX0:
	return(dumpfc[Iinfix0]);
    case INTEGER:
	return(dumpfc[Iint]);
    case LOGIC:
	return(fctable[Ilogic]);
    case LOGICS:
	return(fctable[Ilogics]);
    case NET:
	return(fctable[Inet]);
    case NETS:
	return(fctable[Inets]);
    case RESULT:
	return(fctable[Iresult]);
    case RESULTS:
	return(fctable[Iresults]);
    case SYMBOL:
	return(dumpfc[Isymbol]);
    case TABLE_TYPE:
	return(dumpfc[Itabletype]);
    case VERTEX:
	return(fctable[Ivertex]);
    case VERTICE:
	return(fctable[Ivertice]);
    case FCSTAR:
	return(dumpfc[Istar]);
    case STRING:
	return(dumpfc[Istring]);
    case STRUCT:
	return(fctable[Istruct]);
    case STRUCTS:
	return(fctable[Istructs]);
    case UNARY:
	return(fctable[Iunary]);
    case VERSION:
	return(dumpfc[Iversion]);
    case OP:
	return(dumpfc[Iop]);
    case CP:
	return(dumpfc[Icp]);
    case REF:
	return(dumpfc[Iref]);
    case MINUS:
	return(dumpfc[Iminus]);
    case INDIRECT:
	return(fctable[Iindirect]);
    case PREFIX:
	return(fctable[Iprefix]);
    case UNDEF:
	return(dumpfc[Iundef]);
    case END_OF_FILE:
	return(dumpfc[Ieof]);
    default:
	fprintf(stderr,"Unknown type in symbol table: %d.\n",val);
	return(dumpfc[Iunknown]);
  }
}

void dumpmapitem(df,i)
FILE *df;
int i;
{
    int j;
    opentry *entry = (opentry *)map[i].info;

    fprintf(df,"%s %s",typestr(map[i].type),map[i].text);
    switch (map[i].type) {
    case CONSTANT:
	fprintf(df," -> (%s)",typestr(entry->codomain));
	break;
    case UNARY:
	fprintf(df," (%s)",typestr(entry->domain[0]));
	fprintf(df," -> (%s)",typestr(entry->codomain));
	break;
    case INFIX:
	fprintf(df," (%s,",typestr(entry->domain[0]));
	fprintf(df,"%s)",typestr(entry->domain[1]));
	fprintf(df," -> (%s)",typestr(entry->codomain));
	break;
    case PREFIX:
	fprintf(df," (");
	for ( j = 0; j < entry->a - 1; j++ ) {
	    fprintf(df,"%s,",typestr(entry->domain[j]));
	}
	fprintf(df,"%s)",typestr(entry->domain[j]));
	fprintf(df," -> (%s)",typestr(entry->codomain));
	break;
    default:
	break;
    }
    fprintf(df,"\n");
}

void dumpdeclarations(df)
FILE *df;
{
    int dec=1;
    int i;

    for (i=last_predef+1; i<mapindex; i++) {
	switch (map[i].type) {
	case INFIX:
	case PREFIX:
	case UNARY:
	case CONSTANT:
	    if (dec) {
		fprintf(df,"%s\n",typestr(DECLARATIONS));
		dec--;
	    }
	    dumpmapitem(df,i);
	    break;
	}
    }
}

void dumpmap(df)
FILE *df;
{
    int i;

    if (optshort == TRUE) {
        fctable = shortfc;
    } else {
        fctable = longfc;
    }

    dumpdeclarations(df);
    fprintf(df,"Symbol table dump:\n");
    for (i=0; i<mapindex; i++) {
	fprintf(df,"%2d: ",i);
	dumpmapitem(df,i);
    }
    fflush(df);
}

void dumpnetstable(df,n)
FILE *df;
netstable *n;
{
    int i;

    if (n) {
	fprintf(df,"%s%d\n",typestr(NETS),listcount(n->nlst));
	dumplabelsandtables(df,n->tl);
	FORALL(n->nlst,i) {
	    dumpnetitem(df,getitem(n->nlst,i));
	}
    }
}

void dumpnetitem(df,ni)
FILE *df;
listitem *ni;
{
    int j;
    netitem *i = (netitem *)ni;

    if (i) {
	fprintf(df,typestr(NET));
	if (i->id >= 0) {
            fprintf(df," %d",i->id);
	}
	dumplabelsandtables(df,i->tl);
	fprintf(df,"%s %d\n",typestr(VERTICE),listcount(i->vlst));
	FORALL(i->vlst,j) {
	    dumpvertexitem(df,getitem(i->vlst,j));
	}
    }
}

void dumpvertexitem(df,vi)
FILE *df;
listitem vi;
{
    int i;
    vertexitem *v = (vertexitem *)vi;

    if (v) {
	fprintf(df,typestr(VERTEX));
	if (v->id >= 0) {
	    fprintf(df,"%s%d ",optshort?"":" ",v->id);
	}
	dumplabels(df,v->lab,NONL);
	fprintf(df,"%s%d\n",typestr(EDGES),listcount(v->elst));
	FORALL (v->elst,i) {
	    dumpedgeitem(df,getitem(v->elst,i));
	}
    }
}

void dumpedgeitem(df,ei)
FILE *df;
listitem ei;
{
    int i;
    edgeitem *e = (edgeitem *)ei;

    if (e) {
	if (e->id >= 0) {
	    fprintf(df,typestr(EDGE));
	    fprintf(df,"%d ",e->nbr);
	}
	dumplabels(df,e->lab,NONL);
	i = countexp(e->slst);
	if (i) {
	    if (i > 1) {
		fprintf(df,"%s%d",typestr(RESULTS),i);
	    }
	    fprintf(df,typestr(RESULT));
	    dumpexpr(df,e->slst);
	    fprintf(df,"\n");
	}
    }
}

void dumplabelsandtables(df,tl)
FILE *df;
tlitem *tl;
{
    if (tl) {
	dumplabels(df,tl->lab,NL);
	dumptables(df,tl->tab);
    }
}

void dumplabels(df,l,nl)
FILE *df;
list *l;
int nl;
{
    int i;

    FORALL(l,i) {
	dumplabelitem(df,getitem(l,i));
	if (nl == NL) fprintf(df,"\n");
    }
}

void dumplabelitem(df,li)
FILE *df;
listitem li;
{
    labelitem *l = (labelitem *)li;

    switch (l->typ) {
    case STRUCT:
    case BEHAV:
    case LOGIC:
    case HOOK:
	fprintf(df," %s%s",typestr(l->typ),optshort?"":" ");
	dumpexpr(df,l->exp);
	break;
    }
}

void dumptables(df,t)
FILE *df;
list *t;
{
    int i;

    FORALL(t,i) {
	dumptableitem(df,getitem(t,i));
    }
}

void dumptableitem(df,ti)
FILE *df;
listitem ti;
{
    int i;
    tableitem *t = (tableitem *)ti;

    switch (t->typ) {
    case STRUCT:
	fprintf(df,"%s ",typestr(STRUCTS));
	break;
    case BEHAV:
	fprintf(df,"%s ",typestr(BEHAVS));
	break;
    case LOGIC:
	fprintf(df,"%s ",typestr(LOGICS));
	break;
    case HOOK:
	fprintf(df,"%s ",typestr(HOOKS));
	break;
    default:
	break;
    }
    if (t->ind) { /* indirect ! */
	fprintf(df,"%s%d\n",typestr(INDIRECT),t->ofs);
    } else {
	fprintf(df,"%d\n",listcount(t->ilst));
	FORALL(t->ilst,i) {
	    dumptableentry(df,getitem(t->ilst,i));
	    fprintf(df,"\n");
	}
    }
}

void dumptableentry(df,si)
FILE *df;
listitem si;
{
    tableentry *s = (tableentry *)si;

    if (s) {
	if (s->id >= 0) {
	    fprintf(df,":%d ",s->id);
	}
	dumpexpr(df,s->exp);
	if (s->tilindex >= 0) {
	    fprintf(df,"\t%% (tilindex: %d.%d)",s->tiltype,s->tilindex);
	}
    }
}

void dumpexpr(df,e)
FILE *df;
expr *e;
{
    char *s;
    if (e) {
	s = strexpr(e);
	switch (e->operator) {
	case INTEGER:
	case INDIRECT:
	case REF:
	case CONSTANT:
	case UNARY:
	case PREFIX:
	case INFIX:
	case INFIX0:
	case INFIX1:
	case INFIX2:
	case INFIX3:
	case INFIX4:
	case INFIX5:
	case FCSTAR:
	    fprintf(df,s);
	    PSF_FREE(s);
	    break;
	case STRING:
	    fprintf(df,"\"%s\"",s);
	    PSF_FREE(s);
	    break;
	}
    }
    fflush(df);
}

void dump(df,n)
FILE *df;
netstable *n;
{

    if (optshort == TRUE) {
	fctable = shortfc;
    } else {
	fctable = longfc;
    }

    dumpdeclarations(df);
    dumpnetstable(df,n);

    fflush(df);
}
