
package LISPcore;

$VERSION = "1.01";

use Global;
use FluidGC;

require Exporter;
@ISA = qw( FluidGC Exporter );
@EXPORT = @FluidGC::EXPORT;

my $NIL;
my $T;

###################################
# LISP utilities
###################################

sub make_atom {
    my $v = shift;
    my $a;

    $a = NewFocus('reg');
    AddValueField($a, 'atom', 'str', $v);
    return $a;
}

sub make_cons {
    my $car = shift;
    my $cdr = shift;
    my $cons;

    $cons = NewFocus('reg');
    SetField($cons, 'car', $car);
    SetField($cons, 'cdr', $cdr);
    return $cons;
}

sub make_list {
    my $l = shift;
    my $cons;
    my $car;
    my $cdr;
    my $v;
    my @list;

    # copy it, so we keep the original intact
    @list = @{$l};
    if ($#list == -1) {
	return 0;
    }
    $v = shift @list;
    if (ref $v) {
	$car = make_list($v);
    } else {
	$car = make_atom($v);
    }
    if ($#list >= 0) {
	$cdr = make_list(\@list);
	$cons = make_cons($car, $cdr);
	return $cons;
    } else {
	return $car;
    }
}

sub make_focus {
    my $x = shift;
    my $v = shift;
    my $s;

    $s = atom_value($x);
    SetFocus($s, $v);
}

sub atom_value {
    my $x = shift;

    return GetValue(GetField($x, 'atom'));
}

###################################
# LISP functions
###################################

sub fun_atom {
    my $x = shift;

    if (HasField($x, 'atom')) {
	return $T;
    }
    return $NIL;
}

sub fun_null {
    my $x = shift;

    if (fun_atom($x) == $T && atom_value($x) eq 'NIL') {
	return $T;
    }
    return $NIL;
}

sub fun_eq {
    my $x = shift;
    my $car;
    my $cdr;

    if (fun_atom($x) == $T) {
	return undef;
    }
    $car = fun_car($x);
    $cdr = fun_cdr($x);
    if (fun_atom($car) == $T && fun_atom($cdr) == $T) {
	if (atom_value($car) eq atom_value($cdr)) {
	    return $T;
	}
	return $NIL;
    }
    return undef;
}

sub fun_car {
    my $x = shift;

    if (fun_atom($x) == $T) {
	$s = print_list($x);
	die "CAR: $s not a list\n";
    }
    return GetField($x, 'car');
}

sub fun_cdr {
    my $x = shift;

    if (fun_atom($x) == $T) {
	$s = print_list($x);
	die "CDR: $s not a list\n";
    }
    return GetField($x, 'cdr');
}

sub fun_cons {
    my $x = shift;

#    if (fun_atom($x) == $T) {
#	return undef;
#    }
    return $x;
}

sub fun_quote {
    my $x = shift;
    my $r;

    $r = make_cons(make_atom('QUOTE'), $x);
    return $r;
}
    
sub fun_gc {
    SetFocusNull('reg');
    RestrictedGC();
    return $T;
}

sub assocv {
    my $x = shift;
    my $y = shift;
    my $car;

#print "assocv $x\n";
    if ($y == $NIL) {
	return $NIL;
    }
    $car = fun_car($y);
    if (atom_value(fun_car($car)) eq $x) {
	return fun_cdr($car);
    } else {
	return assocv($x, fun_cdr($y));
    }
}

sub assoc {
    my $x = shift;
    my $y = shift;

    return assocv(atom_value($x), $y);
}

sub pair {
    my $x = shift;
    my $y = shift;
    my $r;

#print "pair $x $y\n";
    $r = make_cons(make_cons($x, $y), GetFocus('OBLIST'));
    SetFocus('OBLIST', $r);
    return $r;
}

sub evlis {
    my $acc = shift;
    my ($rcar, $rcdr);

    if (fun_null($acc) == $T) {
	return $acc;
    }
    $rcar = fun_eval(fun_car($acc));
    $rcdr = evlis(fun_cdr($acc));
    $r = make_cons($rcar, $rcdr);
    return $r;
}

sub evcon {
    my $c = shift;

    $caar = fun_car(fun_car($c));
    $r = fun_eval($caar);
    if (fun_null($r) == $T) {
	return evcon(fun_cdr($c));
    } else {
	$r = fun_eval(fun_car(fun_cdr(fun_car($c))));
	return $r;
    }
}


sub fun_apply {
    my $car = shift;
    my $cdr = shift;
    my $save;

    if (fun_atom($car) == $T) {
	$fun = atom_value($car);
	if ($fun eq 'ATOM') {
	    $r = fun_car($cdr);
	    return fun_atom($r);
	} elsif ($fun eq 'NULL') {
	    $r = fun_car($cdr);
	    return fun_null($r);
	} elsif ($fun eq 'EQ') {
	    $r = make_cons(fun_car($cdr), fun_car(fun_cdr($cdr)));
	    return fun_eq($r);
	} elsif ($fun eq 'CAR') {
	    $r = fun_car($cdr);
	    return fun_car($r);
	} elsif ($fun eq 'CDR') {
	    $r = fun_car($cdr);
	    return fun_cdr($r);
	} elsif ($fun eq 'CONS') {
	    $cddr = fun_cdr($cdr);
	    $caddr = fun_car($cddr);
	    $r = make_cons(fun_car($cdr), $caddr);
	    return $r;
	} elsif ($fun eq 'SET') {
	    $r = fun_car($cdr);
	    if (fun_atom($r) != $T) {
		die "SET: argument not a symbol\n";
	    }
	    $v = fun_car(fun_cdr($cdr));
	    pair($r, $v);
	    return $v;
	} elsif ($fun eq 'DEFINE') {
	    $r = fun_car($cdr);
	    if (fun_atom($r) != $T) {
		die "DEFINE: argument not a symbol\n";
	    }
	    $v = fun_car(fun_cdr($cdr));
	    pair($r, $v);
	    return fun_eval(make_cons(make_atom('QUOTE'), make_cons($r, $NIL)));
	} elsif ($fun eq 'GC') {
	    $r = fun_gc();
	    return $r;
	} else {
	    $r = fun_eval($car);
	    $r = fun_apply($r, $cdr);
	    return $r;
	}

    } else {
	$caar = fun_car($car);
	$fun = atom_value($caar);
	if ($fun eq 'LABEL') {
	    $save = GetFocus('OBLIST');
	    $r = fun_car(fun_cdr($car));
	    $r = pair($r, $car);
	    $r = fun_apply(fun_car(fun_cdr(fun_cdr($car))), $cdr);
	    SetFocus('OBLIST', $save);
	    return $r;
	} elsif ($fun eq 'LAMBDA') {
	    $save = GetFocus('OBLIST');
	    $fp = fun_car(fun_cdr($car));
	    $ap = $cdr;
# should use recursive function pairlis
	    while (fun_null($fp) != $T) {
	    $s = print_list($fp);
#print "fp: $s\n";
	    $s = print_list($ap);
#print "ap: $s\n";
		$r = pair(fun_car($fp), fun_car($ap));
		$fp = fun_cdr($fp);
		$ap = fun_cdr($ap);
	    }
	    $r = fun_eval(fun_car(fun_cdr(fun_cdr($car))));
	    SetFocus('OBLIST', $save);
	    return $r;
	} else {
	    $s = print_list($car);
	    die "list $s cannot be evaluated to a function\n";
	}
    }
}

sub fun_eval {
    my $acc = shift;
    my $car;
    my $cdr;
    my $fun;
    my $r;

    if (fun_atom($acc) == $T) {
	$fun = atom_value($acc);
	# we should look it up on the assoc list, instead of focus
#	$r = assoc($acc, $focus{OBLIST});
	$r = assoc($acc, GetFocus('OBLIST'));
	if ($r != $NIL) {
	    return $r;
	}
#	die "no value for $fun found\n";
	return $r;
    } else {
	$car = fun_car($acc);
	if (fun_atom($car) == $T) {
	    $fun = atom_value($car);
#print "evalfun $fun $acc\n";
	    $cdr = fun_cdr($acc);
	    if ($fun eq 'QUOTE') {
		return fun_car($cdr);
	    } elsif ($fun eq 'COND') {
		$r = evcon($cdr);
		return $r;
	    } elsif ($fun eq 'GC') {
		$r = fun_gc();
		return $r;
	    } elsif ($fun eq 'OBLIST') {
		return $focus{OBLIST};
	    } elsif ($fun eq 'PRINT') {
		$r = fun_eval(fun_car($cdr));
#		$s = print_list($r);
#		print "* $s\n";
		return $NIL;
	    }
	}
    }
    $r = fun_apply(fun_car($acc), evlis(fun_cdr($acc)));
    return $r;
}

###################################
# printing
###################################

sub print_list_rem {
    my $x = shift;
    my $s;
    my $cdr;

    if (fun_atom($x) == $T) {
	$s = atom_value($x);
    } else {
	$cdr = fun_cdr($x);
	if (fun_atom($cdr) == $T) {
	    if (fun_null($cdr) != $T) {
#		$s = "(";
		$s = print_list(fun_car($x));
		$s .= " . ";
		$s .= atom_value($cdr);
#		$s .= ")";
	    } else {
		$s = print_list(fun_car($x));
	    }
	} else {
	    $s = print_list(fun_car($x));
	    $s .= " ";
	    $s .= print_list_rem($cdr);
	}
    }
    return $s;
}

sub print_list2 {
    my $x = shift;
    my $s;
    my $cdr;
    my $l;
    my $end;

    if (fun_atom($x) == $T) {
	$s = atom_value($x);
    } else {
	$s = print_list(fun_car($x));
	$cdr = fun_cdr($x);
	if (fun_null($cdr) == $T) {
	    return ($s, 1);
	} else {
	    ($l, $end) = print_list2($cdr);
	    if ($end) {
		$s = "$s $l";
	    } else {
		$s = "($s . $l)";
	    }
	    return ($s, $end);
	}
    }
    return ($s, 0);
}

sub print_list {
    my $x = shift;
    my $s;
    my $cdr;
    my $NIL;

    if (fun_atom($x) == $T) {
	$s = atom_value($x);
    } else {
	($s, $NIL) = print_list2($x);
	if ($NIL) {
	    $s = "($s)";
	}
    }
    return $s;
}

###################################
# standard BI methods
###################################

sub Exec {
    my $self = shift;
    my @i = @_;
    my $opc;
    my $x;
    my $acc;
    my $r;
    my $s;

    $opc = shift @i;
    if ($opc eq 'LIST') {
	$x = shift @i;
	$acc = make_list($x);
	eval { $r = fun_eval($acc) };
	if ($@) {
	    $main::ui->Msg("ERROR: $@");
	    return 0;
	}
	if (defined $r) {
	    $s = print_list($r);
	    $main::ui->Msg("$s\n");
	    if ($r == $T) {
		return 1;
	    } elsif ($r == $NIL) {
		return 0;
	    } else {
		return 1;
	    }
	} else {
	    return 0;
	}
    } elsif ($opc eq 'SYMBOL') {
	$x = shift @i;
	$r = assocv($x, GetFocus('OBLIST'));
	if ($r != $T) {
	    $s = print_list($r);
	    $main::ui->Msg("$s\n");
	    return 1;
	}
	$main::ui->Msg("ERROR: $x is not a symbol\n");
    } else {
	return 0;
    }
}

sub DumpCore {
    my $self = shift;
    my $f;
    my $i;

    FluidBase->DumpFluid();
}

sub InitCore {
    my $f;

    $::nonull = 1;
    FluidBase->InitFluid();
    $NIL = make_atom('NIL');
    SetFocus('OBLIST', $NIL);
    $T = make_atom('T');
    pair($NIL, $NIL);
    pair($T, $T);
}

1;
